package com.joshiegemfinder.synchronisedblockstates.intermediary.network.client;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import com.joshiegemfinder.synchronisedblockstates.common.SynchronisedBlockstates;
import com.joshiegemfinder.synchronisedblockstates.common.util.BlockRegistryRepresentative;
import com.joshiegemfinder.synchronisedblockstates.common.util.BlockStateRepresentative;
import com.joshiegemfinder.synchronisedblockstates.common.util.StateDefinitionRepresentative;
import com.joshiegemfinder.synchronisedblockstates.intermediary.fabric.SynchronisedBlockstatesClient;
import com.joshiegemfinder.synchronisedblockstates.intermediary.network.packet.ClientboundChunkedBlockstateSyncDataPacket;
import com.joshiegemfinder.synchronisedblockstates.intermediary.network.packet.ClientboundChunkedBlockstateSyncEndPacket;
import com.joshiegemfinder.synchronisedblockstates.intermediary.network.packet.ClientboundChunkedBlockstateSyncStartPacket;

import net.fabricmc.fabric.api.client.networking.v1.ClientConfigurationConnectionEvents;
import net.fabricmc.fabric.api.client.networking.v1.ClientConfigurationNetworking;

public class ChunkedPacketReciever {

	public static Map<UUID, ChunkedPacketReciever> chunkRecievers = new HashMap<UUID, ChunkedPacketReciever>();
	
	public final UUID uuid;
	protected StateDefinitionRepresentative[] allBlocks;
	protected BlockStateRepresentative[] allStates;
	protected boolean finished;
	
	public ChunkedPacketReciever(UUID uuid, int totalStateCount, int totalBlockCount) {
		this.uuid = uuid;
		this.allStates = new BlockStateRepresentative[totalStateCount];
		this.allBlocks = new StateDefinitionRepresentative[totalBlockCount];
	}

	public void onData(int blockStartIndex, StateDefinitionRepresentative[] blocks) {
		// copy blocks in
		System.arraycopy(blocks, 0, this.allBlocks, blockStartIndex, blocks.length);
		
		final BlockStateRepresentative[] allStates = this.allStates;
		
		for(StateDefinitionRepresentative block : blocks) {
			final int stateCount = block.stateCount;
			final int[] stateIndexes = block.stateIndexes;
			final BlockStateRepresentative[] states = block.states;
			
			for(int i = 0; i < stateCount; ++i) {
				allStates[stateIndexes[i]] = states[i];
			}
		}
	}
	
	public BlockRegistryRepresentative build() {
		if(this.finished || this.isFinished()) {
			return null;
		} else {
			return new BlockRegistryRepresentative(this.allBlocks, this.allStates);
		}
	}
	
	public void finish() {
		this.finished = true;
		this.allBlocks = null;
		this.allStates = null;
	}
	
	public boolean isFinished() {
		return this.finished;
	}
	
	public static void handleRecieveChunkStartPacket(ClientboundChunkedBlockstateSyncStartPacket packet, ClientConfigurationNetworking.Context context) {
		UUID uuid = packet.chunkingId();
		
		ChunkedPacketReciever reciever = new ChunkedPacketReciever(uuid, packet.totalStateCount(), packet.totalBlockCount());
		
		chunkRecievers.put(uuid, reciever);
	}

	public static void handleRecieveChunkDataPacket(ClientboundChunkedBlockstateSyncDataPacket packet, ClientConfigurationNetworking.Context context) {
		UUID uuid = packet.chunkingId();
		
		var reciever = chunkRecievers.get(uuid);
		
		if(reciever == null) {
			SynchronisedBlockstates.LOGGER.warn("Recieved chunk data packet for id {}, but it doesn't exist!", uuid);
		} else {
			reciever.onData(packet.blockStartIndex(), packet.blocks());
		}
	}

	public static void handleRecieveChunkEndPacket(ClientboundChunkedBlockstateSyncEndPacket packet, ClientConfigurationNetworking.Context context) {
		UUID uuid = packet.chunkingId();
		
		var reciever = chunkRecievers.get(uuid);
		
		if(reciever == null || reciever.isFinished()) {
			SynchronisedBlockstates.LOGGER.warn("Recieved chunk end packet for id {}, but it doesn't exist!", uuid);
		} else {
			SynchronisedBlockstatesClient.handleRecieveBlockstates(reciever.build(), context);
			reciever.finish();
		}
	}
	
	public static void onConfigurationEnd() {
		for(var reciever : chunkRecievers.values()) {
			reciever.finish();
		}
		
		chunkRecievers.clear();
	}
	
	public static void registerPacketsClient() {
		ClientConfigurationNetworking.registerGlobalReceiver(ClientboundChunkedBlockstateSyncStartPacket.TYPE, ChunkedPacketReciever::handleRecieveChunkStartPacket);
		ClientConfigurationNetworking.registerGlobalReceiver(ClientboundChunkedBlockstateSyncDataPacket.TYPE, ChunkedPacketReciever::handleRecieveChunkDataPacket);
		ClientConfigurationNetworking.registerGlobalReceiver(ClientboundChunkedBlockstateSyncEndPacket.TYPE, ChunkedPacketReciever::handleRecieveChunkEndPacket);

		ClientConfigurationConnectionEvents.COMPLETE.register((handler, client) -> onConfigurationEnd());
		ClientConfigurationConnectionEvents.DISCONNECT.register((handler, client) -> onConfigurationEnd());
	}
	
}
