package com.joshiegemfinder.synchronisedblockstates.common.util;

import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.IntStream;

import it.unimi.dsi.fastutil.objects.ObjectArraySet;

public final class BlockRepresentative<ResourceKeyClass> implements Iterable<PropertyRepresentative> {
	protected final ResourceKeyClass blockKey;
	protected final PropertyRepresentative[] properties;
	
	public BlockRepresentative(ResourceKeyClass blockKey, PropertyRepresentative[] properties) {
		this.blockKey = Objects.requireNonNull(blockKey);
		this.properties = Objects.requireNonNull(properties);
	}
	
	public ResourceKeyClass getKey() {
		return this.blockKey;
	}

	public List<PropertyRepresentative> getProperties() {
		return Arrays.asList(this.properties);
	}
	
	public PropertyRepresentative getProperty(int index) {
		return properties[index];
	}

	public int getPropertyIndex(PropertyRepresentative value) {
		if(value != null)
			for(int i = 0; i < properties.length; ++i) {
				if(properties[i] == value) {
					return i;
				}
			}
		return -1;
	}
	
	public int getPropertyCount() {
		return properties.length;
	}

	@Override
	public Iterator<PropertyRepresentative> iterator() {
		return new Iterator<PropertyRepresentative>() {
			private int index = 0;
			
			@Override
			public boolean hasNext() {
				return index < properties.length;
			}

			@Override
			public PropertyRepresentative next() {
				return properties[index++];
			}
		};
	}
	
	@Override
	public String toString() {
		return blockKey.toString();
	}

	public String asString() {
		return MappingSafeResourceKeyHelper.asString(this.blockKey);
	}
	
	@Override
	public int hashCode() {
		return this.blockKey.hashCode() * 31 + this.properties.length;
	}
	
	@Override
	public boolean equals(Object obj) {
		if(obj == this) {
			return true;
		} else if(obj instanceof BlockRepresentative<?> otherBlock) {
			return Objects.equals(this.blockKey, otherBlock.blockKey);
		} else {
			return false;
		}
	}

	public boolean equalsStrict(Object obj) {
		if(obj == this) {
			return true;
		} else if(obj instanceof BlockRepresentative<?> other) {
			return this.properties.length == other.properties.length && this.blockKey.equals(other.blockKey) && Arrays.equals(this.properties, other.properties);
		} else {
			return false;
		}
	}
	
	public boolean compatible(BlockRepresentative<?> other) {
		if(this == other) return true;
		if(other == null) return false;
		
		if(!this.blockKey.equals(other.blockKey)) {
			return false;
		}
		
		return propertiesCompatible(this.properties, other.properties);
	}

	public boolean propertiesCompatible(PropertyRepresentative[] b) {
		return propertiesCompatible(this.properties, b);
	}
	
	public static boolean propertiesCompatible(PropertyRepresentative[] a, PropertyRepresentative[] b) {
		final int propertyCount = a.length;

		if(propertyCount != b.length) return false;
		
		if(propertyCount == 0) return true;
		
		Set<PropertyRepresentative> propertySet1 = new ObjectArraySet<PropertyRepresentative>(propertyCount);
		Set<PropertyRepresentative> propertySet2 = new ObjectArraySet<PropertyRepresentative>(propertyCount);

		for(int i = 0; i < propertyCount; ++i) {
			propertySet1.add(a[i]);
			propertySet2.add(b[i]);
		}
		
		return propertySet1.containsAll(propertySet2) && propertySet2.containsAll(propertySet1);
	}
	
	public int[] createReorderMapping(BlockRepresentative<?> other) {
		if(this == other) return IntStream.range(0, getPropertyCount()).toArray();

		if(!this.compatible(other)) return null;

		final int propertyCount = this.getPropertyCount();
		
		if(propertyCount == 0) return new int[0];
		
//		Object2IntArrayMap<PropertyRepresentative> propertyMap = new Object2IntArrayMap<PropertyRepresentative>(propertyCount);
		final int[] remapping = new int[propertyCount];
		
		for(int i = 0; i < propertyCount; ++i) {
			PropertyRepresentative property = this.properties[i];
			int index = other.getPropertyIndex(property);
			if(index == -1) return null;
			remapping[i] = index;
		}
//		Arrays.setAll(remapping, index -> other.getPropertyIndex(this.getProperty(index)));
		
		return remapping;
	}
}
