package com.joshiegemfinder.synchronisedblockstates.common.util;

import java.util.Arrays;
import java.util.Objects;

public class StateDefinitionRepresentative {

	public final BlockRepresentative<?> blockType;
	public final int stateCount;
	public final BlockStateRepresentative[] states;
	public final int[] stateIndexes;
	
	public StateDefinitionRepresentative(BlockRepresentative<?> blockType, int stateCount) {
		this.blockType = Objects.requireNonNull(blockType);
		this.stateCount = stateCount;
		this.states = new BlockStateRepresentative[stateCount];
		this.stateIndexes = new int[stateCount];
		Arrays.fill(stateIndexes, -1);
	}

	public StateDefinitionRepresentative(BlockRepresentative<?> blockType, BlockStateRepresentative[] states) {
		this(blockType, states.length);
		System.arraycopy(states, 0, this.states, 0, stateCount);
	}

	public StateDefinitionRepresentative(BlockRepresentative<?> blockType, BlockStateRepresentative[] states, int[] stateIndexes) {
		this(blockType, states);
		System.arraycopy(stateIndexes, 0, this.stateIndexes, 0, stateCount);
	}
	
	public StateDefinitionRepresentative(BlockRepresentative<?> blockType, IndexHolder<BlockStateRepresentative>[] states) {
		this(blockType, states.length);
		for(int i = 0; i < stateCount; ++i) {
			IndexHolder<BlockStateRepresentative> holder = states[i];
			this.states[i] = holder.state();
			this.stateIndexes[i] = holder.index();
		}
	}

	public void update() {}

	public void validate() {
		this.validate(true);
	}
	
	public void validate(boolean enforceOrder) {
		// Ensure we have a block type
		final BlockRepresentative<?> blockType = Objects.requireNonNull(this.blockType);
		
		for(int i = 0; i < stateCount; ++i) {
			// Ensure no states are null
			var state = Objects.requireNonNull(states[i]);
			// Ensure state has a block type
			final BlockRepresentative<?> stateBlockType = Objects.requireNonNull(state.getBlock());
			
			// Ensure every state is compatible with this representative
			if(stateBlockType != blockType && !blockType.equalsStrict(stateBlockType)) {
				if(!blockType.compatible(stateBlockType)) {
					throw new IllegalStateException("state " + state + " has block " + stateBlockType + " that is incompatible with holder block " + blockType);
				} else if(enforceOrder) {
					// Replace stateBlockType properties with blockType
					states[i] = state.reorderFor(blockType);
				}
			}
		}
	}
	
	public StateDefinitionRepresentative withPropertiesReordered(PropertyRepresentative[] propertyOrder) {
		if(!blockType.propertiesCompatible(propertyOrder)) {
			throw new IllegalArgumentException("Incompatible properties " + Arrays.toString(propertyOrder) + " for " + blockType);
		}
		
		this.validate(false);
		
		BlockRepresentative<?> newBlock = new BlockRepresentative<>(blockType.getKey(), propertyOrder);
		int[] remapping = blockType.createReorderMapping(newBlock);
		
		StateDefinitionRepresentative newDefinition = new StateDefinitionRepresentative(newBlock, stateCount);
		
		for(int stateIndex = 0; stateIndex < stateCount; ++stateIndex) {
			var state = states[stateIndex];
			var newState = state.withBlock(newBlock);
			for(int i = 0; i < remapping.length; ++i) {
				int index = remapping[i];
				newState.putValue(index, state.getValue(index));
			}
			newDefinition.states[stateIndex] = newState;
		}

		System.arraycopy(this.stateIndexes, 0, newDefinition.stateIndexes, 0, stateCount);
		
		return newDefinition;
	}
}
