/*
 *    This file is part of the Map Link mod
 *    licensed under the GNU GPL v3 License.
 *
 *    Copyright (C) 2025  Leander Knüttel and contributors
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package de.the_build_craft.maplink.common.clientMapHandlers;

import com.google.common.hash.Hashing;
import de.the_build_craft.maplink.common.AbstractModInitializer;
import de.the_build_craft.maplink.common.HTTP;
import de.the_build_craft.maplink.common.ModConfig;
import de.the_build_craft.maplink.common.waypoints.*;
import de.the_build_craft.maplink.common.wrappers.Text;
import de.the_build_craft.maplink.common.wrappers.Utils;
import java.net.URI;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import net.minecraft.class_1011;
import net.minecraft.class_1043;
import net.minecraft.class_156;
import net.minecraft.class_2960;
import net.minecraft.class_310;

import static de.the_build_craft.maplink.common.CommonModConfig.*;
import static de.the_build_craft.maplink.common.FastUpdateTask.playerPositions;

/**
 * @author Leander Knüttel
 * @version 07.09.2025
 */
public abstract class ClientMapHandler {
    public static final String waypointPrefix = "maplink_";
    private static final Map<String, class_1011> iconLinkToNativeImage = new HashMap<>();
    private static final Map<String, class_1043> iconLinkToTexture = new HashMap<>();

    private static final Map<String, WaypointState> idToWaypointState = new ConcurrentHashMap<>();

    private static ClientMapHandler instance;
    protected final class_310 mc;

    private static final int maxMarkerCountBeforeWarning = 25;
    private boolean markerMessageWasShown = false;

    private int previousPlayerWaypointColor = 0;
    private int previousMarkerWaypointColor = 0;
    private int previousFriendWaypointColor = 0;
    private boolean previousFriendColorOverride = false;
    private int previousFriendListHashCode = 0;

    private final List<PlayerPosition> tempPlayerWaypointPositions = new ArrayList<>();
    protected final Set<String> currentPlayerIds = new HashSet<>();
    protected final Set<String> currentMarkerIds = new HashSet<>();

    public ClientMapHandler() {
        instance = this;
        mc = class_310.method_1551();
    }

    public static ClientMapHandler getInstance() {
        return instance;
    }

    public static WaypointState getWaypointState(String id) {
        return ClientMapHandler.idToWaypointState.get(id);
    }

    public static void registerPlayerPosition(PlayerPosition playerPosition, String iconLink) {
        if (config.general.useMcHeadsPlayerNameIcons) {
            iconLink = "https://mc-heads.net/avatar/" + playerPosition.name + "/32";
        }
        registerPosition(playerPosition, iconLink, true, false);
    }

    public static void registerTempPlayerPosition(PlayerPosition playerPosition) {
        registerPosition(playerPosition, "https://mc-heads.net/avatar/" + playerPosition.name + "/32", true, true);
    }

    public static void registerPosition(Position position, String iconLink) {
        registerPosition(position, iconLink, false, false);
    }

    public static void registerPosition(Position position, String iconLink, boolean isPlayer, boolean isTemp) {
        WaypointState waypointState = idToWaypointState.get(position.id);
        if (waypointState != null && (!waypointState.isTemp || isTemp)) return;
        if (!iconLinkToNativeImage.containsKey(iconLink)) {
            try {
                class_1011 nativeImage;
                try {
                    nativeImage = HTTP.makeImageHttpRequest(URI.create(iconLink).toURL());
                } catch (Exception e) {
                    if (isPlayer) {
                        nativeImage = HTTP.makeImageHttpRequest(URI.create("https://mc-heads.net/avatar/" + position.name + "/32").toURL());
                    } else {
                        throw e;
                    }
                }
                if (nativeImage.method_4307() == nativeImage.method_4323() && nativeImage.method_4307() <= 64) {
                    iconLinkToNativeImage.put(iconLink, nativeImage);
                } else {
                    class_1011 nativeImage1 = new class_1011(64, 64, true);
                    nativeImage.method_4300(0, 0, nativeImage.method_4307(), nativeImage.method_4323(), nativeImage1);
                    iconLinkToNativeImage.put(iconLink, nativeImage1);
                    nativeImage.close();
                }
            } catch (Exception e) {
                iconLink = null;
            }
        }
        WaypointState prev = idToWaypointState.put(position.id, new WaypointState(position.name, iconLink, isPlayer, isTemp));
        if (prev != null) prev.isOld = true;
    }

    public static void clearRegisteredPositions() {
        idToWaypointState.clear();
    }

    //partially from Earthcomputer/minimap-sync licensed under the MIT License
    public static class_1043 getDynamicTexture(String link) {
        if (iconLinkToTexture.containsKey(link)) return iconLinkToTexture.get(link);
        if (!iconLinkToNativeImage.containsKey(link)) return null;
        #if MC_VER >= MC_1_21_5
        DynamicTexture texture = new DynamicTexture(() -> link, iconLinkToNativeImage.get(link));
        #else
        DynamicTexture texture = new DynamicTexture(iconLinkToNativeImage.get(link));
        #endif
        texture.setFilter(false, false);
        Minecraft.getInstance().getTextureManager().register(ClientMapHandler.getIconResourceLocation(link), texture);
        DynamicTexture old = iconLinkToTexture.put(link, texture);
        if (old != null) {
            old.close();
        }
        return texture;
    }

    //partially from Earthcomputer/minimap-sync licensed under the MIT License
    public static class_2960 getIconResourceLocation(String icon) {
        #if MC_VER >= MC_1_21_5
        return ResourceLocation.fromNamespaceAndPath("maplink", "xaeros_" + ClientMapHandler.makeResourceSafeString(icon));
        #elif MC_VER >= MC_1_19_2
        return ResourceLocation.tryBuild("maplink", "xaeros_" + ClientMapHandler.makeResourceSafeString(icon));
        #else
        return new ResourceLocation("maplink", "xaeros_" + ClientMapHandler.makeResourceSafeString(icon));
        #endif
    }

    //from Earthcomputer/minimap-sync licensed under the MIT License
    public static String makeResourceSafeString(String original) {
        //noinspection deprecation
        String hash = Hashing.sha1().hashUnencodedChars(original).toString();
        original = original.toLowerCase(Locale.ROOT);
        original = class_156.method_30309(original, class_2960::method_29184);
        return original + "/" + hash;
    }

    public void handlePlayerWaypoints() {
        if (mc.cameraEntity == null || mc.level == null) return;
        if (config.general.enablePlayerWaypoints) {
            // Keep track of which waypoints were previously shown
            // to remove any that are not to be shown anymore
            currentPlayerIds.clear();

            Map<String, AbstractClientPlayer> playerClientEntityMap = mc.level.players().stream().collect(
                    Collectors.toMap(a -> a.getGameProfile().getName(), a -> a));

            int maxHudD = config.hud.maxPlayerDistance;
            int minMiniD = config.minimap.minPlayerDistance;
            int maxMiniD = config.minimap.maxPlayerDistance;
            int minWorldD = config.worldmap.minPlayerDistance;
            int maxWorldD = config.worldmap.maxPlayerDistance;

            int minIconHudD = config.hud.minPlayerIconDistance;
            int maxIconHudD = config.hud.maxPlayerIconDistance;
            int minIconMiniD = config.minimap.minPlayerIconDistance;
            int maxIconMiniD = config.minimap.maxPlayerIconDistance;
            int minIconWorldD = config.worldmap.minPlayerIconDistance;
            int maxIconWorldD = config.worldmap.maxPlayerIconDistance;

            int onHud = 0;
            int maxOnHud = config.hud.maxPlayerWaypoints;
            int onMiniMap = 0;
            int maxOnMiniMap = config.minimap.maxPlayerWaypoints;
            int onWorldMap = 0;
            int maxOnWorldMap = config.worldmap.maxPlayerWaypoints;

            int iconsOnHud = 0;
            int maxIconsOnHud = config.hud.maxPlayerIconWaypoints;
            int iconsOnMiniMap = 0;
            int maxIconsOnMiniMap = config.minimap.maxPlayerIconWaypoints;
            int iconsOnWorldMap = 0;
            int maxIconsOnWorldMap = config.worldmap.maxPlayerIconWaypoints;

            boolean onlyShowFriends = config.friends.onlyShowFriendsWaypoints;
            boolean onlyShowFriendsIcons = config.friends.onlyShowFriendsIconWaypoints;
            boolean alwaysShowFriends = config.friends.alwaysShowFriendsWaypoints;
            boolean alwaysShowFriendIcons = config.friends.alwaysShowFriendsIconWaypoints;

            boolean enablePlayerIcons = config.general.enablePlayerIconWaypoints;

            Vec3 cameraPos = mc.cameraEntity.position();
            tempPlayerWaypointPositions.clear();
            tempPlayerWaypointPositions.addAll(playerPositions.values());
            tempPlayerWaypointPositions.sort(Comparator.comparing(p -> cameraPos.distanceToSqr(p.pos.x, p.pos.y, p.pos.z)));

            for (PlayerPosition playerPosition : tempPlayerWaypointPositions) {
                if (playerPosition == null) continue;
                String playerName = playerPosition.name;
                WaypointState waypointState = idToWaypointState.get(playerPosition.id);

                boolean isFriend = config.friends.friendList.contains(playerName);
                if (onlyShowFriends && !isFriend) continue;

                waypointState.renderOnHud = config.hud.showPlayerWaypoints != ModConfig.ConditionalActiveMode.NEVER;
                waypointState.renderOnMiniMap = config.minimap.showPlayerWaypoints != ModConfig.ConditionalActiveMode.NEVER;
                waypointState.renderOnWorldMap = config.worldmap.showPlayerWaypoints != ModConfig.ConditionalActiveMode.NEVER;

                // Check if this player is within the server's player entity tracking range
                int minHudD = config.hud.minNotVisiblePlayerDistance;
                if (playerClientEntityMap.containsKey(playerName)) {
                    if (config.hud.hidePlayersInRange) waypointState.renderOnHud = false;
                    if (config.minimap.hidePlayersInRange) waypointState.renderOnMiniMap = false;
                    if (config.worldmap.hidePlayersInRange) waypointState.renderOnWorldMap = false;

                    if (waypointState.renderOnHud) {
                        #if MC_VER >= MC_1_17_1
                        ClipContext clipContext = new ClipContext(mc.cameraEntity.getEyePosition(),
                        #else
                        ClipContext clipContext = new ClipContext(mc.cameraEntity.getEyePosition(1),
                        #endif
                                playerClientEntityMap.get(playerName).position().add(0, 1, 0),
                                ClipContext.Block.VISUAL, ClipContext.Fluid.ANY, mc.cameraEntity);
                        // If this player is visible, don't show waypoint on Hud (depending on the config settings)
                        if (mc.level.clip(clipContext).getType() != HitResult.Type.BLOCK) {
                            if (config.hud.hidePlayersVisible) waypointState.renderOnHud = false;
                            minHudD = config.hud.minVisiblePlayerDistance;
                        }
                    }
                }

                double d = cameraPos.distanceTo(new Vec3(playerPosition.pos.x, playerPosition.pos.y, playerPosition.pos.z));

                if (alwaysShowFriends && isFriend) {
                    waypointState.renderOnHud &= d >= minHudD;
                } else {
                    waypointState.renderOnHud &= onHud < maxOnHud && d >= minHudD && d <= maxHudD;
                    waypointState.renderOnMiniMap &= onMiniMap < maxOnMiniMap && d >= minMiniD && d <= maxMiniD;
                    waypointState.renderOnWorldMap &= onWorldMap < maxOnWorldMap && d >= minWorldD && d <= maxWorldD;
                    if (waypointState.renderOnHud) onHud++;
                    if (waypointState.renderOnMiniMap) onMiniMap++;
                    if (waypointState.renderOnWorldMap) onWorldMap++;
                }

                if (!(waypointState.renderOnHud || waypointState.renderOnMiniMap || waypointState.renderOnWorldMap)) continue;

                if (!enablePlayerIcons || !waypointState.hasIcon) {
                    waypointState.renderIconOnHud = false;
                    waypointState.renderIconOnMiniMap = false;
                    waypointState.renderIconOnWorldMap = false;
                } else if (alwaysShowFriendIcons && isFriend) {
                    waypointState.renderIconOnHud = true;
                    waypointState.renderIconOnMiniMap = true;
                    waypointState.renderIconOnWorldMap = true;
                } else {
                    waypointState.renderIconOnHud = iconsOnHud < maxIconsOnHud && d >= minIconHudD && d <= maxIconHudD;
                    waypointState.renderIconOnMiniMap = iconsOnMiniMap < maxIconsOnMiniMap && d >= minIconMiniD && d <= maxIconMiniD;
                    waypointState.renderIconOnWorldMap = iconsOnWorldMap < maxIconsOnWorldMap && d >= minIconWorldD && d <= maxIconWorldD;
                    if (onlyShowFriendsIcons) {
                        waypointState.renderIconOnHud &= isFriend;
                        waypointState.renderIconOnMiniMap &= isFriend;
                        waypointState.renderIconOnWorldMap &= isFriend;
                    }
                    if (waypointState.renderIconOnHud) iconsOnHud++;
                    if (waypointState.renderIconOnMiniMap) iconsOnMiniMap++;
                    if (waypointState.renderIconOnWorldMap) iconsOnWorldMap++;
                }

                if (waypointState.renderOnHud || waypointState.renderOnMiniMap || waypointState.renderOnWorldMap) {
                    currentPlayerIds.add(playerPosition.id);
                    if (!config.general.showPlayerWaypointsAsTrackedPlayers) addOrUpdatePlayerWaypoint(playerPosition, waypointState);
                }
            }

            removeOldPlayerWaypoints();

            int newPlayerWaypointColor = config.general.playerWaypointColor.ordinal();
            int newFriendWaypointColor = config.friends.friendWaypointColor.ordinal();
            boolean newFriendColorOverride = config.friends.overwriteFriendWaypointColor;
            int newFriendListHashCode = config.friends.friendList.hashCode();
            if ((previousPlayerWaypointColor != newPlayerWaypointColor)
                    || (previousFriendWaypointColor != newFriendWaypointColor)
                    || (previousFriendColorOverride != newFriendColorOverride)
                    || (previousFriendListHashCode != newFriendListHashCode)) {
                previousPlayerWaypointColor = newPlayerWaypointColor;
                previousFriendWaypointColor = newFriendWaypointColor;
                previousFriendColorOverride = newFriendColorOverride;
                previousFriendListHashCode = newFriendListHashCode;

                updatePlayerWaypointColors();
            }
        } else {
            removeAllPlayerWaypoints();
        }
    }

    public abstract void reset();

    abstract void addOrUpdatePlayerWaypoint(PlayerPosition playerPosition, WaypointState waypointState);

    abstract void removeOldPlayerWaypoints();

    abstract void removeAllPlayerWaypoints();

    abstract void updatePlayerWaypointColors();

    public void handleMarkerWaypoints(List<Position> markerPositions) {
        if (mc.cameraEntity == null) return;
        ModConfig.ServerEntry serverEntry = getCurrentServerEntry();
        if (config.general.enableMarkerWaypoints && serverEntry != null) {
            // Keep track of which waypoints were previously shown
            // to remove any that are not to be shown anymore
            currentMarkerIds.clear();

            int minHudD = config.hud.minMarkerDistance;
            int maxHudD = config.hud.maxMarkerDistance;
            int minMiniD = config.minimap.minMarkerDistance;
            int maxMiniD = config.minimap.maxMarkerDistance;
            int minWorldD = config.worldmap.minMarkerDistance;
            int maxWorldD = config.worldmap.maxMarkerDistance;

            int minIconHudD = config.hud.minMarkerIconDistance;
            int maxIconHudD = config.hud.maxMarkerIconDistance;
            int minIconMiniD = config.minimap.minMarkerIconDistance;
            int maxIconMiniD = config.minimap.maxMarkerIconDistance;
            int minIconWorldD = config.worldmap.minMarkerIconDistance;
            int maxIconWorldD = config.worldmap.maxMarkerIconDistance;

            int onHud = 0;
            int maxOnHud = config.hud.maxMarkerWaypoints;
            int onMiniMap = 0;
            int maxOnMiniMap = config.minimap.maxMarkerWaypoints;
            int onWorldMap = 0;
            int maxOnWorldMap = config.worldmap.maxMarkerWaypoints;

            int iconsOnHud = 0;
            int maxIconsOnHud = config.hud.maxMarkerIconWaypoints;
            int iconsOnMiniMap = 0;
            int maxIconsOnMiniMap = config.minimap.maxMarkerIconWaypoints;
            int iconsOnWorldMap = 0;
            int maxIconsOnWorldMap = config.worldmap.maxMarkerIconWaypoints;

            boolean enableMarkerIcons = config.general.enableMarkerIcons;

            Vec3 cameraPos = mc.cameraEntity.position();
            markerPositions.sort(Comparator.comparing(p -> cameraPos.distanceToSqr(p.pos.x, p.pos.y, p.pos.z)));

            for (Position markerPosition : markerPositions) {
                WaypointState waypointState = idToWaypointState.get(markerPosition.id);
                double d = mc.cameraEntity.position().distanceTo(new Vec3(markerPosition.pos.x, markerPosition.pos.y, markerPosition.pos.z));

                waypointState.renderOnHud = config.hud.showMarkerWaypoints != ModConfig.ConditionalActiveMode.NEVER
                        && onHud < maxOnHud && d >= minHudD && d <= maxHudD;
                waypointState.renderOnMiniMap = config.minimap.showMarkerWaypoints != ModConfig.ConditionalActiveMode.NEVER
                        && onMiniMap < maxOnMiniMap && d >= minMiniD && d <= maxMiniD;
                waypointState.renderOnWorldMap = config.worldmap.showMarkerWaypoints != ModConfig.ConditionalActiveMode.NEVER
                        && onWorldMap < maxOnWorldMap && d >= minWorldD && d <= maxWorldD;
                if (waypointState.renderOnHud) onHud++;
                if (waypointState.renderOnMiniMap) onMiniMap++;
                if (waypointState.renderOnWorldMap) onWorldMap++;

                if (!(waypointState.renderOnHud || waypointState.renderOnMiniMap || waypointState.renderOnWorldMap)) continue;

                if (enableMarkerIcons && waypointState.hasIcon
                        && serverEntry.includeIconMarkerLayer(markerPosition.layer.id) && serverEntry.includeIcon(markerPosition.name)) {
                    waypointState.renderIconOnHud = iconsOnHud < maxIconsOnHud && d >= minIconHudD && d <= maxIconHudD;
                    waypointState.renderIconOnMiniMap = iconsOnMiniMap < maxIconsOnMiniMap && d >= minIconMiniD && d <= maxIconMiniD;
                    waypointState.renderIconOnWorldMap = iconsOnWorldMap < maxIconsOnWorldMap && d >= minIconWorldD && d <= maxIconWorldD;
                    if (waypointState.renderIconOnHud) iconsOnHud++;
                    if (waypointState.renderIconOnMiniMap) iconsOnMiniMap++;
                    if (waypointState.renderIconOnWorldMap) iconsOnWorldMap++;
                } else {
                    waypointState.renderIconOnHud = false;
                    waypointState.renderIconOnMiniMap = false;
                    waypointState.renderIconOnWorldMap = false;
                }

                currentMarkerIds.add(markerPosition.id);
                addOrUpdateMarkerWaypoint(markerPosition, waypointState);
            }

            removeOldMarkerWaypoints();

            int newMarkerWaypointColor = config.general.markerWaypointColor.ordinal();
            if (previousMarkerWaypointColor != newMarkerWaypointColor) {
                previousMarkerWaypointColor = newMarkerWaypointColor;
                updateMarkerWaypointColors();
            }

            if (!markerMessageWasShown && (onHud > maxMarkerCountBeforeWarning || onMiniMap > maxMarkerCountBeforeWarning) && !config.general.ignoreMarkerMessage) {
                markerMessageWasShown = true;
                Utils.sendToClientChat(Text.literal("[" + AbstractModInitializer.MOD_NAME + "]: " +
                                "Looks like you have quite a lot of markers from the server visible! " +
                                "Did you know that you can chose the marker layers that are shown in the config, decrease their maximum distance, set a limit on how many are displayed or disable marker waypoints entirely? (The default config already limits the amount to 40) ")
                        .withStyle(Style.EMPTY.withColor(ChatFormatting.GOLD))
                        .append(Text.literal("[Don't show this again]")
                                .withStyle(Style.EMPTY.withClickEvent(
                                    #if MC_VER < MC_1_21_5
                                    new ClickEvent(ClickEvent.Action.SUGGEST_COMMAND, "/" + AbstractModInitializer.MOD_ID + " ignore_marker_message"))
                                    #else
                                    new ClickEvent.SuggestCommand("/" + AbstractModInitializer.MOD_ID + " ignore_marker_message"))
                                    #endif
                                        .withColor(ChatFormatting.GREEN).withBold(true))));
            }
        } else {
            removeAllMarkerWaypoints();
        }
    }

    abstract void addOrUpdateMarkerWaypoint(Position markerPosition, WaypointState waypointState);

    abstract void removeOldMarkerWaypoints();

    public abstract void removeAllMarkerWaypoints();

    abstract void updateMarkerWaypointColors();

    public abstract void handleAreaMarkers(List<AreaMarker> markerPositions);

    public abstract void removeAllAreaMarkers(boolean clearXaeroHash);
}
