/*
 * Decompiled with CFR 0.152.
 */
package net.zenith.hoyocraft.invasion;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.UUID;
import net.minecraft.ChatFormatting;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Holder;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.network.protocol.Packet;
import net.minecraft.network.protocol.game.ClientboundSetTitleTextPacket;
import net.minecraft.network.protocol.game.ClientboundSetTitlesAnimationPacket;
import net.minecraft.network.protocol.game.ClientboundSoundPacket;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.sounds.SoundSource;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.levelgen.Heightmap;
import net.zenith.hoyocraft.entity.ModEntities;
import net.zenith.hoyocraft.entity.custom.abyssal.invasion.AbyssPortalEntity;
import net.zenith.hoyocraft.invasion.IncursionPattern;
import net.zenith.hoyocraft.invasion.IncursionPatternManager;
import net.zenith.hoyocraft.invasion.InvasionSavedData;
import net.zenith.hoyocraft.network.ModNetworking;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class AbyssalInvasionManager {
    private static final Logger LOGGER = LogManager.getLogger();
    private static AbyssalInvasionManager instance;
    private final Random random = new Random();
    private boolean hasReinitializedOnLoad = false;

    private AbyssalInvasionManager() {
    }

    public static AbyssalInvasionManager getInstance() {
        if (instance == null) {
            instance = new AbyssalInvasionManager();
        }
        return instance;
    }

    public void onServerTick(ServerLevel level) {
        if (!this.hasReinitializedOnLoad) {
            this.reinitializeAppropriateState(level);
            this.hasReinitializedOnLoad = true;
        }
        InvasionSavedData savedData = InvasionSavedData.get(level);
        if (savedData.invasionsStopped) {
            return;
        }
        if (savedData.invasionActive && savedData.portalsToSpawn > 0) {
            if (savedData.nextPortalSpawnCooldown > 0) {
                --savedData.nextPortalSpawnCooldown;
            } else {
                this.spawnOnePortalAndScheduleNext(level);
            }
        }
        long currentTime = level.getDayTime();
        long currentDay = currentTime / 24000L;
        Map<ResourceLocation, IncursionPattern> timedPatterns = IncursionPatternManager.getInstance().getTimedPatterns();
        for (Map.Entry<ResourceLocation, IncursionPattern> entry : timedPatterns.entrySet()) {
            ResourceLocation patternId = entry.getKey();
            IncursionPattern pattern = entry.getValue();
            pattern.triggerDay().ifPresent(triggerDay -> {
                if (currentDay > 0L && currentDay == (long)triggerDay.intValue() && currentTime % 24000L == 100L && savedData.lastInvasionTime / 24000L < currentDay) {
                    LOGGER.info("Triggering timed incursion '{}' for day {}.", (Object)patternId, (Object)currentDay);
                    savedData.lastInvasionTime = currentTime;
                    savedData.setDirty();
                    this.startIncursion(level, patternId);
                }
            });
        }
    }

    private void reinitializeAppropriateState(ServerLevel level) {
        LOGGER.info("Hoyocraft: Performing post-load state check for Abyssal Invasions...");
        InvasionSavedData savedData = InvasionSavedData.get(level);
        if (savedData.invasionActive) {
            LOGGER.info("Found active invasion in saved data. Re-awakening portals...");
            Optional<ResourceLocation> patternIdOpt = savedData.activeIncursionPatternId;
            if (patternIdOpt.isEmpty()) {
                LOGGER.error("Active invasion found, but pattern ID is missing from saved data! Cannot resume.");
                this.endIncursion(level, false);
                return;
            }
            Optional<IncursionPattern> patternOpt = IncursionPatternManager.getInstance().getPattern(patternIdOpt.get());
            if (patternOpt.isEmpty()) {
                LOGGER.error("Active invasion found, but pattern '{}' is no longer registered! Cannot resume.", (Object)patternIdOpt.get());
                this.endIncursion(level, false);
                return;
            }
            IncursionPattern pattern = patternOpt.get();
            UUID invasionTargetUUID = null;
            if (!savedData.activePortalUUIDs.isEmpty() && !level.players().isEmpty()) {
                invasionTargetUUID = ((ServerPlayer)level.players().get(this.random.nextInt(level.players().size()))).getUUID();
            }
            savedData.portalsEverSpawned = savedData.activePortalUUIDs.size();
            for (UUID portalUUID : new ArrayList<UUID>(savedData.activePortalUUIDs)) {
                Entity entity = level.getEntity(portalUUID);
                if (entity instanceof AbyssPortalEntity) {
                    AbyssPortalEntity portal = (AbyssPortalEntity)entity;
                    LOGGER.info("Re-initializing portal {}", (Object)portal.getId());
                    portal.reinitializeFromSave(pattern.waves(), pattern.waveIntervalTicks(), invasionTargetUUID);
                    continue;
                }
                LOGGER.warn("Could not find portal entity with UUID {} from saved data.", (Object)portalUUID);
                savedData.activePortalUUIDs.remove(portalUUID);
            }
            savedData.activePortals = savedData.activePortalUUIDs.size();
            savedData.setDirty();
            this.syncInvasionState(level);
        }
    }

    public boolean startIncursion(ServerLevel level, ResourceLocation patternId) {
        InvasionSavedData savedData = InvasionSavedData.get(level);
        if (savedData.invasionActive) {
            LOGGER.warn("Tried to start an incursion while one is already active.");
            return false;
        }
        Optional<IncursionPattern> patternOpt = IncursionPatternManager.getInstance().getPattern(patternId);
        if (patternOpt.isEmpty()) {
            LOGGER.error("Failed to start incursion: Pattern with ID '{}' not found.", (Object)patternId);
            return false;
        }
        IncursionPattern pattern = patternOpt.get();
        List players = level.getServer().getPlayerList().getPlayers();
        if (players.isEmpty()) {
            LOGGER.warn("No players online to start incursion.");
            return false;
        }
        this.announceIncursionStart(level, pattern);
        ServerPlayer targetPlayer = (ServerPlayer)players.get(this.random.nextInt(players.size()));
        LOGGER.info("Starting incursion '{}' using pattern '{}', centered on {}", (Object)pattern.incursionName(), (Object)patternId, (Object)targetPlayer.getName().getString());
        savedData.invasionActive = true;
        savedData.activeIncursionPatternId = Optional.of(patternId);
        savedData.totalPortals = pattern.portalCount();
        savedData.portalsEverSpawned = 0;
        savedData.activePortals = 0;
        savedData.activePortalUUIDs.clear();
        savedData.portalsToSpawn = pattern.portalCount();
        savedData.nextPortalSpawnCooldown = 1;
        savedData.setDirty();
        this.syncInvasionState(level);
        return true;
    }

    private void announceIncursionStart(ServerLevel level, IncursionPattern pattern) {
        Optional soundEventHolder;
        Optional soundEventOpt;
        MutableComponent titleText = Component.literal((String)pattern.announcementTitle()).withStyle(ChatFormatting.DARK_RED);
        ClientboundSetTitlesAnimationPacket titlesAnimationPacket = new ClientboundSetTitlesAnimationPacket(20, 100, 40);
        ClientboundSetTitleTextPacket titleTextPacket = new ClientboundSetTitleTextPacket((Component)titleText);
        Optional<ResourceLocation> startSoundRL = pattern.sounds().startSound();
        if (startSoundRL.isPresent() && (soundEventOpt = BuiltInRegistries.SOUND_EVENT.getOptional(startSoundRL.get())).isPresent() && (soundEventHolder = BuiltInRegistries.SOUND_EVENT.getHolder(startSoundRL.get())).isPresent()) {
            for (ServerPlayer player : level.getServer().getPlayerList().getPlayers()) {
                player.connection.send((Packet)new ClientboundSoundPacket((Holder)soundEventHolder.get(), SoundSource.MASTER, player.getX(), player.getY(), player.getZ(), 2.0f, 0.8f, level.getRandom().nextLong()));
            }
        }
        level.getServer().getPlayerList().broadcastAll((Packet)titlesAnimationPacket);
        level.getServer().getPlayerList().broadcastAll((Packet)titleTextPacket);
        LOGGER.info("Broadcasted incursion announcement for '{}'", (Object)pattern.incursionName());
    }

    private void spawnOnePortalAndScheduleNext(ServerLevel level) {
        InvasionSavedData savedData = InvasionSavedData.get(level);
        Optional patternOpt = savedData.activeIncursionPatternId.flatMap(id -> IncursionPatternManager.getInstance().getPattern((ResourceLocation)id));
        if (patternOpt.isEmpty()) {
            LOGGER.error("Cannot spawn next portal: Active incursion pattern is missing or invalid.");
            this.endIncursion(level, false);
            return;
        }
        IncursionPattern pattern = (IncursionPattern)patternOpt.get();
        List players = level.getServer().getPlayerList().getPlayers();
        if (players.isEmpty()) {
            LOGGER.warn("No players online to spawn next portal. Pausing spawn sequence.");
            savedData.nextPortalSpawnCooldown = 200;
            savedData.setDirty();
            return;
        }
        ServerPlayer targetPlayer = (ServerPlayer)players.get(this.random.nextInt(players.size()));
        this.spawnPortal(level, (Player)targetPlayer, pattern);
        --savedData.portalsToSpawn;
        ++savedData.activePortals;
        this.syncInvasionState(level);
        ++savedData.portalsEverSpawned;
        this.syncInvasionState(level);
        savedData.nextPortalSpawnCooldown = savedData.portalsToSpawn > 0 ? pattern.portalSpawnIntervalTicks() : 0;
        savedData.setDirty();
    }

    private void spawnPortal(ServerLevel level, Player center, IncursionPattern pattern) {
        InvasionSavedData savedData = InvasionSavedData.get(level);
        BlockPos playerPos = center.blockPosition();
        for (int i = 0; i < 10; ++i) {
            AbyssPortalEntity portal;
            int z;
            int y;
            double angle = this.random.nextDouble() * 2.0 * Math.PI;
            double distance = this.random.nextDouble((double)pattern.spawnRadius() / 2.0, pattern.spawnRadius());
            int x = playerPos.getX() + (int)(distance * Math.cos(angle));
            BlockPos spawnPos = new BlockPos(x, y = level.getHeight(Heightmap.Types.WORLD_SURFACE, x, z = playerPos.getZ() + (int)(distance * Math.sin(angle))), z);
            if (!level.isLoaded(spawnPos) || !level.getWorldBorder().isWithinBounds(spawnPos) || (portal = (AbyssPortalEntity)ModEntities.ABYSS_PORTAL.get().create((Level)level)) == null) continue;
            portal.setPos((double)spawnPos.getX() + 0.5, spawnPos.getY(), (double)spawnPos.getZ() + 0.5);
            portal.setInvasionData(pattern.waves(), pattern.waveIntervalTicks(), center.getUUID());
            level.addFreshEntity((Entity)portal);
            savedData.activePortalUUIDs.add(portal.getUUID());
            LOGGER.info("Spawned Abyss Portal with UUID {} for incursion '{}'", (Object)portal.getUUID(), (Object)pattern.incursionName());
            return;
        }
    }

    public void onPortalDestroyed(ServerLevel level, AbyssPortalEntity portal) {
        InvasionSavedData savedData = InvasionSavedData.get(level);
        if (!savedData.invasionActive || !savedData.activePortalUUIDs.contains(portal.getUUID())) {
            return;
        }
        savedData.activePortalUUIDs.remove(portal.getUUID());
        savedData.activePortals = savedData.activePortalUUIDs.size();
        savedData.setDirty();
        this.syncInvasionState(level);
        LOGGER.info("Portal {} destroyed. Remaining portals: {}", (Object)portal.getId(), (Object)savedData.activePortals);
        if (savedData.activePortals <= 0) {
            this.endIncursion(level, true);
        }
    }

    public void endIncursion(ServerLevel level, boolean success) {
        InvasionSavedData savedData = InvasionSavedData.get(level);
        if (!savedData.invasionActive) {
            return;
        }
        LOGGER.info("Incursion ended. Success: {}", (Object)success);
        savedData.resetActiveInvasion();
        this.syncInvasionState(level);
    }

    public void forceStopAllIncursions(ServerLevel level) {
        InvasionSavedData savedData = InvasionSavedData.get(level);
        savedData.invasionsStopped = true;
        if (savedData.invasionActive) {
            for (UUID portalUUID : new ArrayList<UUID>(savedData.activePortalUUIDs)) {
                Entity portal = level.getEntity(portalUUID);
                if (!(portal instanceof AbyssPortalEntity)) continue;
                portal.discard();
            }
            this.endIncursion(level, false);
        }
        savedData.setDirty();
        LOGGER.info("All Abyssal Incursions have been forcefully stopped by command.");
    }

    public void resetInvasionSystem(ServerLevel level) {
        this.forceStopAllIncursions(level);
        InvasionSavedData savedData = InvasionSavedData.get(level);
        savedData.lastInvasionTime = 0L;
        savedData.invasionsStopped = false;
        savedData.setDirty();
        LOGGER.info("Abyssal Incursion system has been reset by command.");
    }

    private void syncInvasionState(ServerLevel level) {
        InvasionSavedData savedData = InvasionSavedData.get(level);
        ModNetworking.sendInvasionStateSync(savedData.invasionActive, savedData.activePortals, savedData.totalPortals, savedData.portalsEverSpawned);
    }
}

