package com.patapoke.client;

import com.patapoke.PatAPoke;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.gui.components.Button;
import net.minecraft.client.gui.components.EditBox;
import net.minecraft.client.gui.screens.Screen;
import net.minecraft.network.chat.CommonComponents;
import net.minecraft.network.chat.Component;

public class PatAPokeConfigScreen extends Screen {
    private static final int PANEL_WIDTH = 300;
    private static final int PANEL_PADDING = 18;
    private static final int ROW_SPACING = 34;
    private final Screen parent;
    private EditBox animationLengthField;
    private EditBox friendshipGainField;
    private EditBox cooldownField;
    private int panelLeft;
    private int panelTop;
    private int panelHeight;
    private int contentLeft;
    private int contentWidth;

    public PatAPokeConfigScreen(Screen parent) {
        super(Component.translatable("config.patapoke.title"));
        this.parent = parent;
    }

    @Override
    protected void init() {
        int centerX = this.width / 2;
        this.panelLeft = centerX - PANEL_WIDTH / 2;
        this.panelTop = Math.max(18, this.height / 5);
        this.contentLeft = this.panelLeft + PANEL_PADDING;
        this.contentWidth = PANEL_WIDTH - (PANEL_PADDING * 2);
        int fieldWidth = this.contentWidth;
        int fieldHeight = 20;
        int currentY = this.panelTop + 40;

        animationLengthField = new EditBox(
            this.font,
            this.contentLeft,
            currentY,
            fieldWidth,
            fieldHeight,
            Component.translatable("config.patapoke.animation_length")
        );
        animationLengthField.setMaxLength(16);
        animationLengthField.setValue(String.valueOf(PatAPoke.CONFIG.animationLengthSeconds));
        addRenderableWidget(animationLengthField);

        currentY += ROW_SPACING;
        friendshipGainField = new EditBox(
            this.font,
            this.contentLeft,
            currentY,
            fieldWidth,
            fieldHeight,
            Component.translatable("config.patapoke.friendship_gain")
        );
        friendshipGainField.setMaxLength(8);
        friendshipGainField.setValue(String.valueOf(PatAPoke.CONFIG.friendshipGain));
        addRenderableWidget(friendshipGainField);

        currentY += ROW_SPACING;
        cooldownField = new EditBox(
            this.font,
            this.contentLeft,
            currentY,
            fieldWidth,
            fieldHeight,
            Component.translatable("config.patapoke.cooldown")
        );
        cooldownField.setMaxLength(8);
        cooldownField.setValue(String.valueOf(PatAPoke.CONFIG.cooldownSeconds));
        addRenderableWidget(cooldownField);

        currentY += ROW_SPACING + 8;
        int buttonWidth = (fieldWidth - 10) / 2;
        addRenderableWidget(Button.builder(CommonComponents.GUI_DONE, button -> {
            applyAndSave();
            this.minecraft.setScreen(parent);
        }).bounds(this.contentLeft, currentY, buttonWidth, 20).build());

        addRenderableWidget(Button.builder(CommonComponents.GUI_CANCEL, button -> this.minecraft.setScreen(parent))
            .bounds(this.contentLeft + buttonWidth + 10, currentY, buttonWidth, 20).build());

        this.panelHeight = (currentY - this.panelTop) + 48;
    }

    @Override
    public void render(GuiGraphics guiGraphics, int mouseX, int mouseY, float partialTick) {
        super.render(guiGraphics, mouseX, mouseY, partialTick);
        drawOverlayText(guiGraphics);
    }

    @Override
    public void renderBackground(GuiGraphics guiGraphics, int mouseX, int mouseY, float partialTick) {
        this.renderTransparentBackground(guiGraphics);
        int panelRight = this.panelLeft + PANEL_WIDTH;
        int panelBottom = this.panelTop + this.panelHeight;

        guiGraphics.fill(this.panelLeft, this.panelTop, panelRight, panelBottom, 0xD0141419);
        guiGraphics.fillGradient(this.panelLeft, this.panelTop, panelRight, this.panelTop + 28, 0x603E3E5A, 0x201E1E27);
        guiGraphics.fill(this.panelLeft, this.panelTop, panelRight, this.panelTop + 2, 0x50FFFFFF);
        guiGraphics.fill(panelRight - 2, this.panelTop, panelRight, panelBottom, 0x50000000);
        guiGraphics.fill(this.panelLeft, panelBottom - 2, panelRight, panelBottom, 0x50000000);
        guiGraphics.fill(this.contentLeft - 4, this.panelTop + 32, this.contentLeft + this.contentWidth + 4, panelBottom - 12, 0x50282834);
    }

    private void applyAndSave() {
        double animationLength = parseDouble(animationLengthField.getValue(), PatAPoke.CONFIG.animationLengthSeconds);
        int friendshipGain = parseInt(friendshipGainField.getValue(), PatAPoke.CONFIG.friendshipGain);
        int cooldownSeconds = parseInt(cooldownField.getValue(), PatAPoke.CONFIG.cooldownSeconds);

        PatAPoke.CONFIG.animationLengthSeconds = Math.max(animationLength, 0.25D);
        PatAPoke.CONFIG.friendshipGain = Math.max(friendshipGain, 0);
        PatAPoke.CONFIG.cooldownSeconds = Math.max(cooldownSeconds, 0);
        PatAPoke.CONFIG.save(PatAPoke.LOGGER);
    }

    private static double parseDouble(String text, double fallback) {
        try {
            return Double.parseDouble(text);
        } catch (NumberFormatException ignored) {
            return fallback;
        }
    }

    private static int parseInt(String text, int fallback) {
        try {
            return Integer.parseInt(text);
        } catch (NumberFormatException ignored) {
            return fallback;
        }
    }

    private void drawOverlayText(GuiGraphics guiGraphics) {
        if (this.font == null || animationLengthField == null || friendshipGainField == null || cooldownField == null) {
            return;
        }
        int centerX = this.width / 2;
        int titleY = this.panelTop + 12;
        int labelColor = 0xFFFFFF;

        guiGraphics.drawCenteredString(this.font, this.title, centerX, titleY, labelColor);

        guiGraphics.drawString(this.font, Component.translatable("config.patapoke.animation_length"), animationLengthField.getX(), animationLengthField.getY() - 12, labelColor, false);
        guiGraphics.drawString(this.font, Component.translatable("config.patapoke.friendship_gain"), friendshipGainField.getX(), friendshipGainField.getY() - 12, labelColor, false);
        guiGraphics.drawString(this.font, Component.translatable("config.patapoke.cooldown"), cooldownField.getX(), cooldownField.getY() - 12, labelColor, false);
    }
}
