package com.patapoke.config;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.nio.file.Files;
import java.nio.file.Path;
import net.fabricmc.loader.api.FabricLoader;
import org.slf4j.Logger;

public final class PatAPokeConfig {
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final String FILE_NAME = "patapoke.json";
    private static Path configPath;

    public static final double DEFAULT_ANIMATION_LENGTH_SECONDS = 1.15D;
    public static final int DEFAULT_FRIENDSHIP_GAIN = 10;
    public static final int DEFAULT_COOLDOWN_SECONDS = 300;

    public double animationLengthSeconds = DEFAULT_ANIMATION_LENGTH_SECONDS;
    public int friendshipGain = DEFAULT_FRIENDSHIP_GAIN;
    public int cooldownSeconds = DEFAULT_COOLDOWN_SECONDS;

    public static PatAPokeConfig load(Logger logger) {
        Path path = FabricLoader.getInstance().getConfigDir().resolve(FILE_NAME);
        configPath = path;
        PatAPokeConfig defaults = new PatAPokeConfig();
        if (Files.exists(path)) {
            try (Reader reader = Files.newBufferedReader(path)) {
                PatAPokeConfig loaded = GSON.fromJson(reader, PatAPokeConfig.class);
                if (loaded != null) {
                    return loaded.normalized(path, logger);
                }
            } catch (IOException e) {
                logger.warn("Failed to read {}: {}", FILE_NAME, e.getMessage());
            }
        }
        return defaults.persist(path, logger);
    }

    private PatAPokeConfig normalized(Path path, Logger logger) {
        boolean updated = false;
        if (animationLengthSeconds <= 0.0D) {
            animationLengthSeconds = DEFAULT_ANIMATION_LENGTH_SECONDS;
            updated = true;
        }
        if (friendshipGain < 0) {
            friendshipGain = DEFAULT_FRIENDSHIP_GAIN;
            updated = true;
        }
        if (cooldownSeconds < 0) {
            cooldownSeconds = DEFAULT_COOLDOWN_SECONDS;
            updated = true;
        }
        if (updated) {
            persist(path, logger);
        }
        return this;
    }

    private PatAPokeConfig persist(Path path, Logger logger) {
        try {
            Files.createDirectories(path.getParent());
            try (Writer writer = Files.newBufferedWriter(path)) {
                GSON.toJson(this, writer);
            }
        } catch (IOException e) {
            logger.warn("Failed to save {}: {}", FILE_NAME, e.getMessage());
        }
        return this;
    }

    public float animationLengthSeconds() {
        return (float) Math.max(animationLengthSeconds, 0.25D);
    }

    public int friendshipGain() {
        return Math.max(friendshipGain, 0);
    }

    public long cooldownTicks() {
        return Math.max(cooldownSeconds, 0) * 20L;
    }

    public void save(Logger logger) {
        if (configPath == null) {
            logger.warn("Config path not initialized; skipping save.");
            return;
        }
        persist(configPath, logger);
    }
}
