package com.patapoke.petting;

import static com.cobblemon.mod.common.api.scheduling.SchedulingFunctionsKt.afterOnServer;

import com.cobblemon.mod.common.client.render.models.blockbench.PosableState;
import com.cobblemon.mod.common.entity.pokemon.PokemonEntity;
import com.cobblemon.mod.common.net.messages.client.animation.PlayPosableAnimationPacket;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import kotlin.Unit;
import net.minecraft.commands.arguments.EntityAnchorArgument;
import net.minecraft.core.particles.ParticleTypes;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.phys.Vec3;
import net.minecraft.network.chat.Component;

public final class PettingHandler {
    private static final double APPROACH_DISTANCE_SQ = 4.0D;
    private static final int MAX_APPROACH_ATTEMPTS = 40;
    private static final double APPROACH_SPEED = 1.15D;
    private static final int HEART_PARTICLES = 8;
    private static final int MID_HEART_PARTICLES = 4;
    private static final double HEAD_LOOK_OFFSET = 0.15D;
    private static final double ARM_WAVE_AMPLITUDE = 0.12D;

    private PettingHandler() {}

    public static void startPetting(ServerPlayer player, UUID pokemonId) {
        Objects.requireNonNull(player, "player");
        ServerLevel level = player.serverLevel();
        Entity entity = level.getEntity(pokemonId);
        if (!(entity instanceof PokemonEntity pokemonEntity)) {
            return;
        }

        if (!pokemonEntity.isAlive() || pokemonEntity.isBattleClone() || pokemonEntity.getPokemon().isFainted()) {
            player.displayClientMessage(Component.translatable("message.patapoke.petting.unavailable"), true);
            return;
        }

        if (!pokemonEntity.getPokemon().belongsTo(player)) {
            player.displayClientMessage(Component.translatable("message.patapoke.petting.not_owner"), true);
            return;
        }

        if (pokemonEntity.isBusy() || pokemonEntity.isBattling()) {
            player.displayClientMessage(Component.translatable("message.patapoke.petting.busy"), true);
            return;
        }

        Object lock = new Object();
        pokemonEntity.getBusyLocks().add(lock);
        guidePokemonTowardPlayer(pokemonEntity, player);
        waitForArrival(player, pokemonEntity, lock, player.isShiftKeyDown(), 0);
    }

    private static void waitForArrival(ServerPlayer player, PokemonEntity pokemon, Object lock, boolean wasSneaking, int attempts) {
        if (!isValid(player, pokemon)) {
            releaseLock(pokemon, lock);
            return;
        }

        double distance = pokemon.distanceToSqr(player);
        if (distance > APPROACH_DISTANCE_SQ && attempts < MAX_APPROACH_ATTEMPTS) {
            guidePokemonTowardPlayer(pokemon, player);
            afterOnServer(0.1F, () -> {
                waitForArrival(player, pokemon, lock, wasSneaking, attempts + 1);
                return Unit.INSTANCE;
            });
            return;
        }

        if (distance > APPROACH_DISTANCE_SQ) {
            releaseLock(pokemon, lock);
            return;
        }

        beginPettingAnimation(player, pokemon, lock, wasSneaking);
    }

    private static void beginPettingAnimation(ServerPlayer player, PokemonEntity pokemon, Object lock, boolean wasSneaking) {
        boolean shouldCrouch = pokemon.getBbHeight() < 1.2F;
        if (shouldCrouch && !wasSneaking) {
            player.setShiftKeyDown(true);
        }

        facePokemon(player, pokemon, HEAD_LOOK_OFFSET);
        pokemon.getNavigation().stop();
        pokemon.getLookControl().setLookAt(player, 45.0F, 45.0F);
        playPetAnimation(pokemon);
        pokemon.cry();
        player.swing(InteractionHand.MAIN_HAND, true);

        afterOnServer(0.25F, () -> {
            if (!isValid(player, pokemon)) {
                if (!wasSneaking && shouldCrouch) {
                    player.setShiftKeyDown(false);
                }
                releaseLock(pokemon, lock);
                return Unit.INSTANCE;
            }
            facePokemon(player, pokemon, HEAD_LOOK_OFFSET + ARM_WAVE_AMPLITUDE);
            player.swing(InteractionHand.OFF_HAND, true);
            spawnHeartParticles(player.serverLevel(), pokemon, MID_HEART_PARTICLES, 0.01D);
            return Unit.INSTANCE;
        });

        afterOnServer(0.55F, () -> {
            if (!isValid(player, pokemon)) {
                if (!wasSneaking && shouldCrouch) {
                    player.setShiftKeyDown(false);
                }
                releaseLock(pokemon, lock);
                return Unit.INSTANCE;
            }
            facePokemon(player, pokemon, HEAD_LOOK_OFFSET - ARM_WAVE_AMPLITUDE);
            player.swing(InteractionHand.MAIN_HAND, true);
            return Unit.INSTANCE;
        });

        afterOnServer(0.85F, () -> {
            if (!isValid(player, pokemon)) {
                if (!wasSneaking && shouldCrouch) {
                    player.setShiftKeyDown(false);
                }
                releaseLock(pokemon, lock);
                return Unit.INSTANCE;
            }
            facePokemon(player, pokemon, HEAD_LOOK_OFFSET + ARM_WAVE_AMPLITUDE * 0.5D);
            player.swing(InteractionHand.OFF_HAND, true);
            return Unit.INSTANCE;
        });

        afterOnServer(1.15F, () -> {
            if (!isValid(player, pokemon)) {
                releaseLock(pokemon, lock);
                return Unit.INSTANCE;
            }
            facePokemon(player, pokemon, HEAD_LOOK_OFFSET);
            finishPetting(player, pokemon, lock, wasSneaking, shouldCrouch);
            return Unit.INSTANCE;
        });
    }

    private static void finishPetting(ServerPlayer player, PokemonEntity pokemon, Object lock, boolean wasSneaking, boolean shouldCrouch) {
        if (!isValid(player, pokemon)) {
            releaseLock(pokemon, lock);
            return;
        }

        if (!wasSneaking && shouldCrouch) {
            player.setShiftKeyDown(false);
        }

        ServerLevel level = player.serverLevel();
        spawnHeartParticles(level, pokemon, HEART_PARTICLES, 0.02D);
        pokemon.getPokemon().incrementFriendship(2, true);

        releaseLock(pokemon, lock);
    }

    private static void guidePokemonTowardPlayer(PokemonEntity pokemon, ServerPlayer player) {
        Vec3 target = player.position();
        pokemon.getNavigation().moveTo(target.x, target.y, target.z, APPROACH_SPEED);
        pokemon.getLookControl().setLookAt(player, 45.0F, 45.0F);
    }

    private static void facePokemon(ServerPlayer player, PokemonEntity pokemon) {
        facePokemon(player, pokemon, 0.0D);
    }

    private static void facePokemon(ServerPlayer player, PokemonEntity pokemon, double verticalOffset) {
        Vec3 target = getPokemonHeadPosition(pokemon).add(0.0D, verticalOffset, 0.0D);
        player.lookAt(EntityAnchorArgument.Anchor.EYES, target);
    }

    private static boolean isValid(ServerPlayer player, PokemonEntity pokemon) {
        return pokemon.isAlive() && player.serverLevel() == pokemon.level();
    }

    private static void playPetAnimation(PokemonEntity pokemon) {
        String speciesName = pokemon.getPokemon().getSpecies().getName();
        if (speciesName == null || speciesName.isBlank()) {
            return;
        }

        LinkedHashSet<String> animations = new LinkedHashSet<>(
            List.of(
                "animation." + speciesName + ".happy",
                "animation." + speciesName + ".interaction",
                "animation." + speciesName + ".ground_idle"
            )
        );

        Object delegate = pokemon.getDelegate();
        if (delegate instanceof PosableState posableState) {
            posableState.addFirstAnimation(animations);
            return;
        }

        if (pokemon.level() instanceof ServerLevel serverLevel) {
            PlayPosableAnimationPacket packet = new PlayPosableAnimationPacket(pokemon.getId(), animations, Collections.emptyList());
            for (ServerPlayer nearbyPlayer : serverLevel.players()) {
                if (nearbyPlayer.distanceTo(pokemon) < 128.0F) {
                    packet.sendToPlayer(nearbyPlayer);
                }
            }
        }
    }

    private static void spawnHeartParticles(ServerLevel level, PokemonEntity pokemon, int count, double speed) {
        Vec3 pos = getPokemonHeadPosition(pokemon).add(0.0D, 0.15D, 0.0D);
        level.sendParticles(ParticleTypes.HEART, pos.x, pos.y, pos.z, count, 0.35D, 0.25D, 0.35D, speed);
    }

    private static void releaseLock(PokemonEntity pokemon, Object lock) {
        pokemon.getBusyLocks().remove(lock);
    }

    private static Vec3 getPokemonHeadPosition(PokemonEntity pokemon) {
        return new Vec3(pokemon.getX(), pokemon.getEyeY(), pokemon.getZ());
    }
}
