package com.hexagram2021.chromosomelib.common.command;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.hexagram2021.chromosomelib.ChromosomeLib;
import com.hexagram2021.chromosomelib.common.chromosome.Chromosome;
import com.hexagram2021.chromosomelib.common.chromosome.ChromosomeInstance;
import com.hexagram2021.chromosomelib.common.chromosome.ChromosomeType;
import com.hexagram2021.chromosomelib.common.entity.IChromosomeCarrier;
import com.hexagram2021.chromosomelib.common.gene.Gene;
import com.hexagram2021.chromosomelib.common.gene_locus.GeneLocusInstance;
import com.hexagram2021.chromosomelib.registry.AbstractRegisterEntry;
import com.hexagram2021.chromosomelib.registry.CLRegistries;
import com.mojang.brigadier.Command;
import com.mojang.brigadier.arguments.IntegerArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.brigadier.exceptions.Dynamic2CommandExceptionType;
import com.mojang.brigadier.suggestion.SuggestionProvider;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import org.jetbrains.annotations.ApiStatus;

import javax.annotation.Nullable;
import net.minecraft.class_1297;
import net.minecraft.class_2168;
import net.minecraft.class_2170;
import net.minecraft.class_2172;
import net.minecraft.class_2186;
import net.minecraft.class_2232;
import net.minecraft.class_2321;
import net.minecraft.class_2378;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3222;
import net.minecraft.class_5321;
import net.minecraft.class_6880;
import net.minecraft.class_7923;
import java.util.Comparator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.ToIntFunction;

/**
 * Commands for showing chromosome information of entities.
 */
@SuppressWarnings("unchecked")
@ApiStatus.Internal
public final class ChromosomeLibCommand {
	private static final String ENTITY_ARGUMENT = "entity";
	private static final String CHROMOSOME_INDEX_ARGUMENT = "chromosome_index";
	private static final String GENE_LOCUS_INDEX_ARGUMENT = "gene_locus_index";
	private static final String GENE_1_ARGUMENT = "gene_1";
	private static final String GENE_2_ARGUMENT = "gene_2";

	/**
	 * Register commands.
	 */
	public static LiteralArgumentBuilder<class_2168> register() {
		return class_2170.method_9247("chromosomelib").requires(stack -> stack.method_9259(2)).then(
				class_2170.method_9247("show")
						.executes(ctx -> show(ctx.getSource().method_44023(), ctx.getSource().method_9228())).then(
								class_2170.method_9244(ENTITY_ARGUMENT, class_2186.method_9309())
										.executes(ctx -> show(ctx.getSource().method_44023(), class_2186.method_9313(ctx, ENTITY_ARGUMENT)))
						)
		).then(
				class_2170.method_9247("reassign")
						.executes(ctx -> reassign(ctx.getSource().method_9228())).then(
								class_2170.method_9244(ENTITY_ARGUMENT, class_2186.method_9309())
										.executes(ctx -> reassign(class_2186.method_9313(ctx, ENTITY_ARGUMENT)))
						)
		).then(
				class_2170.method_9247("set").then(
						class_2170.method_9244(ENTITY_ARGUMENT, class_2186.method_9309()).then(
								class_2170.method_9244(CHROMOSOME_INDEX_ARGUMENT, IntegerArgumentType.integer(0)).then(
										class_2170.method_9244(GENE_LOCUS_INDEX_ARGUMENT, IntegerArgumentType.integer(0)).then(
												class_2170.method_9244(GENE_1_ARGUMENT, class_2232.method_9441()).suggests(GENE_SUGGESTIONS).executes(ctx -> set(
														class_2186.method_9313(ctx, ENTITY_ARGUMENT),
														IntegerArgumentType.getInteger(ctx, CHROMOSOME_INDEX_ARGUMENT),
														IntegerArgumentType.getInteger(ctx, GENE_LOCUS_INDEX_ARGUMENT),
														class_2232.method_9443(ctx, GENE_1_ARGUMENT),
														class_2232.method_9443(ctx, GENE_1_ARGUMENT)
												)).then(
														class_2170.method_9244(GENE_2_ARGUMENT, class_2232.method_9441()).suggests(GENE_SUGGESTIONS)
																.executes(ctx -> set(
																		class_2186.method_9313(ctx, ENTITY_ARGUMENT),
																		IntegerArgumentType.getInteger(ctx, CHROMOSOME_INDEX_ARGUMENT),
																		IntegerArgumentType.getInteger(ctx, GENE_LOCUS_INDEX_ARGUMENT),
																		class_2232.method_9443(ctx, GENE_1_ARGUMENT),
																		class_2232.method_9443(ctx, GENE_2_ARGUMENT)
																))
												)
										)

								)
						)
				)
		);
	}

	private static final class_2378<Chromosome> chromosomeRegistry = (class_2378<Chromosome>) Objects.requireNonNull(class_7923.field_41167.method_10223(CLRegistries.CHROMOSOMES.method_29177()));

	private static final class_2378<Gene> geneRegistry = (class_2378<Gene>) Objects.requireNonNull(class_7923.field_41167.method_10223(CLRegistries.GENES.method_29177()));

	private static final SuggestionProvider<class_2168> GENE_SUGGESTIONS = class_2321.method_10022(
			new class_2960(ChromosomeLib.MODID, "genes"),
			(context, builder) -> class_2172.method_9257(geneRegistry.method_40270().flatMap(holder -> holder.method_40230().stream().map(class_5321::method_29177)), builder)
	);

	private static String chromosome2Loc(class_6880<Chromosome> holder) {
		return holder.method_40229().map(key -> key.method_29177().toString(), value -> {
			class_2960 key = chromosomeRegistry.method_10221(value);
			if(key == null) {
				return "";
			}
			return key.toString();
		});
	}

	private static String gene2Loc(class_6880<Gene> holder) {
		return holder.method_40229().map(key -> key.method_29177().toString(), value -> {
			class_2960 key = geneRegistry.method_10221(value);
			if(key == null) {
				return "";
			}
			return key.toString();
		});
	}

	private static int show(@Nullable class_3222 player, @Nullable class_1297 entity) {
		Map<class_6880<Chromosome>, Object2IntMap<class_6880<Gene>>> toShow = AbstractRegisterEntry.newHolderTreeMap();
		if(player != null && entity instanceof IChromosomeCarrier carrier) {
			ToIntFunction<class_6880<Gene>> activeGenes = carrier.chromosomelib$getActiveGenes();
			carrier.chromosomelib$getChromosomes().forEach(chromosomeInstance -> {
				Object2IntMap<class_6880<Gene>> chromosomeGenes = toShow.computeIfAbsent(chromosomeInstance.chromosome(), ignored -> AbstractRegisterEntry.newHolderObject2IntTreeMap());
				chromosomeInstance.geneLocusInstances().forEach((index, geneLocusInstance) -> chromosomeGenes.put(geneLocusInstance.gene(), activeGenes.applyAsInt(geneLocusInstance.gene())));
			});
			StringBuilder builder = new StringBuilder();
			toShow.keySet().stream().sorted(Comparator.comparing(ChromosomeLibCommand::chromosome2Loc)).forEach(chromosome -> {
				Object2IntMap<class_6880<Gene>> chromosomeGenes = toShow.get(chromosome);
				if(!chromosomeGenes.isEmpty()) {
					builder.append(chromosome2Loc(chromosome)).append(":\n");
					chromosomeGenes.keySet().forEach(gene -> builder.append("  - ")
							.append(gene2Loc(gene))
							.append(": ")
							.append(activeGenes.applyAsInt(gene))
							.append('\n'));
				}
			});

			player.method_43496(class_2561.method_43470(builder.toString()));
		}
		return Command.SINGLE_SUCCESS;
	}

	private static int reassign(@Nullable class_1297 entity) {
		if(entity instanceof IChromosomeCarrier carrier) {
			carrier.chromosomelib$resetTraits();
			carrier.chromosomelib$assignTraits();
		}
		return Command.SINGLE_SUCCESS;
	}

	public static final Dynamic2CommandExceptionType GENE_LOCUS_INDEX_MISMATCHED = new Dynamic2CommandExceptionType(
			(expected, found) -> class_2561.method_43470("Gene Locus index mismatched. Expected %d, found %d.".formatted((int)expected, (int)found))
	);

	private static int set(@Nullable class_1297 entity, int chromosomeIndex, int geneLocusIndex, class_2960 geneId1, class_2960 geneId2) throws CommandSyntaxException {
		class_6880<Gene> gene1 = geneRegistry.method_40290(class_5321.method_29179(CLRegistries.GENES, geneId1));
		class_6880<Gene> gene2 = geneRegistry.method_40290(class_5321.method_29179(CLRegistries.GENES, geneId2));
		if(entity instanceof IChromosomeCarrier carrier) {
			AtomicInteger cnt = new AtomicInteger(0);
			ImmutableSet.Builder<ChromosomeInstance> builder = ImmutableSet.builder();
			for(ChromosomeInstance chromosomeInstance: carrier.chromosomelib$getChromosomes()) {
				int index = Chromosome.index(chromosomeInstance.chromosome(), entity.method_5864());
				if(index == chromosomeIndex) {
					Set<GeneLocusInstance> geneLocusInstances = Sets.newIdentityHashSet();
					Int2ObjectMap<GeneLocusInstance> original = chromosomeInstance.geneLocusInstances();
					for(int locusIndex: original.keySet()) {
						if(locusIndex == geneLocusIndex) {
							int currentCnt = cnt.get();
							if((currentCnt & 1) == 0) {
								checkGeneLocusIndex(gene1, chromosomeInstance.type(), geneLocusIndex);
								geneLocusInstances.add(new GeneLocusInstance(gene1));
							} else {
								checkGeneLocusIndex(gene2, chromosomeInstance.type(), geneLocusIndex);
								geneLocusInstances.add(new GeneLocusInstance(gene2));
							}
							cnt.set(currentCnt + 1);
						} else {
							geneLocusInstances.add(original.get(locusIndex));
						}
					}
					builder.add(ChromosomeInstance.of(chromosomeInstance.chromosome(), chromosomeInstance.type(), geneLocusInstances));
				} else {
					builder.add(chromosomeInstance);
				}
			}
			carrier.chromosomelib$setChromosomes(builder.build());
		}
		return Command.SINGLE_SUCCESS;
	}

	private static void checkGeneLocusIndex(class_6880<Gene> gene, ChromosomeType type, int index) throws CommandSyntaxException {
		int found = Objects.requireNonNull(gene.comp_349().geneLocus).comp_349().index(type);
		if(found != index) {
			throw GENE_LOCUS_INDEX_MISMATCHED.create(index, found);
		}
	}

	private ChromosomeLibCommand() {
	}
}
