package com.hexagram2021.chromosomelib.common.util;

import com.google.common.collect.*;
import com.hexagram2021.chromosomelib.common.chromosome.Chromosome;
import com.hexagram2021.chromosomelib.common.chromosome.ChromosomeInstance;
import com.hexagram2021.chromosomelib.common.gene_locus.GeneLocus;
import com.hexagram2021.chromosomelib.common.gene_locus.GeneLocusInstance;
import com.hexagram2021.chromosomelib.common.util.exception.UnpairedChromosomeException;
import com.hexagram2021.chromosomelib.platform.Services;
import com.hexagram2021.chromosomelib.registry.AbstractRegisterEntry;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import java.util.Collection;
import java.util.Map;
import net.minecraft.class_1299;
import net.minecraft.class_156;
import net.minecraft.class_5819;
import net.minecraft.class_6880;

/**
 * Utility class for entity breeding.
 */
public final class Breeders {
	/**
	 * Cross over.
	 * @param chromosomes	chromosomes to cross over.
	 * @param random		random source
	 */
	public static void crossOver(Collection<ChromosomeInstance> chromosomes, class_5819 random) {
		Map<class_6880<Chromosome>, ChromosomeInstance> temporary = Maps.newIdentityHashMap();
		chromosomes.forEach(chromosomeInstance -> {
			class_6880<Chromosome> chromosome = chromosomeInstance.chromosome();
			ChromosomeInstance oldInstance = temporary.put(chromosome, chromosomeInstance);
			boolean swap = false;
			if(oldInstance != null) {
				// Start cross over for two chromosomes.
				int offset = chromosome.comp_349().homoSegmentStartDiff() * (chromosomeInstance.type().ordinal() - oldInstance.type().ordinal());
				for(Int2ObjectMap.Entry<GeneLocusInstance> entry : chromosomeInstance.geneLocusInstances().int2ObjectEntrySet()) {
					GeneLocusInstance current = entry.getValue();
					int otherIndex = entry.getIntKey() + offset;
					GeneLocusInstance other = oldInstance.geneLocusInstances().get(otherIndex);
					class_6880<GeneLocus> geneLocus = current.geneLocus();
					if(other != null && other.geneLocus() == geneLocus) {
						if(random.method_43058() <= geneLocus.comp_349().possibilityOfCrossingOver) {
							swap = !swap;
						}
						if(swap) {
							entry.setValue(other);
							oldInstance.geneLocusInstances().put(otherIndex, current);
						}
					} else {
						swap = false;
					}
				}
			}
		});
	}

	/**
	 * Mutate.
	 * @param chromosomes	Chromosomes to mutate.
	 * @param random		Random source.
	 */
	public static void mutate(Collection<ChromosomeInstance> chromosomes, class_5819 random) {
		chromosomes.forEach(chromosomeInstance -> {
			for(Int2ObjectMap.Entry<GeneLocusInstance> entry : chromosomeInstance.geneLocusInstances().int2ObjectEntrySet()) {
				GeneLocusInstance geneLocusInstance = entry.getValue();
				if(random.method_43058() <= geneLocusInstance.geneLocus().comp_349().possibilityOfMutation) {
					entry.setValue(geneLocusInstance.mutate(random));
				}
			}
		});
	}

	/**
	 * Breed.
	 * @param parentAType	type of parent A
	 * @param parentBType	type of parent B
	 * @param parentA		chromosome instances of parent A
	 * @param parentB		chromosome instances of parent B
	 * @param random		random source
	 * @return the chromosome instances of the child
	 */
	public static Collection<ChromosomeInstance> breed(class_1299<?> parentAType, class_1299<?> parentBType,
													   Collection<ChromosomeInstance> parentA, Collection<ChromosomeInstance> parentB,
													   class_5819 random) {
		ImmutableSet.Builder<ChromosomeInstance> builder = ImmutableSet.builder();
		class_6880<Chromosome> failed = null;
		Multimap<class_6880<Chromosome>, ChromosomeInstance> temporary = Multimaps.newListMultimap(AbstractRegisterEntry.newHolderTreeMap(), Lists::newArrayList);
		parentA.forEach(chromosomeInstance -> temporary.put(chromosomeInstance.chromosome(), chromosomeInstance));
		for(class_6880<Chromosome> chromosome : temporary.keySet()) {
			Collection<ChromosomeInstance> chromosomeInstances = temporary.get(chromosome);
			if(chromosomeInstances.size() % 2 == 1) {
				failed = chromosome;
				break;
			}
			ObjectArrayList<ChromosomeInstance> shuffled = new ObjectArrayList<>(chromosomeInstances);
			class_156.method_43028(shuffled, random);
			for(int i = 0; i < shuffled.size(); i += 2) {
				builder.add(shuffled.get(i).copy());
			}
		}
		if(failed != null) {
			if(!Services.PLATFORM.solveUnpairedChromosomesToBreed(parentAType, parentA, random, builder)) {
				throw new UnpairedChromosomeException(parentAType, "breed", failed, temporary.get(failed));
			}
			failed = null;
		}
		temporary.clear();
		parentB.forEach(chromosomeInstance -> temporary.put(chromosomeInstance.chromosome(), chromosomeInstance));
		for(class_6880<Chromosome> chromosome : temporary.keySet()) {
			Collection<ChromosomeInstance> chromosomeInstances = temporary.get(chromosome);
			if(chromosomeInstances.size() % 2 == 1) {
				failed = chromosome;
				break;
			}
			ObjectArrayList<ChromosomeInstance> shuffled = new ObjectArrayList<>(chromosomeInstances);
			class_156.method_43028(shuffled, random);
			for(int i = 0; i < shuffled.size(); i += 2) {
				builder.add(shuffled.get(i).copy());
			}
		}
		if(failed != null && !Services.PLATFORM.solveUnpairedChromosomesToBreed(parentBType, parentB, random, builder)) {
			throw new UnpairedChromosomeException(parentBType, "breed", failed, temporary.get(failed));
		}

		return builder.build();
	}

	private Breeders() {
	}
}
