package com.hexagram2021.chromosomelib.registry;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.hexagram2021.chromosomelib.common.gene.Gene;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import javax.annotation.Nullable;
import net.minecraft.class_1959;
import net.minecraft.class_6880;
import net.minecraft.class_6885;
import java.util.Iterator;
import java.util.Map;
import java.util.stream.Stream;

/**
 * Biome-specific weighted gene list.
 */
public class BiomeSpecificWeightedGeneList implements IWeightedGeneList {
	/**
	 * Common gene list - all biomes share the same gene list.
	 */
	protected final IWeightedGeneList commonList;
	/**
	 * Biome-specific gene lists - each biome has its own gene list.
	 */
	protected final Map<class_6880<class_1959>, IWeightedGeneList> biomeSpecificLists;

	/**
	 * Create a BiomeSpecificWeightedGeneList.
	 * @param commonList			the common gene list
	 * @param biomeSpecificLists	the biome-specific gene lists
	 */
	protected BiomeSpecificWeightedGeneList(IWeightedGeneList commonList, Map<class_6880<class_1959>, IWeightedGeneList> biomeSpecificLists) {
		this.commonList = commonList;
		this.biomeSpecificLists = biomeSpecificLists;
	}

	/**
	 * Get a random gene from this BiomeSpecificWeightedGeneList.
	 * @param context	the context, where biome is passed.
	 * @return a random gene
	 */
	@Override
	public class_6880<Gene> getRandomGene(Context context) {
		IWeightedGeneList biomeSpecificList = this.get(context.biome());
		int commonTotalWeight = this.commonList.totalWeight(context);
		int index = context.random().method_43048(commonTotalWeight + biomeSpecificList.totalWeight(context));
		if(index < commonTotalWeight) {
			return this.commonList.getRandomGene(context);
		}
		return biomeSpecificList.getRandomGene(context);
	}

	/**
	 * Get the total weight of this list.
	 * @param context	the context
	 * @return the total weight
	 */
	@Override
	public int totalWeight(Context context) {
		return this.commonList.totalWeight(context) + this.get(context.biome()).totalWeight(context);
	}

	/**
	 * Get all possible genes in this list.
	 * @return all possible genes
	 */
	@Override
	public Stream<class_6880<Gene>> allGenes() {
		return Stream.concat(this.commonList.allGenes(), this.biomeSpecificLists.values().stream().flatMap(IWeightedGeneList::allGenes)).distinct();
	}

	private IWeightedGeneList get(@Nullable class_6880<class_1959> biome) {
		return biome == null ? IWeightedGeneList.EMPTY : this.biomeSpecificLists.getOrDefault(biome, IWeightedGeneList.EMPTY);
	}

	/**
	 * Create a builder for BiomeSpecificWeightedGeneList. SimpleWeightedGeneList is applied to all biomes.
	 * @return a builder
	 */
	public static com.hexagram2021.chromosomelib.registry.BiomeSpecificWeightedGeneList.Builder simpleBuilder() {
		return new com.hexagram2021.chromosomelib.registry.BiomeSpecificWeightedGeneList.Builder(SimpleWeightedGeneList::new);
	}

	/**
	 * Create a builder for BiomeSpecificWeightedGeneList. StableWeightedGeneList is applied to all biomes.
	 * @param possibilityOfStable	Possibility for stable weighted gene list
	 * @see StableWeightedGeneList#getRandomGene
	 * @return a builder
	 */
	public static com.hexagram2021.chromosomelib.registry.BiomeSpecificWeightedGeneList.Builder stableBuilder(double possibilityOfStable) {
		return new com.hexagram2021.chromosomelib.registry.BiomeSpecificWeightedGeneList.Builder(entries -> new StableWeightedGeneList(possibilityOfStable, entries));
	}

	/**
	 * Builder for BiomeSpecificWeightedGeneList.
	 */
	public static class Builder extends IWeightedGeneList.Builder {
		/**
		 * Biome-specific gene lists.
		 */
		protected final Map<class_6880<class_1959>, ImmutableList.Builder<Entry>> biomeSpecificShadowed = new Object2ObjectOpenHashMap<>();
		/**
		 * Factory for creating a weighted gene list.
		 */
		private final WeightedGeneListFactory factory;

		/**
		 * @param factory	factory for creating a weighted gene list
		 */
		public Builder(WeightedGeneListFactory factory) {
			super();
			this.factory = factory;
		}

		/**
		 * Add a gene to a biome to the builder.
		 * @param biome		the biome
		 * @param gene		the gene
		 * @param weight	the weight
		 * @return this builder
		 */
		public com.hexagram2021.chromosomelib.registry.BiomeSpecificWeightedGeneList.Builder add(class_6880<class_1959> biome, class_6880<Gene> gene, int weight) {
			this.biomeSpecificShadowed.computeIfAbsent(biome, ignored -> ImmutableList.builder()).add(Entry.of(gene, weight));
			return this;
		}
		/**
		 * Add some genes to a biome to the builder.
		 * @param biome		the biome
		 * @param elements	elements
		 * @return this builder
		 */
		public com.hexagram2021.chromosomelib.registry.BiomeSpecificWeightedGeneList.Builder add(class_6880<class_1959> biome, Entry... elements) {
			this.biomeSpecificShadowed.computeIfAbsent(biome, ignored -> ImmutableList.builder()).add(elements);
			return this;
		}
		/**
		 * Add all genes from a collection to a biome to the builder.
		 * @param biome		the biome
		 * @param elements	a collection
		 * @return this builder
		 */
		public com.hexagram2021.chromosomelib.registry.BiomeSpecificWeightedGeneList.Builder addAll(class_6880<class_1959> biome, Iterable<Entry> elements) {
			this.biomeSpecificShadowed.computeIfAbsent(biome, ignored -> ImmutableList.builder()).addAll(elements);
			return this;
		}
		/**
		 * Add all genes from a collection to a biome to the builder.
		 * @param biome		the biome
		 * @param elements	a collection
		 * @return this builder
		 */
		public com.hexagram2021.chromosomelib.registry.BiomeSpecificWeightedGeneList.Builder addAll(class_6880<class_1959> biome, Iterator<Entry> elements) {
			this.biomeSpecificShadowed.computeIfAbsent(biome, ignored -> ImmutableList.builder()).addAll(elements);
			return this;
		}

		/**
		 * Add a gene to some biomes to the builder.
		 * @param biomes	a set of biomes
		 * @param gene		the gene
		 * @param weight	the weight
		 * @return this builder
		 */
		public com.hexagram2021.chromosomelib.registry.BiomeSpecificWeightedGeneList.Builder add(class_6885<class_1959> biomes, class_6880<Gene> gene, int weight) {
			biomes.forEach(biome -> this.biomeSpecificShadowed.computeIfAbsent(biome, ignored -> ImmutableList.builder()).add(Entry.of(gene, weight)));
			return this;
		}
		/**
		 * Add some genes to some biomes to the builder.
		 * @param biomes	a set of biomes
		 * @param elements	elements
		 * @return this builder
		 */
		public com.hexagram2021.chromosomelib.registry.BiomeSpecificWeightedGeneList.Builder add(class_6885<class_1959> biomes, Entry... elements) {
			biomes.forEach(biome -> this.biomeSpecificShadowed.computeIfAbsent(biome, ignored -> ImmutableList.builder()).add(elements));
			return this;
		}
		/**
		 * Add all genes from a collection to some biomes to the builder.
		 * @param biomes	a set of biomes
		 * @param elements	a collection
		 * @return this builder
		 */
		public com.hexagram2021.chromosomelib.registry.BiomeSpecificWeightedGeneList.Builder addAll(class_6885<class_1959> biomes, Iterable<Entry> elements) {
			biomes.forEach(biome -> this.biomeSpecificShadowed.computeIfAbsent(biome, ignored -> ImmutableList.builder()).addAll(elements));
			return this;
		}
		/**
		 * Add all genes from a collection to some biomes to the builder.
		 * @param biomes	a set of biomes
		 * @param elements	a collection
		 * @return this builder
		 */
		public com.hexagram2021.chromosomelib.registry.BiomeSpecificWeightedGeneList.Builder addAll(class_6885<class_1959> biomes, Iterator<Entry> elements) {
			biomes.forEach(biome -> this.biomeSpecificShadowed.computeIfAbsent(biome, ignored -> ImmutableList.builder()).addAll(elements));
			return this;
		}

		/**
		 * Build the BiomeSpecificWeightedGeneList.
		 * @return the BiomeSpecificWeightedGeneList
		 */
		@Override
		BiomeSpecificWeightedGeneList build() {
			ImmutableMap.Builder<class_6880<class_1959>, IWeightedGeneList> biomeSpecificShadowedBuilder = ImmutableMap.builder();
			this.biomeSpecificShadowed.forEach((biome, builder) -> biomeSpecificShadowedBuilder.put(biome, this.factory.create(builder.build())));
			return new BiomeSpecificWeightedGeneList(this.factory.create(this.shadowed.build()), biomeSpecificShadowedBuilder.build());
		}
	}
}
