package com.hexagram2021.chromosomelib.registry;

import com.google.common.collect.*;
import com.google.common.graph.GraphBuilder;
import com.google.common.graph.ImmutableGraph;
import com.hexagram2021.chromosomelib.common.chromosome.Chromosome;
import com.hexagram2021.chromosomelib.common.chromosome.ChromosomeType;
import com.hexagram2021.chromosomelib.common.entity.type.IChromosomeLibEntityType;
import com.hexagram2021.chromosomelib.common.gene.Gene;
import com.hexagram2021.chromosomelib.common.gene_locus.GeneLocus;
import com.hexagram2021.chromosomelib.common.trait.Trait;
import com.hexagram2021.chromosomelib.common.trait.TraitType;
import com.hexagram2021.chromosomelib.common.util.CLLogger;
import com.hexagram2021.chromosomelib.common.util.exception.GeneFrequencyNotPresentException;
import com.hexagram2021.chromosomelib.common.util.exception.InvalidGeneFromGeneLocusException;
import com.hexagram2021.chromosomelib.common.util.exception.RegistryConcurrentModificationException;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectMaps;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2IntRBTreeMap;
import org.jetbrains.annotations.ApiStatus;

import java.util.*;
import java.util.stream.Collectors;
import net.minecraft.class_1299;
import net.minecraft.class_2378;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_6885;
import net.minecraft.class_7923;

/**
 * This class is used to register relations between registries.
 * Please ONLY call these APIs in main thread to avoid {@code ConcurrentModificationException}!
 */
@SuppressWarnings("UnstableApiUsage")
public final class RegistryRelations {
	private static boolean isFrozen = false;

	private static final Map<class_1299<?>, ImmutableList.Builder<class_6880<Chromosome>>> entityType2ChromosomeMap = Maps.newIdentityHashMap();
	private static final Map<class_6880<Chromosome>, ImmutableList.Builder<class_6880<GeneLocus>>> chromosome2GeneLocusMap = Maps.newIdentityHashMap();
	private static final Map<class_6880<GeneLocus>, ImmutableList.Builder<class_6880<Gene>>> geneLocus2GeneMap = Maps.newIdentityHashMap();
	private static final Map<class_1299<?>, ImmutableList.Builder<class_6880<TraitType>>> entityType2TraitTypeMap = Maps.newIdentityHashMap();
	private static final Map<class_6862<class_1299<?>>, ImmutableList.Builder<class_6880<Chromosome>>> entityTypeTag2ChromosomeMap = Maps.newHashMap();
	private static final Map<class_6862<class_1299<?>>, ImmutableList.Builder<class_6880<TraitType>>> entityTypeTag2TraitTypeMap = Maps.newHashMap();

	private static final ImmutableGraph.Builder<class_6880<Gene>> disableRelations = GraphBuilder.directed().immutable();

	private static final ImmutableMap.Builder<class_6880<GeneLocus>, IWeightedGeneList> geneFrequency = ImmutableMap.builder();

	private static final Object2IntMap<RemapKey> chromosomeIndexRemapper = new Object2IntOpenHashMap<>();
	private static final Object2IntMap<RemapTagKey> taggedChromosomeIndexRemapper = new Object2IntOpenHashMap<>();
	private static final ImmutableMap.Builder<class_6880<Chromosome>, ChromosomeType> necessaryChromosomeTypes = ImmutableMap.builder();

	private static int countEntityType2Chromosome = 0;
	private static int countChromosome2GeneLocus = 0;
	private static int countGeneLocus2Gene = 0;
	private static int countEntityType2Trait = 0;

	private RegistryRelations() {
	}

	/**
	 * Register the chromosome of an entity type.
	 * @param entityType	the entity type
	 * @param chromosome	the chromosome
	 */
	public static void registerEntityType2Chromosome(class_1299<?> entityType, class_6880<Chromosome> chromosome) {
		try {
			getBuilder(entityType2ChromosomeMap, entityType).add(chromosome);
		} catch (ConcurrentModificationException | IndexOutOfBoundsException e) {
			throw new RegistryConcurrentModificationException(RegistryRelations.class.getName(), "entityType2ChromosomeMap", "Map", e);
		}
		countEntityType2Chromosome += 1;
	}
	/**
	 * Register the chromosome of an entity type and remap the index of chromosome.
	 * @param entityType	the entity type
	 * @param chromosome	the chromosome
	 * @param remappedIndex	the remapped index of the chromosome
	 */
	public static void registerEntityType2Chromosome(class_1299<?> entityType, class_6880<Chromosome> chromosome, int remappedIndex) {
		try {
			getBuilder(entityType2ChromosomeMap, entityType).add(chromosome);
		} catch (ConcurrentModificationException | IndexOutOfBoundsException e) {
			throw new RegistryConcurrentModificationException(RegistryRelations.class.getName(), "entityType2ChromosomeMap", "Map", e);
		}
		try {
			chromosomeIndexRemapper.put(new RemapKey(entityType, chromosome), remappedIndex);
		} catch (ConcurrentModificationException | IndexOutOfBoundsException e) {
			throw new RegistryConcurrentModificationException(RegistryRelations.class.getName(), "chromosomeIndexRemapper", "Object2IntMap", e);
		}
		countEntityType2Chromosome += 1;
	}
	/**
	 * Register the chromosome of an entity type.
	 * @param chromosome	the chromosome
	 * @param geneLocus		the gene locus
	 */
	public static void registerChromosome2GeneLocus(class_6880<Chromosome> chromosome, class_6880<GeneLocus> geneLocus) {
		try {
			getBuilder(chromosome2GeneLocusMap, chromosome).add(geneLocus);
		} catch (ConcurrentModificationException | IndexOutOfBoundsException e) {
			throw new RegistryConcurrentModificationException(RegistryRelations.class.getName(), "chromosome2GeneLocusMap", "Map", e);
		}
		countChromosome2GeneLocus += 1;
	}
	/**
	 * Register the gene locus of a chromosome.
	 * @param geneLocus	the gene locus
	 * @param gene		the gene
	 */
	public static void registerGeneLocus2Gene(class_6880<GeneLocus> geneLocus, class_6880<Gene> gene) {
		try {
			getBuilder(geneLocus2GeneMap, geneLocus).add(gene);
		} catch (ConcurrentModificationException | IndexOutOfBoundsException e) {
			throw new RegistryConcurrentModificationException(RegistryRelations.class.getName(), "geneLocus2GeneMap", "Map", e);
		}
		countGeneLocus2Gene += 1;
	}
	/**
	 * Register the trait of an entity type.
	 * @param entityType	the entity type
	 * @param traitType		the trait type
	 */
	public static void registerEntityType2TraitType(class_1299<?> entityType, class_6880<TraitType> traitType) {
		try {
			getBuilder(entityType2TraitTypeMap, entityType).add(traitType);
		} catch (ConcurrentModificationException | IndexOutOfBoundsException e) {
			throw new RegistryConcurrentModificationException(RegistryRelations.class.getName(), "entityType2TraitMap", "Map", e);
		}
		countEntityType2Trait += 1;
	}

	/**
	 * Sometimes some different mobs are the same species (e.g. Villager, Wandering Trader, Pillager, etc.), so they share same chromosomes.
	 * @param entityTypeTag	the tag of entity types
	 * @param chromosome	the chromosome
	 */
	public static void registerEntityTypeTag2Chromosome(class_6862<class_1299<?>> entityTypeTag, class_6880<Chromosome> chromosome) {
		try {
			getBuilder(entityTypeTag2ChromosomeMap, entityTypeTag).add(chromosome);
		} catch (ConcurrentModificationException | IndexOutOfBoundsException e) {
			throw new RegistryConcurrentModificationException(RegistryRelations.class.getName(), "entityTypeTag2ChromosomeMap", "Map", e);
		}
	}
	/**
	 * Sometimes some different mobs are the same species (e.g. Villager, Wandering Trader, Pillager, etc.), so they share same chromosomes.
	 * <p>The index of chromosome is also remapped.
	 * @param entityTypeTag	the tag of entity types
	 * @param chromosome	the chromosome
	 * @param remappedIndex	the remapped index of the chromosome
	 */
	public static void registerEntityTypeTag2Chromosome(class_6862<class_1299<?>> entityTypeTag, class_6880<Chromosome> chromosome, int remappedIndex) {
		try {
			getBuilder(entityTypeTag2ChromosomeMap, entityTypeTag).add(chromosome);
		} catch (ConcurrentModificationException | IndexOutOfBoundsException e) {
			throw new RegistryConcurrentModificationException(RegistryRelations.class.getName(), "entityTypeTag2ChromosomeMap", "Map", e);
		}
		try {
			taggedChromosomeIndexRemapper.put(new RemapTagKey(entityTypeTag, chromosome), remappedIndex);
		} catch (ConcurrentModificationException | IndexOutOfBoundsException e) {
			throw new RegistryConcurrentModificationException(RegistryRelations.class.getName(), "taggedChromosomeIndexRemapper", "Object2IntMap", e);
		}
	}

	/**
	 * Sometimes some different mobs are the same species (e.g. Villager, Wandering Trader, Pillager, etc.), so they may share same traits.
	 * @param entityTypeTag	the tag of entity types
	 * @param traitType		the trait type
	 */
	public static void registerEntityTypeTag2TraitType(class_6862<class_1299<?>> entityTypeTag, class_6880<TraitType> traitType) {
		try {
			getBuilder(entityTypeTag2TraitTypeMap, entityTypeTag).add(traitType);
		} catch (ConcurrentModificationException | IndexOutOfBoundsException e) {
			throw new RegistryConcurrentModificationException(RegistryRelations.class.getName(), "entityTypeTag2TraitMap", "Map", e);
		}
	}

	private static <T, R> ImmutableList.Builder<class_6880<R>> getBuilder(Map<T, ImmutableList.Builder<class_6880<R>>> map, T key) {
		if(isFrozen) {
			throw new IllegalStateException("Relations registry is already frozen!");
		}
		return map.computeIfAbsent(key, k -> new ImmutableList.Builder<>());
	}

	/**
	 * Dominant genes can render recessive genes ineffective, while many genes in reality are not binary.
	 * Please use this function to register all relative explicit implicit relationships.
	 * For genes of incomplete dominance, co dominance, and mosaic dominance types, please specify rules in {@link com.hexagram2021.chromosomelib.common.trait.TraitHandler#HANDLERS} instead of this method.
	 * Two genes should be (multiple) geneLocusInstances of the same gene locus.
	 *
	 * @param dominant	the dominant gene
	 * @param recessive	the recessive gene
	 */
	public static void registerDisableRelation(class_6880<Gene> dominant, class_6880<Gene> recessive) {
		if(isFrozen) {
			throw new IllegalStateException("Relations registry is already frozen!");
		}
		if(dominant instanceof AbstractRegisterEntry<Gene> registerEntry) {
			dominant = registerEntry.asHolder();
		}
		if(recessive instanceof AbstractRegisterEntry<Gene> registerEntry) {
			recessive = registerEntry.asHolder();
		}
		disableRelations.putEdge(dominant, recessive);
	}

	/**
	 * Register the gene frequency of a gene locus.
	 * <p>Notice that holders are NOT typed {@link class_6880.class_6881}. Either {@link AbstractRegisterEntry} or {@link class_6880.class_6883} is acceptable.
	 * @param geneLocus	the gene locus
	 * @param entries	genes and their weights
	 */
	public static void registerGeneFrequency(class_6880<GeneLocus> geneLocus, IWeightedGeneList.Builder entries) {
		if(isFrozen) {
			throw new IllegalStateException("Relations registry is already frozen!");
		}
		geneFrequency.put(geneLocus, entries.build());
	}

	/**
	 * Register a necessary chromosome type of chromosome of a diploid.
	 * <p>For example, X is necessary in XX/XY system, while Z is necessary in ZZ/ZW system.
	 */
	public static void registerNecessaryChromosomeTypes(class_6880<Chromosome> chromosome, ChromosomeType chromosomeType) {
		if(isFrozen) {
			throw new IllegalStateException("Relations registry is already frozen!");
		}
		try {
			necessaryChromosomeTypes.put(chromosome, chromosomeType);
		} catch (ConcurrentModificationException | IndexOutOfBoundsException e) {
			throw new RegistryConcurrentModificationException(RegistryRelations.class.getName(), "necessaryChromosomeTypes", "Map", e);
		}
	}

	@ApiStatus.Internal
	@SuppressWarnings("unchecked")
	public static void freezeAndBuild() {
		if(isFrozen) {
			return;
		}

		class_2378<class_1299<?>> entityTypeRegistry = class_7923.field_41177;
		class_2378<Gene> geneRegistry = (class_2378<Gene>) Objects.requireNonNull(class_7923.field_41167.method_10223(CLRegistries.GENES.method_29177()));
		class_2378<GeneLocus> geneLocusRegistry = (class_2378<GeneLocus>) Objects.requireNonNull(class_7923.field_41167.method_10223(CLRegistries.GENE_LOCI.method_29177()));
		class_2378<Trait> traitRegistry = (class_2378<Trait>) Objects.requireNonNull(class_7923.field_41167.method_10223(CLRegistries.TRAITS.method_29177()));

		buildEntityType2ChromosomeRelations(entityTypeRegistry);
		buildChromosome2GeneLocusRelations();
		buildGeneLocus2GeneRelations();
		buildEntityType2TraitRelations(entityTypeRegistry);

		checkGeneFrequencyLists(geneLocusRegistry);

		Chromosome.setNecessaryChromosomeTypes(necessaryChromosomeTypes.build());

		buildTraitType2TraitsReversedRelations(traitRegistry);

		buildGeneTopologicalOrder(geneRegistry);

		isFrozen = true;
		CLLogger.info("Register successfully. Relations registry is frozen.");
	}

	private static void checkGeneFrequencyLists(class_2378<GeneLocus> geneLocusRegistry) {
		Set<class_6880<GeneLocus>> baseline = geneLocusRegistry.method_40270().collect(Collectors.toCollection(AbstractRegisterEntry::newHolderTreeSet));
		Map<class_6880<GeneLocus>, IWeightedGeneList> geneFrequencyLists = geneFrequency.build();
		geneFrequencyLists.forEach((geneLocus, list) -> {
			Set<class_6880<Gene>> set = Objects.requireNonNull(geneLocus.comp_349().genes).method_40239().collect(Collectors.toCollection(AbstractRegisterEntry::newHolderTreeSet));
			list.allGenes().forEach(gene -> {
				if(!set.remove(gene)) {
					throw new InvalidGeneFromGeneLocusException(gene, geneLocus);
				}
			});
			if(!set.isEmpty()) {
				throw new GeneFrequencyNotPresentException(set, geneLocus);
			}
			baseline.remove(geneLocus);
		});
		baseline.forEach(geneLocus -> {
			GeneLocus value = geneLocus.comp_349();
			if(value.genes != null && value.genes.method_40247() != 0) {
				throw new GeneFrequencyNotPresentException(value.genes.method_40239().collect(Collectors.toCollection(Sets::newIdentityHashSet)), geneLocus);
			}
		});
		GeneLocus.setGeneFrequency(geneFrequencyLists);
	}

	private static void buildEntityType2TraitRelations(class_2378<class_1299<?>> entityTypeRegistry) {
		CLLogger.info("Extracting {} EntityType tags for EntityType to Trait Type relations...", entityTypeTag2TraitTypeMap.size());
		entityTypeTag2TraitTypeMap.forEach((tag, builder) -> {
			ImmutableList<class_6880<TraitType>> traitTypes = builder.build();
			class_7923.field_41177.method_40266(tag).ifPresent(entries -> entries.forEach(holder -> holder.method_40230().ifPresent(entityTypeKey -> {
				class_1299<?> entityType = entityTypeRegistry.method_29107(entityTypeKey);
				if(entityType != null) {
					getBuilder(entityType2TraitTypeMap, entityType).addAll(traitTypes);
					countEntityType2Trait += traitTypes.size();
				}
			})));
		});
		entityTypeTag2TraitTypeMap.clear();
		entityType2TraitTypeMap.forEach((entityType, builder) -> {
			ImmutableList<class_6880<TraitType>> traitTypes = builder.build();
			if(entityType instanceof IChromosomeLibEntityType clEntityType) {
				clEntityType.chromosomelib$setTraitTypes(class_6885.method_40242(traitTypes));
			}
		});
		CLLogger.info("Registered {} EntityType to TraitType relations from {} EntityTypes.", countEntityType2Trait, entityType2TraitTypeMap.size());
	}

	private static void buildGeneLocus2GeneRelations() {
		geneLocus2GeneMap.forEach((geneLocusHolder, builder) -> {
			GeneLocus geneLocus = geneLocusHolder.comp_349();
			ImmutableList<class_6880<Gene>> genes = builder.build();
			geneLocus.genes = class_6885.method_40242(genes);
			genes.forEach(geneHolder -> geneHolder.comp_349().geneLocus = class_6880.method_40223(geneLocus));
		});
		CLLogger.info("Registered {} GeneLocus to Gene relations from {} GeneLoci.", countGeneLocus2Gene, geneLocus2GeneMap.size());
	}

	private static void buildChromosome2GeneLocusRelations() {
		chromosome2GeneLocusMap.forEach((chromosomeHolder, builder) -> {
			Chromosome chromosome = chromosomeHolder.comp_349();
			ImmutableList<class_6880<GeneLocus>> geneLoci = builder.build();
			Int2ObjectMap<class_6880<GeneLocus>> leftGeneLocusMap = new Int2ObjectOpenHashMap<>();
			Int2ObjectMap<class_6880<GeneLocus>> rightGeneLocusMap = new Int2ObjectOpenHashMap<>();
			for(class_6880<GeneLocus> geneLocusHolder : geneLoci) {
				GeneLocus geneLocus = geneLocusHolder.comp_349();
				int leftIndex = geneLocus.index(ChromosomeType.LEFT);
				int rightIndex = geneLocus.index(ChromosomeType.RIGHT);
				if(leftIndex >= 0) {
					class_6880<GeneLocus> old = leftGeneLocusMap.put(leftIndex, geneLocusHolder);
					if(old != null) {
						throw new IllegalStateException("GeneLocus conflict at left side, index " + leftIndex + "for chromosome " + chromosomeHolder +
								". Registered " + old.method_40230().orElse(null) + " and " + geneLocusHolder.method_40230().orElse(null) + ".");
					}
				}
				if(rightIndex >= 0) {
					class_6880<GeneLocus> old = rightGeneLocusMap.put(rightIndex, geneLocusHolder);
					if(old != null) {
						throw new IllegalStateException("GeneLocus conflict at right side, index " + rightIndex + "for chromosome " + chromosomeHolder +
								". Registered " + old.method_40230().orElse(null) + " and " + geneLocusHolder.method_40230().orElse(null) + ".");
					}
				}
			}
			chromosome.maintain(Int2ObjectMaps.unmodifiable(leftGeneLocusMap), Int2ObjectMaps.unmodifiable(rightGeneLocusMap));
		});
		CLLogger.info("Registered {} Chromosome to GeneLocus relations from {} Chromosomes.", countChromosome2GeneLocus, chromosome2GeneLocusMap.size());
	}

	private static void buildEntityType2ChromosomeRelations(class_2378<class_1299<?>> entityTypeRegistry) {
		CLLogger.info("Extracting {} EntityType tags for Chromosome index remapper...", taggedChromosomeIndexRemapper.size());
		taggedChromosomeIndexRemapper.forEach((remapTagKey, remappedIndex) -> class_7923.field_41177.method_40266(remapTagKey.entityTypeTag()).ifPresent(entries -> entries.forEach(holder -> holder.method_40230().ifPresent(entityTypeKey -> {
			class_1299<?> entityType = entityTypeRegistry.method_29107(entityTypeKey);
			if(entityType != null) {
				chromosomeIndexRemapper.put(new RemapKey(entityType, remapTagKey.chromosome()), remappedIndex.intValue());
			}
		}))));
		taggedChromosomeIndexRemapper.clear();

		CLLogger.info("Extracting {} EntityType tags for EntityType to Chromosome relations...", entityTypeTag2ChromosomeMap.size());
		entityTypeTag2ChromosomeMap.forEach((tag, builder) -> {
			ImmutableList<class_6880<Chromosome>> chromosomes = builder.build();
			class_7923.field_41177.method_40266(tag).ifPresent(entries -> entries.forEach(holder -> holder.method_40230().ifPresent(entityTypeKey -> {
				class_1299<?> entityType = entityTypeRegistry.method_29107(entityTypeKey);
				if(entityType != null) {
					getBuilder(entityType2ChromosomeMap, entityType).addAll(chromosomes);
					countEntityType2Chromosome += chromosomes.size();
				}
			})));
		});
		entityTypeTag2ChromosomeMap.clear();
		entityType2ChromosomeMap.forEach((entityType, builder) -> {
			ImmutableList<class_6880<Chromosome>> chromosomes = builder.build();
			Int2ObjectMap<class_6880<Chromosome>> chromosomeMap = new Int2ObjectOpenHashMap<>();
			for(class_6880<Chromosome> chromosomeHolder : chromosomes) {
				Chromosome chromosome = chromosomeHolder.comp_349();
				int index = chromosomeIndexRemapper.getInt(new RemapKey(entityType, chromosomeHolder));
				if(index < 0) {
					index = chromosome.index();
				}
				class_6880<Chromosome> old = chromosomeMap.put(index, chromosomeHolder);
				if(old != null) {
					throw new IllegalStateException("Chromosome conflict at index " + chromosome.index() + " for entity type " + entityType +
							". Registered " + old.method_40230().orElse(null) + " and " + chromosomeHolder.method_40230().orElse(null) + ".");
				}
			}
			if(entityType instanceof IChromosomeLibEntityType clEntityType) {
				clEntityType.chromosomelib$setChromosomes(chromosomeMap);
			}
		});
		CLLogger.info("Registered {} EntityType to Chromosome relations from {} EntityTypes.", countEntityType2Chromosome, entityType2ChromosomeMap.size());
	}

	private static void buildTraitType2TraitsReversedRelations(class_2378<Trait> traitRegistry) {
		ListMultimap<class_6880<TraitType>, class_6880<Trait>> traitType2TraitsMap = Multimaps.newListMultimap(Maps.newIdentityHashMap(), Lists::newArrayList);
		traitRegistry.method_40270().forEach(traitHolder -> traitType2TraitsMap.put(traitHolder.comp_349().getType(), traitHolder));
		for(class_6880<TraitType> traitTypeHolder: traitType2TraitsMap.keySet()) {
			traitTypeHolder.comp_349().setValues(class_6885.method_40242(traitType2TraitsMap.get(traitTypeHolder)));
		}
	}

	private static void buildGeneTopologicalOrder(class_2378<Gene> geneRegistry) {
		geneRegistry.method_40295().forEach(disableRelations::addNode);

		ImmutableGraph<class_6880<Gene>> geneGraph = disableRelations.build();
		Object2IntMap<class_6880<Gene>> degrees = new Object2IntRBTreeMap<>(Comparator.comparingInt(class_6880::hashCode));
		Queue<class_6880<Gene>> queue = Queues.newArrayDeque();
		for(class_6880<Gene> gene : geneGraph.nodes()) {
			if(gene instanceof AbstractRegisterEntry<Gene> registerEntry) {
				gene = registerEntry.asHolder();
			}
			int degree = geneGraph.inDegree(gene);
			if(degree == 0) {
				queue.add(gene);
				gene.comp_349().topologicalOrder = 0;
			}
			degrees.put(gene, degree);
		}
		while(!queue.isEmpty()) {
			class_6880<Gene> gene = queue.poll();
			for(class_6880<Gene> neighbor : geneGraph.successors(gene)) {
				int degree = degrees.getInt(neighbor) - 1;
				degrees.put(neighbor, degree);
				if(degree == 0) {
					queue.add(neighbor);
					neighbor.comp_349().topologicalOrder = gene.comp_349().topologicalOrder + 1;
				}
			}
		}
		List<class_6880<Gene>> cyclicGenes = Lists.newArrayList();
		for(class_6880<Gene> gene : geneGraph.nodes()) {
			if(degrees.getInt(gene) != 0) {
				cyclicGenes.add(gene);
			}
		}
		if(!cyclicGenes.isEmpty()) {
			throw new IllegalStateException("Cyclic gene relations detected: " + cyclicGenes);
		}
		Gene.setDisableRelationship(geneGraph);
	}

	private record RemapKey(class_1299<?> entityType, class_6880<Chromosome> chromosome) {
		@Override
		public boolean equals(Object obj) {
			if(this == obj) {
				return true;
			}
			if(obj instanceof RemapKey other) {
				return this.entityType == other.entityType && AbstractRegisterEntry.equals(this.chromosome, other.chromosome);
			}
			return false;
		}

		@Override
		public int hashCode() {
			return this.entityType.hashCode() + AbstractRegisterEntry.hashCode(this.chromosome);
		}
	}

	private record RemapTagKey(class_6862<class_1299<?>> entityTypeTag, class_6880<Chromosome> chromosome) {
		@Override
		public boolean equals(Object obj) {
			if(this == obj) {
				return true;
			}
			if(obj instanceof RemapTagKey other) {
				return this.entityTypeTag.equals(other.entityTypeTag) && AbstractRegisterEntry.equals(this.chromosome, other.chromosome);
			}
			return false;
		}

		@Override
		public int hashCode() {
			return this.entityTypeTag.hashCode() + AbstractRegisterEntry.hashCode(this.chromosome);
		}
	}

	static {
		chromosomeIndexRemapper.defaultReturnValue(-1);
	}
}
