/*
 * Decompiled with CFR 0.152.
 */
package net.hollowcube.polar;

import com.github.luben.zstd.Zstd;
import java.io.DataInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Map;
import net.hollowcube.polar.PaletteUtil;
import net.hollowcube.polar.PolarChunk;
import net.hollowcube.polar.PolarDataConverter;
import net.hollowcube.polar.PolarSection;
import net.hollowcube.polar.PolarWorld;
import net.kyori.adventure.key.Key;
import net.kyori.adventure.nbt.BinaryTag;
import net.kyori.adventure.nbt.CompoundBinaryTag;
import net.minestom.server.coordinate.CoordConversion;
import net.minestom.server.network.NetworkBuffer;
import net.minestom.server.utils.nbt.BinaryTagReader;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;

public class PolarReader {
    static final NetworkBuffer.Type<byte[]> LIGHT_DATA = NetworkBuffer.FixedRawBytes(2048);
    private static final boolean FORCE_LEGACY_NBT = Boolean.getBoolean("polar.debug.force-legacy-nbt");
    static final int MAX_BLOCK_PALETTE_SIZE = 4096;
    static final int MAX_BIOME_PALETTE_SIZE = 512;

    private PolarReader() {
    }

    @NotNull
    public static PolarWorld read(byte @NotNull [] data) {
        return PolarReader.read(data, PolarDataConverter.NOOP);
    }

    @NotNull
    public static PolarWorld read(byte @NotNull [] data, @NotNull PolarDataConverter dataConverter) {
        NetworkBuffer buffer = NetworkBuffer.wrap(data, 0, data.length);
        buffer.writeIndex(data.length);
        Integer magicNumber = buffer.read(NetworkBuffer.INT);
        PolarReader.assertThat(magicNumber == 1349479538, "Invalid magic number");
        short version = buffer.read(NetworkBuffer.SHORT);
        PolarReader.validateVersion(version);
        int dataVersion = version >= 6 ? buffer.read(NetworkBuffer.VAR_INT).intValue() : dataConverter.defaultDataVersion();
        PolarWorld.CompressionType compression = PolarWorld.CompressionType.fromId(buffer.read(NetworkBuffer.BYTE).byteValue());
        PolarReader.assertThat(compression != null, "Invalid compression type");
        Integer compressedDataLength = buffer.read(NetworkBuffer.VAR_INT);
        buffer = PolarReader.decompressBuffer(buffer, compression, compressedDataLength);
        byte minSection = buffer.read(NetworkBuffer.BYTE);
        byte maxSection = buffer.read(NetworkBuffer.BYTE);
        PolarReader.assertThat(minSection < maxSection, "Invalid section range");
        byte[] userData = new byte[]{};
        if (version > 4) {
            userData = buffer.read(NetworkBuffer.BYTE_ARRAY);
        }
        int chunkCount = buffer.read(NetworkBuffer.VAR_INT);
        ArrayList<PolarChunk> chunks = new ArrayList<PolarChunk>(chunkCount);
        for (int i = 0; i < chunkCount; ++i) {
            chunks.add(PolarReader.readChunk(dataConverter, version, dataVersion, buffer, maxSection - minSection + 1));
        }
        return new PolarWorld(version, dataVersion, compression, minSection, maxSection, userData, chunks);
    }

    @NotNull
    private static PolarChunk readChunk(@NotNull PolarDataConverter dataConverter, short version, int dataVersion, @NotNull NetworkBuffer buffer, int sectionCount) {
        Integer chunkX = buffer.read(NetworkBuffer.VAR_INT);
        Integer chunkZ = buffer.read(NetworkBuffer.VAR_INT);
        PolarSection[] sections = new PolarSection[sectionCount];
        for (int i = 0; i < sectionCount; ++i) {
            sections[i] = PolarReader.readSection(dataConverter, version, dataVersion, buffer);
        }
        int blockEntityCount = buffer.read(NetworkBuffer.VAR_INT);
        ArrayList<PolarChunk.BlockEntity> blockEntities = new ArrayList<PolarChunk.BlockEntity>(blockEntityCount);
        for (int i = 0; i < blockEntityCount; ++i) {
            blockEntities.add(PolarReader.readBlockEntity(dataConverter, version, dataVersion, buffer));
        }
        int[][] heightmaps = PolarReader.readHeightmapData(buffer, false);
        byte[] userData = new byte[]{};
        if (version > 2) {
            userData = buffer.read(NetworkBuffer.BYTE_ARRAY);
        }
        return new PolarChunk(chunkX, chunkZ, sections, blockEntities, heightmaps, userData);
    }

    @NotNull
    private static PolarSection readSection(@NotNull PolarDataConverter dataConverter, short version, int dataVersion, @NotNull NetworkBuffer buffer) {
        if (buffer.read(NetworkBuffer.BOOLEAN).booleanValue()) {
            return new PolarSection();
        }
        String[] blockPalette = (String[])buffer.read(NetworkBuffer.STRING.list(4096)).toArray(String[]::new);
        if (dataVersion < dataConverter.dataVersion()) {
            dataConverter.convertBlockPalette(blockPalette, dataVersion, dataConverter.dataVersion());
        }
        PolarReader.upgradeGrassInPalette(blockPalette, version);
        int[] blockData = null;
        if (blockPalette.length > 1) {
            blockData = new int[4096];
            long[] rawBlockData = buffer.read(NetworkBuffer.LONG_ARRAY);
            int bitsPerEntry = (int)Math.ceil(Math.log(blockPalette.length) / Math.log(2.0));
            PaletteUtil.unpack(blockData, rawBlockData, bitsPerEntry);
        }
        String[] biomePalette = (String[])buffer.read(NetworkBuffer.STRING.list(512)).toArray(String[]::new);
        int[] biomeData = null;
        if (biomePalette.length > 1) {
            biomeData = new int[64];
            long[] rawBiomeData = buffer.read(NetworkBuffer.LONG_ARRAY);
            int bitsPerEntry = (int)Math.ceil(Math.log(biomePalette.length) / Math.log(2.0));
            PaletteUtil.unpack(biomeData, rawBiomeData, bitsPerEntry);
        }
        PolarSection.LightContent blockLightContent = PolarSection.LightContent.MISSING;
        PolarSection.LightContent skyLightContent = PolarSection.LightContent.MISSING;
        byte[] blockLight = null;
        byte[] skyLight = null;
        if (version > 1) {
            PolarSection.LightContent lightContent = version >= 7 ? PolarSection.LightContent.VALUES[buffer.read(NetworkBuffer.BYTE)] : (blockLightContent = buffer.read(NetworkBuffer.BOOLEAN) != false ? PolarSection.LightContent.PRESENT : PolarSection.LightContent.MISSING);
            if (blockLightContent == PolarSection.LightContent.PRESENT) {
                blockLight = buffer.read(LIGHT_DATA);
            }
            PolarSection.LightContent lightContent2 = version >= 7 ? PolarSection.LightContent.VALUES[buffer.read(NetworkBuffer.BYTE)] : (skyLightContent = buffer.read(NetworkBuffer.BOOLEAN) != false ? PolarSection.LightContent.PRESENT : PolarSection.LightContent.MISSING);
            if (skyLightContent == PolarSection.LightContent.PRESENT) {
                skyLight = buffer.read(LIGHT_DATA);
            }
        } else if (buffer.read(NetworkBuffer.BOOLEAN).booleanValue()) {
            blockLightContent = PolarSection.LightContent.PRESENT;
            blockLight = buffer.read(LIGHT_DATA);
            skyLightContent = PolarSection.LightContent.PRESENT;
            skyLight = buffer.read(LIGHT_DATA);
        }
        return new PolarSection(blockPalette, blockData, biomePalette, biomeData, blockLightContent, blockLight, skyLightContent, skyLight);
    }

    static void upgradeGrassInPalette(String[] blockPalette, int version) {
        if (version <= 5) {
            for (int i = 0; i < blockPalette.length; ++i) {
                String strippedID;
                if (!blockPalette[i].contains("grass") || !Key.key(strippedID = blockPalette[i].split("\\[")[0]).value().equals("grass")) continue;
                blockPalette[i] = "short_grass";
            }
        }
    }

    static int[][] readHeightmapData(@NotNull NetworkBuffer buffer, boolean skip) {
        int[][] heightmaps = !skip ? new int[32][] : null;
        int heightmapMask = buffer.read(NetworkBuffer.INT);
        for (int i = 0; i < 32; ++i) {
            if ((heightmapMask & 1 << i) == 0) continue;
            if (!skip) {
                long[] packed = buffer.read(NetworkBuffer.LONG_ARRAY);
                if (packed.length == 0) {
                    heightmaps[i] = new int[0];
                    continue;
                }
                int bitsPerEntry = packed.length * 64 / 256;
                heightmaps[i] = new int[256];
                PaletteUtil.unpack(heightmaps[i], packed, bitsPerEntry);
                continue;
            }
            buffer.advanceRead(buffer.read(NetworkBuffer.VAR_INT) * 8);
        }
        return heightmaps;
    }

    @NotNull
    static PolarChunk.BlockEntity readBlockEntity(@NotNull PolarDataConverter dataConverter, int version, int dataVersion, @NotNull NetworkBuffer buffer) {
        int posIndex = buffer.read(NetworkBuffer.INT);
        String id = buffer.read(NetworkBuffer.STRING.optional());
        CompoundBinaryTag nbt = CompoundBinaryTag.empty();
        if (version <= 2 || buffer.read(NetworkBuffer.BOOLEAN).booleanValue()) {
            nbt = version <= 3 || FORCE_LEGACY_NBT ? (CompoundBinaryTag)PolarReader.legacyReadNBT(buffer) : (CompoundBinaryTag)buffer.read(NetworkBuffer.NBT);
        }
        if (dataVersion < dataConverter.dataVersion()) {
            Map.Entry<String, CompoundBinaryTag> converted = dataConverter.convertBlockEntityData(id == null ? "" : id, nbt, dataVersion, dataConverter.dataVersion());
            if ((id = converted.getKey()).isEmpty()) {
                id = null;
            }
            if ((nbt = converted.getValue()).size() == 0) {
                nbt = null;
            }
        }
        return new PolarChunk.BlockEntity(CoordConversion.chunkBlockIndexGetX(posIndex), CoordConversion.chunkBlockIndexGetY(posIndex), CoordConversion.chunkBlockIndexGetZ(posIndex), id, nbt);
    }

    static void validateVersion(int version) {
        String invalidVersionError = String.format("Unsupported Polar version. Up to %d is supported, found %d.", (short)7, version);
        PolarReader.assertThat(version <= 7, invalidVersionError);
    }

    @NotNull
    private static NetworkBuffer decompressBuffer(@NotNull NetworkBuffer buffer, @NotNull PolarWorld.CompressionType compression, int length) {
        return switch (compression) {
            default -> throw new MatchException(null, null);
            case PolarWorld.CompressionType.NONE -> buffer;
            case PolarWorld.CompressionType.ZSTD -> {
                byte[] bytes = Zstd.decompress(buffer.read(NetworkBuffer.RAW_BYTES), length);
                NetworkBuffer newBuffer = NetworkBuffer.wrap(bytes, 0, 0);
                newBuffer.writeIndex(bytes.length);
                yield newBuffer;
            }
        };
    }

    private static BinaryTag legacyReadNBT(final @NotNull NetworkBuffer buffer) {
        try {
            BinaryTagReader nbtReader = new BinaryTagReader(new DataInputStream(new InputStream(){

                @Override
                public int read() {
                    return buffer.read(NetworkBuffer.BYTE) & 0xFF;
                }

                @Override
                public int available() {
                    return (int)buffer.readableBytes();
                }
            }));
            return nbtReader.readNamed().getValue();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Contract(value="false, _ -> fail")
    static void assertThat(boolean condition, @NotNull String message) {
        if (!condition) {
            throw new Error(message);
        }
    }

    public static class Error
    extends RuntimeException {
        private Error(String message) {
            super(message);
        }
    }
}

