/*
 * Decompiled with CFR 0.152.
 */
package nsz.bedrock;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;
import javax.imageio.ImageIO;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.fabricmc.loader.api.FabricLoader;
import org.slf4j.Logger;

@Environment(value=EnvType.CLIENT)
public final class McpackConverter {
    private static final Gson GSON = new Gson();
    private static final Map<String, String> ITEM_NAME_MAP = McpackConverter.createItemNameMap();
    private static final Map<String, String> BLOCK_NAME_MAP = McpackConverter.createBlockNameMap();

    private McpackConverter() {
    }

    private static ZipEntry findEntryBySuffix(ZipFile inZip, String candidate) {
        String cand = McpackConverter.normalize(candidate).toLowerCase();
        Enumeration<? extends ZipEntry> entries = inZip.entries();
        while (entries.hasMoreElements()) {
            String n;
            ZipEntry e = entries.nextElement();
            if (e.isDirectory() || !(n = McpackConverter.normalize(e.getName()).toLowerCase()).endsWith(cand)) continue;
            return e;
        }
        return null;
    }

    private static Integer lookupAnimHint(Map<String, Integer> hints, String entryName) {
        String suf;
        if (hints == null || hints.isEmpty() || entryName == null) {
            return null;
        }
        String key = McpackConverter.normalize(entryName).toLowerCase();
        Integer v = hints.get(key);
        if (v != null) {
            return v;
        }
        int idx = key.lastIndexOf("textures/");
        if (idx >= 0 && (v = hints.get(suf = key.substring(idx))) != null) {
            return v;
        }
        return null;
    }

    public static void convertAll(Logger logger) {
        try {
            Path gameDir = FabricLoader.getInstance().getGameDir();
            Path bedrockPacksDir = gameDir.resolve("bedrockpacks");
            Path resourcePacksDir = gameDir.resolve("resourcepacks");
            Files.createDirectories(bedrockPacksDir, new FileAttribute[0]);
            Files.createDirectories(resourcePacksDir, new FileAttribute[0]);
            McpackConverter.convertFromDir(bedrockPacksDir, resourcePacksDir, logger);
            McpackConverter.convertFromDir(resourcePacksDir, resourcePacksDir, logger);
        }
        catch (IOException e) {
            logger.error("Error preparing pack directories: {}", (Object)e.toString(), (Object)e);
        }
    }

    private static void convertFromDir(Path srcDir, Path resourcePacksDir, Logger logger) {
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(srcDir, "*.mcpack");){
            for (Path mcpack : stream) {
                try {
                    McpackConverter.convertSingle(mcpack, resourcePacksDir, logger);
                }
                catch (Exception e) {
                    logger.error("Failed to convert {}: {}", new Object[]{mcpack.getFileName(), e.toString(), e});
                }
            }
        }
        catch (IOException e) {
            logger.error("Error scanning {} for .mcpack files: {}", new Object[]{srcDir, e.toString(), e});
        }
    }

    public static void startResourcePackWatcher(Logger logger) {
        Thread t = new Thread(() -> {
            try {
                Path gameDir = FabricLoader.getInstance().getGameDir();
                Path resourcePacksDir = gameDir.resolve("resourcepacks");
                Files.createDirectories(resourcePacksDir, new FileAttribute[0]);
                try (WatchService ws = resourcePacksDir.getFileSystem().newWatchService();){
                    WatchKey key;
                    boolean valid;
                    resourcePacksDir.register(ws, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_MODIFY);
                    logger.info("BedrockConnector: watching resourcepacks for .mcpack drops...");
                    do {
                        key = ws.take();
                        for (WatchEvent<?> event : key.pollEvents()) {
                            Path p;
                            String lower;
                            Object ctx;
                            WatchEvent.Kind<?> kind = event.kind();
                            if (kind == StandardWatchEventKinds.OVERFLOW || !((ctx = event.context()) instanceof Path) || !(lower = (p = (Path)ctx).getFileName().toString().toLowerCase()).endsWith(".mcpack")) continue;
                            Path mcpack = resourcePacksDir.resolve(p);
                            try {
                                Thread.sleep(300L);
                            }
                            catch (InterruptedException interruptedException) {
                                // empty catch block
                            }
                            try {
                                McpackConverter.convertSingle(mcpack, resourcePacksDir, logger);
                            }
                            catch (Exception ex) {
                                logger.error("Failed to convert dropped pack {}: {}", new Object[]{p, ex.toString(), ex});
                            }
                        }
                    } while (valid = key.reset());
                }
            }
            catch (Exception e) {
                logger.warn("Resource pack watcher stopped: {}", (Object)e.toString());
            }
        }, "BedrockConnector-ResourcePackWatcher");
        t.setDaemon(true);
        t.start();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static int getImageWidth(ZipFile inZip, ZipEntry entry, boolean isTga) {
        try (InputStream is = inZip.getInputStream(entry);){
            if (isTga) {
                TgaImage tga = McpackConverter.readTga(is);
                int n2 = tga != null ? tga.width : -1;
                return n2;
            }
            BufferedImage img = ImageIO.read(is);
            int n = img != null ? img.getWidth() : -1;
            return n;
        }
        catch (Exception e) {
            return -1;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static int getImageHeight(ZipFile inZip, ZipEntry entry, boolean isTga) {
        try (InputStream is = inZip.getInputStream(entry);){
            if (isTga) {
                TgaImage tga = McpackConverter.readTga(is);
                int n2 = tga != null ? tga.height : -1;
                return n2;
            }
            BufferedImage img = ImageIO.read(is);
            int n = img != null ? img.getHeight() : -1;
            return n;
        }
        catch (Exception e) {
            return -1;
        }
    }

    private static byte[] convertTgaToPngBytes(InputStream is, Logger logger) {
        try {
            TgaImage tga = McpackConverter.readTga(is);
            if (tga == null) {
                return null;
            }
            BufferedImage out = new BufferedImage(tga.width, tga.height, tga.hasAlpha ? 2 : 1);
            out.setRGB(0, 0, tga.width, tga.height, tga.argb, 0, tga.width);
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ImageIO.write((RenderedImage)out, "png", baos);
            return baos.toByteArray();
        }
        catch (Exception e) {
            logger.warn("Failed to convert TGA: {}", (Object)e.toString());
            return null;
        }
    }

    private static byte[] convertRasterToPngBytes(InputStream is, Logger logger) {
        try {
            BufferedImage img = ImageIO.read(is);
            if (img == null) {
                return null;
            }
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ImageIO.write((RenderedImage)img, "png", baos);
            return baos.toByteArray();
        }
        catch (Exception e) {
            logger.warn("Failed to convert raster image to PNG: {}", (Object)e.toString());
            return null;
        }
    }

    private static TgaImage readTga(InputStream is) throws IOException {
        int bytesPerPixel;
        int imageSize;
        byte[] data;
        byte[] header = is.readNBytes(18);
        if (header.length < 18) {
            return null;
        }
        int idLen = header[0] & 0xFF;
        int colorMapType = header[1] & 0xFF;
        int imageType = header[2] & 0xFF;
        int width = header[12] & 0xFF | (header[13] & 0xFF) << 8;
        int height = header[14] & 0xFF | (header[15] & 0xFF) << 8;
        int bpp = header[16] & 0xFF;
        int desc = header[17] & 0xFF;
        if (imageType != 2 || colorMapType != 0 || bpp != 24 && bpp != 32) {
            return null;
        }
        if (idLen > 0) {
            is.readNBytes(idLen);
        }
        if ((data = is.readNBytes(imageSize = width * height * (bytesPerPixel = bpp / 8))).length < imageSize) {
            return null;
        }
        boolean flipVert = (desc & 0x20) == 0;
        int[] argb = new int[width * height];
        int idx = 0;
        for (int y = 0; y < height; ++y) {
            int row = flipVert ? height - 1 - y : y;
            for (int x = 0; x < width; ++x) {
                int b = data[idx++] & 0xFF;
                int g = data[idx++] & 0xFF;
                int r = data[idx++] & 0xFF;
                int a = bytesPerPixel == 4 ? data[idx++] & 0xFF : 255;
                argb[row * width + x] = a << 24 | r << 16 | g << 8 | b;
            }
        }
        TgaImage t = new TgaImage();
        t.width = width;
        t.height = height;
        t.hasAlpha = bytesPerPixel == 4;
        t.argb = argb;
        return t;
    }

    private static void copySounds(ZipFile inZip, ZipOutputStream out, Logger logger, Set<String> writtenPaths) throws IOException {
        Enumeration<? extends ZipEntry> entries = inZip.entries();
        boolean hasSoundDefs = false;
        ZipEntry soundDefsEntry = null;
        String clickCandidate = null;
        JsonObject outSoundsAll = new JsonObject();
        boolean hasUiClickEvent = false;
        while (entries.hasMoreElements()) {
            String outPath;
            ZipEntry e = entries.nextElement();
            if (e.isDirectory()) continue;
            String name = McpackConverter.normalize(e.getName());
            String lower = name.toLowerCase();
            if (lower.equals("sounds/sound_definitions.json") || lower.equals("sound_definitions.json")) {
                hasSoundDefs = true;
                soundDefsEntry = e;
            }
            if (!lower.startsWith("sounds/") || !lower.endsWith(".ogg")) continue;
            String rel = name.substring("sounds/".length());
            String relLower = rel.toLowerCase();
            if ((relLower.endsWith("click.ogg") || relLower.contains("/click.ogg")) && clickCandidate == null) {
                clickCandidate = McpackConverter.normalize(McpackConverter.stripExtension(rel));
            }
            if (!McpackConverter.registerAndCheck(writtenPaths, outPath = "assets/minecraft/sounds/" + rel)) continue;
            McpackConverter.writeZipEntry(out, outPath, inZip, e);
        }
        if (hasSoundDefs && soundDefsEntry != null) {
            try (InputStream is = inZip.getInputStream(soundDefsEntry);){
                JsonObject root = JsonParser.parseReader((Reader)new InputStreamReader(is, StandardCharsets.UTF_8)).getAsJsonObject();
                JsonObject defs = null;
                if (root.has("sound_definitions") && root.get("sound_definitions").isJsonObject()) {
                    defs = root.getAsJsonObject("sound_definitions");
                }
                if (defs != null) {
                    for (String key : defs.keySet()) {
                        JsonObject def = defs.getAsJsonObject(key);
                        if (def == null || !def.has("sounds")) continue;
                        JsonArray arr = def.getAsJsonArray("sounds");
                        JsonArray javaSounds = new JsonArray();
                        for (JsonElement el : arr) {
                            JsonObject so;
                            String sname = null;
                            if (el.isJsonPrimitive()) {
                                sname = el.getAsString();
                            } else if (el.isJsonObject() && (so = el.getAsJsonObject()).has("name")) {
                                sname = so.get("name").getAsString();
                            }
                            if (sname == null) continue;
                            if ((sname = McpackConverter.normalize(sname)).startsWith("sounds/")) {
                                sname = sname.substring("sounds/".length());
                            }
                            if (sname.endsWith(".ogg")) {
                                sname = sname.substring(0, sname.length() - 4);
                            }
                            javaSounds.add(sname);
                        }
                        if (javaSounds.size() <= 0) continue;
                        JsonObject ev = new JsonObject();
                        ev.addProperty("replace", Boolean.valueOf(true));
                        ev.add("sounds", (JsonElement)javaSounds);
                        outSoundsAll.add(key, (JsonElement)ev);
                        if (!"ui.button.click".equals(key)) continue;
                        hasUiClickEvent = true;
                    }
                }
            }
            catch (Exception ex) {
                logger.warn("Unable to convert Bedrock sound_definitions.json: {}", (Object)ex.toString());
            }
        }
        if (!hasUiClickEvent && clickCandidate != null) {
            JsonObject ev = new JsonObject();
            ev.addProperty("replace", Boolean.valueOf(true));
            JsonArray arr = new JsonArray();
            arr.add(clickCandidate);
            ev.add("sounds", (JsonElement)arr);
            outSoundsAll.add("ui.button.click", (JsonElement)ev);
        }
        if (outSoundsAll.size() > 0) {
            McpackConverter.writeStringEntry(out, "assets/minecraft/sounds.json", GSON.toJson((JsonElement)outSoundsAll));
        }
    }

    private static Set<String> readTexturesList(ZipFile inZip, Logger logger) {
        HashSet<String> s = new HashSet<String>();
        try {
            Enumeration<? extends ZipEntry> entries = inZip.entries();
            while (entries.hasMoreElements()) {
                String n;
                ZipEntry e = entries.nextElement();
                if (e.isDirectory() || !(n = McpackConverter.normalize(e.getName()).toLowerCase()).endsWith("/textures_list.json") && !n.equals("textures_list.json")) continue;
                try {
                    InputStream is = inZip.getInputStream(e);
                    try {
                        JsonArray arr = JsonParser.parseReader((Reader)new InputStreamReader(is, StandardCharsets.UTF_8)).getAsJsonArray();
                        for (JsonElement el : arr) {
                            if (!el.isJsonPrimitive()) continue;
                            s.add(McpackConverter.normalize(el.getAsString()).toLowerCase());
                        }
                    }
                    finally {
                        if (is == null) continue;
                        is.close();
                    }
                }
                catch (Exception exception) {}
            }
            if (!s.isEmpty()) {
                logger.info("Found {} textures_list.json entries aggregated: {} items", (Object)1, (Object)s.size());
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return s;
    }

    private static void copyFromTextureJson(ZipFile inZip, ZipOutputStream out, Logger logger, Set<String> writtenPaths, Map<String, Integer> animHints, String jsonPath, boolean isBlock) throws IOException {
        try {
            ZipEntry e = inZip.getEntry(jsonPath);
            if (e == null || e.isDirectory()) {
                return;
            }
            try (InputStream is = inZip.getInputStream(e);){
                JsonObject texturesObj;
                JsonObject root = JsonParser.parseReader((Reader)new InputStreamReader(is, StandardCharsets.UTF_8)).getAsJsonObject();
                JsonObject texData = null;
                if (root.has("texture_data") && root.get("texture_data").isJsonObject()) {
                    texData = root.getAsJsonObject("texture_data");
                } else if (root.has("textures") && root.get("textures").isJsonObject() && (texturesObj = root.getAsJsonObject("textures")).has("texture_data") && texturesObj.get("texture_data").isJsonObject()) {
                    texData = texturesObj.getAsJsonObject("texture_data");
                }
                if (texData == null) {
                    return;
                }
                for (String key : texData.keySet()) {
                    JsonObject def = texData.getAsJsonObject(key);
                    if (def == null) continue;
                    ArrayList<String> texPaths = new ArrayList<String>();
                    if (def.has("textures")) {
                        if (def.get("textures").isJsonPrimitive()) {
                            texPaths.add(McpackConverter.normalize(def.get("textures").getAsString()));
                        } else if (def.get("textures").isJsonArray()) {
                            for (JsonElement el2 : def.getAsJsonArray("textures")) {
                                if (!el2.isJsonPrimitive()) continue;
                                texPaths.add(McpackConverter.normalize(el2.getAsString()));
                            }
                        }
                    }
                    String mappedNameBase = isBlock ? McpackConverter.mapBlockName(key.toLowerCase()) : McpackConverter.mapItemName(key.toLowerCase());
                    for (String p : texPaths) {
                        String mcmetaPath;
                        String[] candidates = new String[]{p, p + ".png", p + ".tga", p + ".jpg", p + ".jpeg"};
                        ZipEntry src = null;
                        String srcName = null;
                        boolean isTga = false;
                        for (String cand : candidates) {
                            src = McpackConverter.findEntryBySuffix(inZip, cand);
                            if (src == null || src.isDirectory()) continue;
                            srcName = McpackConverter.normalize(src.getName());
                            isTga = srcName.toLowerCase().endsWith(".tga");
                            break;
                        }
                        if (src == null) continue;
                        String target = (isBlock ? "assets/minecraft/textures/block/" : "assets/minecraft/textures/item/") + mappedNameBase + ".png";
                        if (isTga) {
                            if (McpackConverter.registerAndCheck(writtenPaths, target)) {
                                InputStream tis = inZip.getInputStream(src);
                                try {
                                    byte[] pngBytes = McpackConverter.convertTgaToPngBytes(tis, logger);
                                    if (pngBytes == null) continue;
                                    McpackConverter.writeZipEntryBytes(out, target, pngBytes);
                                }
                                finally {
                                    if (tis == null) continue;
                                    tis.close();
                                    continue;
                                }
                            }
                        } else if (McpackConverter.registerAndCheck(writtenPaths, target)) {
                            McpackConverter.writeZipEntry(out, target, inZip, src);
                        }
                        if (!McpackConverter.registerAndCheck(writtenPaths, mcmetaPath = target + ".mcmeta")) continue;
                        int width = McpackConverter.getImageWidth(inZip, src, isTga);
                        int height = McpackConverter.getImageHeight(inZip, src, isTga);
                        if (width > 0 && height > 0 && height >= width * 2 && height % width == 0) {
                            int frameTime = 2;
                            Integer hint = McpackConverter.lookupAnimHint(animHints, srcName);
                            if (hint != null && hint > 0) {
                                frameTime = hint;
                            }
                            JsonObject anim = new JsonObject();
                            JsonObject rootOut = new JsonObject();
                            anim.addProperty("frametime", (Number)frameTime);
                            anim.addProperty("height", (Number)width);
                            rootOut.add("animation", (JsonElement)anim);
                            McpackConverter.writeStringEntry(out, mcmetaPath, GSON.toJson((JsonElement)rootOut));
                            continue;
                        }
                        writtenPaths.remove(McpackConverter.normalize(mcmetaPath));
                    }
                }
            }
        }
        catch (Exception ex) {
            logger.debug("Ignoring {} mapping pass due to error: {}", (Object)jsonPath, (Object)ex.toString());
        }
    }

    private static void convertSingle(Path mcpackPath, Path resourcePacksDir, Logger logger) throws IOException {
        long outTime;
        String baseName = McpackConverter.stripExtension(mcpackPath.getFileName().toString());
        String safeName = McpackConverter.sanitize(baseName);
        Path outZip = resourcePacksDir.resolve(safeName + "-converted.zip");
        long inTime = Files.getLastModifiedTime(mcpackPath, new LinkOption[0]).toMillis();
        if (Files.exists(outZip, new LinkOption[0]) && (outTime = Files.getLastModifiedTime(outZip, new LinkOption[0]).toMillis()) >= inTime) {
            logger.info("Skipping {} (already converted)", (Object)mcpackPath.getFileName());
            return;
        }
        if (!Files.isRegularFile(mcpackPath, new LinkOption[0]) || !McpackConverter.isLikelyZip(mcpackPath)) {
            logger.warn("Skipping {}: not a valid ZIP/MCPACK (missing PK header or unreadable)", (Object)mcpackPath.getFileName());
            return;
        }
        logger.info("Converting Bedrock pack: {} -> {}", (Object)mcpackPath.getFileName(), (Object)outZip.getFileName());
        try (ZipFile inZip = new ZipFile(mcpackPath.toFile());
             BufferedOutputStream fos = new BufferedOutputStream(Files.newOutputStream(outZip, new OpenOption[0]));
             ZipOutputStream out = new ZipOutputStream(fos);){
            PackMeta meta = McpackConverter.readBedrockManifest(inZip, logger);
            int packFormat = McpackConverter.getCurrentResourcePackFormat();
            String description = McpackConverter.buildDescription(meta);
            McpackConverter.writeStringEntry(out, "pack.mcmeta", McpackConverter.buildPackMcmeta(packFormat, description));
            McpackConverter.copyPackIcon(inZip, out);
            McpackConverter.copyTranslations(inZip, out, logger, new HashSet<String>());
            McpackConverter.logUnsupportedJsons(inZip, logger);
            McpackConverter.copyTextures(inZip, out, logger, new HashSet<String>());
            McpackConverter.copySounds(inZip, out, logger, new HashSet<String>());
        }
    }

    private static int getCurrentResourcePackFormat() {
        return 46;
    }

    private static void copyPackIcon(ZipFile inZip, ZipOutputStream out) throws IOException {
        ZipEntry icon = inZip.getEntry("pack_icon.png");
        if (icon != null && !icon.isDirectory()) {
            McpackConverter.writeZipEntry(out, "pack.png", inZip, icon);
        }
    }

    private static void copyTranslations(ZipFile inZip, ZipOutputStream out, Logger logger, Set<String> writtenPaths) throws IOException {
        HashMap<String, JsonObject> localeMap = new HashMap<String, JsonObject>();
        Enumeration<? extends ZipEntry> entries = inZip.entries();
        while (entries.hasMoreElements()) {
            String localeRaw;
            String rel;
            String fileName;
            String ext;
            String name;
            String lower;
            ZipEntry e = entries.nextElement();
            if (e.isDirectory() || !(lower = (name = McpackConverter.normalize(e.getName())).toLowerCase()).startsWith("texts/") || lower.contains("/font/") || !".lang".equals(ext = McpackConverter.getExtension(fileName = McpackConverter.filenameOnly(rel = name.substring("texts/".length()))).toLowerCase()) && !".json".equals(ext) || (localeRaw = McpackConverter.stripExtension(fileName)).isEmpty()) continue;
            String locale = localeRaw.replace('-', '_').toLowerCase();
            JsonObject outObj = localeMap.computeIfAbsent(locale, k -> new JsonObject());
            try {
                if (".lang".equals(ext)) {
                    BufferedReader br = new BufferedReader(new InputStreamReader(inZip.getInputStream(e), StandardCharsets.UTF_8));
                    try {
                        String line;
                        boolean first = true;
                        while ((line = br.readLine()) != null) {
                            if (first && !line.isEmpty() && line.charAt(0) == '\ufeff') {
                                line = line.substring(1);
                            }
                            first = false;
                            if ((line = line.trim()).isEmpty() || line.startsWith("#") || line.startsWith("//") || line.startsWith(";")) continue;
                            int sep = line.indexOf(61);
                            if (sep < 0) {
                                sep = line.indexOf(58);
                            }
                            if (sep <= 0) continue;
                            String key = line.substring(0, sep).trim();
                            String val = line.substring(sep + 1).trim();
                            if (key.isEmpty()) continue;
                            outObj.addProperty(key, val);
                        }
                        continue;
                    }
                    finally {
                        br.close();
                        continue;
                    }
                }
                InputStream is = inZip.getInputStream(e);
                try {
                    JsonElement el = JsonParser.parseReader((Reader)new InputStreamReader(is, StandardCharsets.UTF_8));
                    if (el == null || !el.isJsonObject()) continue;
                    JsonObject obj = el.getAsJsonObject();
                    for (Map.Entry entry : obj.entrySet()) {
                        if (((JsonElement)entry.getValue()).isJsonPrimitive()) {
                            outObj.add((String)entry.getKey(), (JsonElement)entry.getValue());
                            continue;
                        }
                        outObj.addProperty((String)entry.getKey(), ((JsonElement)entry.getValue()).toString());
                    }
                }
                finally {
                    if (is == null) continue;
                    is.close();
                }
            }
            catch (Exception ex) {
                logger.debug("Skipping texts file {}: {}", (Object)name, (Object)ex.toString());
            }
        }
        for (Map.Entry ent : localeMap.entrySet()) {
            String outPath;
            if (((JsonObject)ent.getValue()).size() == 0 || !McpackConverter.registerAndCheck(writtenPaths, outPath = "assets/minecraft/lang/" + (String)ent.getKey() + ".json")) continue;
            McpackConverter.writeStringEntry(out, outPath, GSON.toJson((JsonElement)ent.getValue()));
        }
    }

    private static void logUnsupportedJsons(ZipFile inZip, Logger logger) {
        try {
            Enumeration<? extends ZipEntry> entries = inZip.entries();
            while (entries.hasMoreElements()) {
                ZipEntry e = entries.nextElement();
                if (e.isDirectory()) continue;
                String n = McpackConverter.normalize(e.getName()).toLowerCase();
                if (n.equals("biomes_client.json") || n.endsWith("/biomes_client.json")) {
                    logger.info("Found Bedrock biomes_client.json (not convertible to Java). Skipping.");
                    continue;
                }
                if (n.equals("blocks.json") || n.endsWith("/blocks.json")) {
                    logger.info("Found Bedrock blocks.json (not convertible to Java). Skipping.");
                    continue;
                }
                if (!n.startsWith("particles/") || !n.endsWith(".json")) continue;
                logger.info("Found Bedrock particle definition {} (not convertible to Java). Skipping.", (Object)e.getName());
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private static void copyTextures(ZipFile inZip, ZipOutputStream out, Logger logger, Set<String> writtenPaths) throws IOException {
        Map<String, Integer> animHints = McpackConverter.readBedrockAnimationHints(inZip, logger);
        McpackConverter.readTexturesList(inZip, logger);
        HashMap<String, byte[]> convertedPngCache = new HashMap<String, byte[]>();
        boolean hasDrinkableBase = false;
        boolean hasSplashBase = false;
        boolean hasLingeringBase = false;
        Enumeration<? extends ZipEntry> scan = inZip.entries();
        while (scan.hasMoreElements()) {
            String nl;
            ZipEntry se = scan.nextElement();
            if (se.isDirectory() || !(nl = McpackConverter.normalize(se.getName()).toLowerCase()).endsWith(".png") && !nl.endsWith(".tga") && !nl.endsWith(".jpg") && !nl.endsWith(".jpeg") || !nl.contains("/potion_bottle_")) continue;
            if (nl.endsWith("potion_bottle_drinkable.png") || nl.endsWith("potion_bottle_drinkable.tga") || nl.endsWith("potion_bottle_drinkable.jpg") || nl.endsWith("potion_bottle_drinkable.jpeg")) {
                hasDrinkableBase = true;
            }
            if (nl.endsWith("potion_bottle_splash.png") || nl.endsWith("potion_bottle_splash.tga") || nl.endsWith("potion_bottle_splash.jpg") || nl.endsWith("potion_bottle_splash.jpeg")) {
                hasSplashBase = true;
            }
            if (!nl.endsWith("potion_bottle_lingering.png") && !nl.endsWith("potion_bottle_lingering.tga") && !nl.endsWith("potion_bottle_lingering.jpg") && !nl.endsWith("potion_bottle_lingering.jpeg")) continue;
            hasLingeringBase = true;
        }
        Enumeration<? extends ZipEntry> entries = inZip.entries();
        while (entries.hasMoreElements()) {
            String mcmetaPath;
            String lowerMapped;
            boolean isJpeg;
            ZipEntry e = entries.nextElement();
            if (e.isDirectory()) continue;
            String name = McpackConverter.normalize(e.getName());
            String nameLower = name.toLowerCase();
            if (!name.startsWith("textures/") && !name.startsWith("font/") && !name.startsWith("ui/") && !name.startsWith("texts/") && !name.startsWith("assets/")) continue;
            boolean isTga = nameLower.endsWith(".tga");
            boolean bl = isJpeg = nameLower.endsWith(".jpg") || nameLower.endsWith(".jpeg");
            if (!isTga && !nameLower.endsWith(".png") && !isJpeg) continue;
            Object mapped = name.startsWith("textures/") ? McpackConverter.mapBedrockPathToJava(name) : (name.startsWith("font/") ? McpackConverter.mapFontPathToJava(name) : (name.startsWith("ui/") ? McpackConverter.mapUiPathToJava(name) : (name.startsWith("texts/") ? McpackConverter.mapTextsPathToJava(name) : McpackConverter.normalize(name))));
            if (mapped == null) {
                if (name.startsWith("textures/")) {
                    String rel = name.substring("textures/".length());
                    mapped = rel.startsWith("blocks/") ? "assets/minecraft/textures/block/" + rel.substring("blocks/".length()) : (rel.startsWith("items/") ? "assets/minecraft/textures/item/" + rel.substring("items/".length()) : "assets/minecraft/textures/" + rel);
                } else {
                    logger.warn("Skipping non-texture asset not mapped: {}", (Object)name);
                    continue;
                }
            }
            String baseLower = McpackConverter.stripExtension(McpackConverter.filenameOnly(nameLower));
            String mappedLower = ((String)mapped).toLowerCase();
            if (baseLower.startsWith("potion_bottle_")) {
                if (mappedLower.endsWith("/textures/item/potion.png")) {
                    if (hasDrinkableBase && !baseLower.equals("potion_bottle_drinkable")) {
                        continue;
                    }
                } else if (!mappedLower.endsWith("/textures/item/splash_potion.png") ? mappedLower.endsWith("/textures/item/lingering_potion.png") && hasLingeringBase && !baseLower.equals("potion_bottle_lingering") : hasSplashBase && !baseLower.equals("potion_bottle_splash")) continue;
            }
            if (isTga || isJpeg) {
                Object pngOut = mapped;
                String lowerOut = ((String)pngOut).toLowerCase();
                if (!lowerOut.endsWith(".png")) {
                    int dot = ((String)pngOut).lastIndexOf(46);
                    pngOut = dot >= 0 ? ((String)pngOut).substring(0, dot) + ".png" : (String)pngOut + ".png";
                }
                if (McpackConverter.registerAndCheck(writtenPaths, (String)pngOut)) {
                    InputStream is = inZip.getInputStream(e);
                    try {
                        byte[] pngBytes = isTga ? McpackConverter.convertTgaToPngBytes(is, logger) : McpackConverter.convertRasterToPngBytes(is, logger);
                        if (pngBytes == null) continue;
                        McpackConverter.writeZipEntryBytes(out, (String)pngOut, pngBytes);
                        convertedPngCache.put(McpackConverter.normalize(e.getName()), pngBytes);
                        mapped = pngOut;
                    }
                    finally {
                        if (is == null) continue;
                        is.close();
                        continue;
                    }
                }
            } else if (McpackConverter.registerAndCheck(writtenPaths, (String)mapped)) {
                McpackConverter.writeZipEntry(out, (String)mapped, inZip, e);
            }
            if ((lowerMapped = ((String)mapped).toLowerCase()).endsWith(".png")) {
                boolean isEnder;
                byte[] aliasBytes;
                byte[] byArray = aliasBytes = isTga || isJpeg ? (byte[])convertedPngCache.get(McpackConverter.normalize(e.getName())) : null;
                if (lowerMapped.contains("/gui/sprites/crosshair/crosshair")) {
                    String ext = McpackConverter.getExtension((String)mapped);
                    p1 = "assets/minecraft/textures/gui/sprites/crosshair" + ext;
                    String p2 = "assets/minecraft/textures/gui/crosshair" + ext;
                    if (aliasBytes != null) {
                        if (McpackConverter.registerAndCheck(writtenPaths, p1)) {
                            McpackConverter.writeZipEntryBytes(out, p1, aliasBytes);
                        }
                        if (McpackConverter.registerAndCheck(writtenPaths, p2)) {
                            McpackConverter.writeZipEntryBytes(out, p2, aliasBytes);
                        }
                    } else {
                        McpackConverter.safeWrite(out, p1, inZip, e, writtenPaths);
                        McpackConverter.safeWrite(out, p2, inZip, e, writtenPaths);
                    }
                } else if (lowerMapped.contains("/gui/sprites/hotbar/hotbar")) {
                    String ext = McpackConverter.getExtension((String)mapped);
                    p1 = "assets/minecraft/textures/gui/sprites/hotbar" + ext;
                    String p2 = "assets/minecraft/textures/gui/hotbar" + ext;
                    if (aliasBytes != null) {
                        if (McpackConverter.registerAndCheck(writtenPaths, p1)) {
                            McpackConverter.writeZipEntryBytes(out, p1, aliasBytes);
                        }
                        if (McpackConverter.registerAndCheck(writtenPaths, p2)) {
                            McpackConverter.writeZipEntryBytes(out, p2, aliasBytes);
                        }
                    } else {
                        McpackConverter.safeWrite(out, p1, inZip, e, writtenPaths);
                        McpackConverter.safeWrite(out, p2, inZip, e, writtenPaths);
                    }
                } else if (lowerMapped.endsWith("/gui/icons.png")) {
                    String p = "assets/minecraft/textures/gui/sprites/hud/icons.png";
                    if (aliasBytes != null) {
                        if (McpackConverter.registerAndCheck(writtenPaths, p)) {
                            McpackConverter.writeZipEntryBytes(out, p, aliasBytes);
                        }
                    } else {
                        McpackConverter.safeWrite(out, p, inZip, e, writtenPaths);
                    }
                } else if (lowerMapped.endsWith("/gui/gui.png")) {
                    String p = "assets/minecraft/textures/gui/sprites/hotbar/hotbar.png";
                    if (aliasBytes != null) {
                        if (McpackConverter.registerAndCheck(writtenPaths, p)) {
                            McpackConverter.writeZipEntryBytes(out, p, aliasBytes);
                        }
                    } else {
                        McpackConverter.safeWrite(out, p, inZip, e, writtenPaths);
                    }
                } else if (lowerMapped.endsWith("/gui/gui2.png")) {
                    String p = "assets/minecraft/textures/gui/sprites/hotbar/hotbar.png";
                    if (aliasBytes != null) {
                        if (McpackConverter.registerAndCheck(writtenPaths, p)) {
                            McpackConverter.writeZipEntryBytes(out, p, aliasBytes);
                        }
                    } else {
                        McpackConverter.safeWrite(out, p, inZip, e, writtenPaths);
                    }
                } else if (lowerMapped.endsWith("/textures/font/default.png")) {
                    String p = "assets/minecraft/textures/font/ascii.png";
                    if (aliasBytes != null) {
                        if (McpackConverter.registerAndCheck(writtenPaths, p)) {
                            McpackConverter.writeZipEntryBytes(out, p, aliasBytes);
                        }
                    } else {
                        McpackConverter.safeWrite(out, p, inZip, e, writtenPaths);
                    }
                } else if (lowerMapped.contains("/textures/entity/chest/") && !(isEnder = lowerMapped.endsWith("/entity/chest/ender.png")) && (baseLower.contains("double") || baseLower.contains("large"))) {
                    String extOut = McpackConverter.getExtension((String)mapped);
                    String var = "normal";
                    if (lowerMapped.contains("/entity/chest/trapped")) {
                        var = "trapped";
                    } else if (lowerMapped.contains("/entity/chest/christmas")) {
                        var = "christmas";
                    }
                    boolean wroteLeft = lowerMapped.endsWith("/entity/chest/" + var + "_left.png");
                    boolean wroteRight = lowerMapped.endsWith("/entity/chest/" + var + "_right.png");
                    String leftPath = "assets/minecraft/textures/entity/chest/" + var + "_left" + extOut;
                    String rightPath = "assets/minecraft/textures/entity/chest/" + var + "_right" + extOut;
                    if (!wroteLeft) {
                        if (aliasBytes != null) {
                            if (McpackConverter.registerAndCheck(writtenPaths, leftPath)) {
                                McpackConverter.writeZipEntryBytes(out, leftPath, aliasBytes);
                            }
                        } else {
                            McpackConverter.safeWrite(out, leftPath, inZip, e, writtenPaths);
                        }
                    }
                    if (!wroteRight) {
                        if (aliasBytes != null) {
                            if (McpackConverter.registerAndCheck(writtenPaths, rightPath)) {
                                McpackConverter.writeZipEntryBytes(out, rightPath, aliasBytes);
                            }
                        } else {
                            McpackConverter.safeWrite(out, rightPath, inZip, e, writtenPaths);
                        }
                    }
                }
            }
            if (lowerMapped.endsWith("/misc/enchanted_glint_item.png")) {
                byte[] aliasBytes;
                String p = "assets/minecraft/textures/misc/enchanted_glint_entity.png";
                byte[] byArray = aliasBytes = isTga || isJpeg ? (byte[])convertedPngCache.get(McpackConverter.normalize(e.getName())) : null;
                if (aliasBytes != null) {
                    if (McpackConverter.registerAndCheck(writtenPaths, p)) {
                        McpackConverter.writeZipEntryBytes(out, p, aliasBytes);
                    }
                } else {
                    McpackConverter.safeWrite(out, p, inZip, e, writtenPaths);
                }
            }
            if ((lowerMapped.endsWith("/block/fire_0.png") || lowerMapped.endsWith("/block/fire_1.png")) && McpackConverter.registerAndCheck(writtenPaths, mcmetaPath = (String)mapped + ".mcmeta")) {
                int width = 16;
                try {
                    int w = McpackConverter.getImageWidth(inZip, e, isTga);
                    if (w > 0) {
                        width = w;
                    }
                }
                catch (Exception w) {
                    // empty catch block
                }
                JsonObject anim = new JsonObject();
                JsonObject root = new JsonObject();
                anim.addProperty("frametime", (Number)2);
                anim.addProperty("height", (Number)width);
                root.add("animation", (JsonElement)anim);
                McpackConverter.writeStringEntry(out, mcmetaPath, GSON.toJson((JsonElement)root));
            }
            if (isTga || !lowerMapped.contains("/textures/block/") && !lowerMapped.contains("/textures/item/") || !McpackConverter.registerAndCheck(writtenPaths, mcmetaPath = (String)mapped + ".mcmeta")) continue;
            int width = McpackConverter.getImageWidth(inZip, e, false);
            int height = McpackConverter.getImageHeight(inZip, e, false);
            if (width > 0 && height > 0 && height >= width * 2 && height % width == 0) {
                int frameTime = 2;
                Integer hint = McpackConverter.lookupAnimHint(animHints, name);
                if (hint != null && hint > 0) {
                    frameTime = hint;
                }
                JsonObject anim = new JsonObject();
                JsonObject root = new JsonObject();
                anim.addProperty("frametime", (Number)frameTime);
                anim.addProperty("height", (Number)width);
                root.add("animation", (JsonElement)anim);
                McpackConverter.writeStringEntry(out, mcmetaPath, GSON.toJson((JsonElement)root));
                continue;
            }
            writtenPaths.remove(McpackConverter.normalize(mcmetaPath));
        }
        Enumeration<? extends ZipEntry> all = inZip.entries();
        while (all.hasMoreElements()) {
            ZipEntry ze = all.nextElement();
            if (ze.isDirectory()) continue;
            String n = McpackConverter.normalize(ze.getName()).toLowerCase();
            if (n.endsWith("/terrain_texture.json") || n.equals("terrain_texture.json")) {
                McpackConverter.copyFromTextureJson(inZip, out, logger, writtenPaths, animHints, ze.getName(), true);
                continue;
            }
            if (!n.endsWith("/item_texture.json") && !n.equals("item_texture.json")) continue;
            McpackConverter.copyFromTextureJson(inZip, out, logger, writtenPaths, animHints, ze.getName(), false);
        }
    }

    private static Map<String, Integer> readBedrockAnimationHints(ZipFile inZip, Logger logger) {
        HashMap<String, Integer> hints = new HashMap<String, Integer>();
        Enumeration<? extends ZipEntry> entries = inZip.entries();
        while (entries.hasMoreElements()) {
            ZipEntry e = entries.nextElement();
            if (e.isDirectory()) continue;
            String n = McpackConverter.normalize(e.getName()).toLowerCase();
            if (n.endsWith("/terrain_texture.json") || n.equals("terrain_texture.json")) {
                McpackConverter.readAnimFromJson(inZip, logger, e.getName(), hints);
                continue;
            }
            if (n.endsWith("/item_texture.json") || n.equals("item_texture.json")) {
                McpackConverter.readAnimFromJson(inZip, logger, e.getName(), hints);
                continue;
            }
            if (!n.endsWith("/flipbook_textures.json") && !n.equals("flipbook_textures.json")) continue;
            McpackConverter.readFlipbookHints(inZip, logger, e.getName(), hints);
        }
        return hints;
    }

    private static void readAnimFromJson(ZipFile inZip, Logger logger, String path, Map<String, Integer> hints) {
        try {
            ZipEntry e = inZip.getEntry(path);
            if (e == null || e.isDirectory()) {
                return;
            }
            try (InputStream is = inZip.getInputStream(e);){
                JsonObject texturesObj;
                JsonObject root = JsonParser.parseReader((Reader)new InputStreamReader(is, StandardCharsets.UTF_8)).getAsJsonObject();
                JsonObject texData = null;
                if (root.has("texture_data") && root.get("texture_data").isJsonObject()) {
                    texData = root.getAsJsonObject("texture_data");
                } else if (root.has("textures") && root.get("textures").isJsonObject() && (texturesObj = root.getAsJsonObject("textures")).has("texture_data") && texturesObj.get("texture_data").isJsonObject()) {
                    texData = texturesObj.getAsJsonObject("texture_data");
                }
                if (texData == null) {
                    return;
                }
                for (String key : texData.keySet()) {
                    JsonObject def = texData.getAsJsonObject(key);
                    if (def == null) continue;
                    JsonObject anim = null;
                    if (def.has("animation") && def.get("animation").isJsonObject()) {
                        anim = def.getAsJsonObject("animation");
                    } else if (def.has("animated") && def.get("animated").isJsonObject()) {
                        anim = def.getAsJsonObject("animated");
                    }
                    if (anim == null) continue;
                    int frameTime = -1;
                    if (anim.has("frametime")) {
                        frameTime = McpackConverter.safeInt(anim.get("frametime"), -1);
                    }
                    if (frameTime <= 0 && anim.has("frame_time")) {
                        frameTime = McpackConverter.safeInt(anim.get("frame_time"), -1);
                    }
                    if (frameTime <= 0) {
                        frameTime = 2;
                    }
                    ArrayList<String> texPaths = new ArrayList<String>();
                    if (def.has("textures")) {
                        if (def.get("textures").isJsonPrimitive()) {
                            String p = McpackConverter.normalize(def.get("textures").getAsString());
                            texPaths.add(p);
                        } else if (def.get("textures").isJsonArray()) {
                            for (JsonElement el : def.getAsJsonArray("textures")) {
                                if (!el.isJsonPrimitive()) continue;
                                texPaths.add(McpackConverter.normalize(el.getAsString()));
                            }
                        }
                    }
                    for (String p : texPaths) {
                        String lower = p.toLowerCase();
                        if (!(lower.endsWith(".png") || lower.endsWith(".tga") || lower.endsWith(".jpg") || lower.endsWith(".jpeg"))) {
                            McpackConverter.putAnimHint(hints, lower + ".png", frameTime);
                            McpackConverter.putAnimHint(hints, lower + ".tga", frameTime);
                            continue;
                        }
                        McpackConverter.putAnimHint(hints, lower, frameTime);
                    }
                }
            }
        }
        catch (Exception ex) {
            logger.debug("Ignoring animation json {}: {}", (Object)path, (Object)ex.toString());
        }
    }

    private static void putAnimHint(Map<String, Integer> hints, String key, int frameTime) {
        hints.putIfAbsent(McpackConverter.normalize(key), frameTime);
    }

    private static int safeInt(JsonElement el, int def) {
        try {
            return el.getAsInt();
        }
        catch (Exception ignore) {
            return def;
        }
    }

    private static void readFlipbookHints(ZipFile inZip, Logger logger, String path, Map<String, Integer> hints) {
        try {
            ZipEntry e = inZip.getEntry(path);
            if (e == null || e.isDirectory()) {
                return;
            }
            try (InputStream is = inZip.getInputStream(e);){
                JsonObject root = JsonParser.parseReader((Reader)new InputStreamReader(is, StandardCharsets.UTF_8)).getAsJsonObject();
                if (!root.has("flipbook_textures") || !root.get("flipbook_textures").isJsonArray()) {
                    return;
                }
                for (JsonElement el : root.getAsJsonArray("flipbook_textures")) {
                    JsonObject obj;
                    if (!el.isJsonObject() || !(obj = el.getAsJsonObject()).has("texture")) continue;
                    String tex = McpackConverter.normalize(obj.get("texture").getAsString());
                    int tpf = 2;
                    if (obj.has("ticks_per_frame")) {
                        tpf = McpackConverter.safeInt(obj.get("ticks_per_frame"), 2);
                    }
                    String base = tex.toLowerCase();
                    McpackConverter.putAnimHint(hints, base + ".png", tpf);
                    McpackConverter.putAnimHint(hints, base + ".tga", tpf);
                }
            }
        }
        catch (Exception ex) {
            logger.debug("Ignoring flipbook json {}: {}", (Object)path, (Object)ex.toString());
        }
    }

    private static String mapBedrockPathToJava(String fullPath) {
        String file;
        String fileName;
        String base;
        String rel = fullPath.substring("textures/".length());
        String ext = McpackConverter.getExtension(fullPath);
        if (rel.startsWith("ui/")) {
            String file2 = rel.substring("ui/".length());
            String fileName2 = McpackConverter.filenameOnly(file2).toLowerCase();
            String base2 = McpackConverter.stripExtension(fileName2);
            if (base2.contains("crosshair")) {
                return "assets/minecraft/textures/gui/sprites/crosshair/crosshair" + ext;
            }
            if (base2.contains("hotbar")) {
                return "assets/minecraft/textures/gui/sprites/hotbar/hotbar" + ext;
            }
            return null;
        }
        if (rel.startsWith("gui/")) {
            String file3 = rel.substring("gui/".length());
            String fileName3 = McpackConverter.filenameOnly(file3).toLowerCase();
            String base3 = McpackConverter.stripExtension(fileName3);
            if (base3.contains("crosshair") || base3.equals("crosshair")) {
                return "assets/minecraft/textures/gui/sprites/crosshair/crosshair" + ext;
            }
            if (base3.contains("hotbar") || base3.equals("gui") || base3.equals("gui2")) {
                return "assets/minecraft/textures/gui/sprites/hotbar/hotbar" + ext;
            }
            if (base3.equals("icons")) {
                return "assets/minecraft/textures/gui/icons" + ext;
            }
            if (base3.equals("title") || base3.contains("logo")) {
                return "assets/minecraft/textures/gui/title/minecraft" + ext;
            }
            return "assets/minecraft/textures/gui/" + fileName3;
        }
        if (rel.startsWith("particles/")) {
            String file4 = rel.substring("particles/".length());
            String fileName4 = McpackConverter.filenameOnly(file4).toLowerCase();
            return "assets/minecraft/textures/particle/" + fileName4;
        }
        if (rel.startsWith("environment/overworld_cubemap/")) {
            String file5 = rel.substring("environment/overworld_cubemap/".length());
            String faceBase = McpackConverter.stripExtension(McpackConverter.filenameOnly(file5)).toLowerCase();
            int idx = McpackConverter.mapPanoramaIndex(faceBase);
            if (idx >= 0) {
                return "assets/minecraft/textures/gui/title/background/panorama_" + idx + ext;
            }
            return "assets/minecraft/textures/" + rel;
        }
        if (rel.startsWith("environment/")) {
            String fileName5 = McpackConverter.filenameOnly(rel).toLowerCase();
            String baseName = McpackConverter.stripExtension(fileName5);
            if (baseName.equals("grass") || baseName.equals("grass_color") || baseName.equals("grasscolor")) {
                return "assets/minecraft/textures/colormap/grass" + ext;
            }
            if (baseName.equals("foliage") || baseName.equals("leaves_color") || baseName.equals("leavescolor")) {
                return "assets/minecraft/textures/colormap/foliage" + ext;
            }
            return "assets/minecraft/textures/" + rel;
        }
        if (rel.startsWith("colormap/")) {
            String fileName6 = McpackConverter.filenameOnly(rel).toLowerCase();
            String baseName = McpackConverter.stripExtension(fileName6);
            if (baseName.equals("grass") || baseName.equals("grass_color") || baseName.equals("grasscolor")) {
                return "assets/minecraft/textures/colormap/grass" + ext;
            }
            if (baseName.equals("foliage") || baseName.equals("leaves_color") || baseName.equals("leavescolor")) {
                return "assets/minecraft/textures/colormap/foliage" + ext;
            }
            return "assets/minecraft/textures/colormap/" + fileName6;
        }
        if (rel.startsWith("misc/") && ((base = McpackConverter.stripExtension(fileName = McpackConverter.filenameOnly(rel).toLowerCase())).contains("enchanted_glint") || base.equals("glint") || base.contains("enchant_glint") || base.contains("enchanted_item_glint"))) {
            return "assets/minecraft/textures/misc/enchanted_glint_item" + ext;
        }
        if (rel.startsWith("models/armor/") || rel.startsWith("armor/") || rel.startsWith("armour/")) {
            file = rel.startsWith("models/armor/") ? rel.substring("models/armor/".length()) : (rel.startsWith("armor/") ? rel.substring("armor/".length()) : rel.substring("armour/".length()));
            String fileName7 = McpackConverter.filenameOnly(file).toLowerCase();
            String base4 = McpackConverter.stripExtension(fileName7);
            if ((base4 = base4.replace("_armor_", "_")).endsWith("_layer_1")) {
                String material = base4.substring(0, base4.length() - "_layer_1".length());
                return "assets/minecraft/textures/models/armor/" + material + "_layer_1" + ext;
            }
            if (base4.endsWith("_layer_2")) {
                String material = base4.substring(0, base4.length() - "_layer_2".length());
                return "assets/minecraft/textures/models/armor/" + material + "_layer_2" + ext;
            }
            if (base4.endsWith("_1_overlay")) {
                String material = base4.substring(0, base4.length() - "_1_overlay".length());
                material = McpackConverter.normalizeArmorMaterial(material);
                return "assets/minecraft/textures/models/armor/" + material + "_layer_1_overlay" + ext;
            }
            if (base4.endsWith("_2_overlay")) {
                String material = base4.substring(0, base4.length() - "_2_overlay".length());
                material = McpackConverter.normalizeArmorMaterial(material);
                return "assets/minecraft/textures/models/armor/" + material + "_layer_2_overlay" + ext;
            }
            if (base4.endsWith("_1")) {
                String material = base4.substring(0, base4.length() - 2);
                material = McpackConverter.normalizeArmorMaterial(material);
                return "assets/minecraft/textures/models/armor/" + material + "_layer_1" + ext;
            }
            if (base4.endsWith("_2")) {
                String material = base4.substring(0, base4.length() - 2);
                material = McpackConverter.normalizeArmorMaterial(material);
                return "assets/minecraft/textures/models/armor/" + material + "_layer_2" + ext;
            }
            return "assets/minecraft/textures/models/armor/" + fileName7;
        }
        if (rel.startsWith("items/") || rel.startsWith("items-opaque/") || rel.startsWith("item/")) {
            String color;
            file = rel.startsWith("items/") ? rel.substring("items/".length()) : (rel.startsWith("items-opaque/") ? rel.substring("items-opaque/".length()) : rel.substring("item/".length()));
            base = McpackConverter.stripExtension(McpackConverter.filenameOnly(file)).toLowerCase();
            if (base.startsWith("bed_") && base.length() > 4 && !(color = base.substring(4)).contains("_")) {
                return "assets/minecraft/textures/item/" + color + "_bed" + ext;
            }
            if (base.endsWith("_bed") && base.length() > 4 && !(color = base.substring(0, base.length() - 4)).contains("_")) {
                return "assets/minecraft/textures/item/" + color + "_bed" + ext;
            }
            if (base.equals("potion_overlay") || base.equals("potion_bottle_overlay") || base.startsWith("potion_overlay")) {
                return "assets/minecraft/textures/item/potion_overlay" + ext;
            }
            if (base.startsWith("potion_bottle_") && (base.endsWith("_overlay") || base.contains("_overlay_"))) {
                return "assets/minecraft/textures/item/potion_overlay" + ext;
            }
            if (base.equals("potion_bottle_empty")) {
                return "assets/minecraft/textures/item/glass_bottle" + ext;
            }
            if (base.equals("potion_bottle_drinkable")) {
                return "assets/minecraft/textures/item/potion" + ext;
            }
            if (base.equals("potion_bottle_splash")) {
                return "assets/minecraft/textures/item/splash_potion" + ext;
            }
            if (base.equals("potion_bottle_lingering")) {
                return "assets/minecraft/textures/item/lingering_potion" + ext;
            }
            if (base.startsWith("potion_bottle_")) {
                String rest = base.substring("potion_bottle_".length());
                if (rest.startsWith("splash_")) {
                    return "assets/minecraft/textures/item/splash_potion" + ext;
                }
                if (rest.startsWith("lingering_")) {
                    return "assets/minecraft/textures/item/lingering_potion" + ext;
                }
                if (rest.startsWith("drinkable_")) {
                    return "assets/minecraft/textures/item/potion" + ext;
                }
                return "assets/minecraft/textures/item/potion" + ext;
            }
            if (base.equals("bow_standby")) {
                return "assets/minecraft/textures/item/bow" + ext;
            }
            if (base.startsWith("bow_pulling_")) {
                return "assets/minecraft/textures/item/" + base + ext;
            }
            String mappedName = McpackConverter.mapItemName(base);
            return "assets/minecraft/textures/item/" + mappedName + ext;
        }
        if (rel.startsWith("blocks/")) {
            String color;
            file = rel.substring("blocks/".length());
            base = McpackConverter.stripExtension(McpackConverter.filenameOnly(file)).toLowerCase();
            if (base.equals("fire_0") || base.equals("fire_layer_0")) {
                return "assets/minecraft/textures/block/fire_0" + ext;
            }
            if (base.equals("fire_1") || base.equals("fire_layer_1")) {
                return "assets/minecraft/textures/block/fire_1" + ext;
            }
            if (base.startsWith("bed_") && !(color = base.substring(4)).contains("_")) {
                return "assets/minecraft/textures/entity/bed/" + color + ext;
            }
            if (base.endsWith("_bed") && !(color = base.substring(0, base.length() - 4)).contains("_")) {
                return "assets/minecraft/textures/entity/bed/" + color + ext;
            }
            if (base.contains("chest")) {
                if (base.contains("ender")) {
                    return "assets/minecraft/textures/entity/chest/ender" + ext;
                }
                String variant = "normal";
                if (base.contains("trapped") || base.contains("trap")) {
                    variant = "trapped";
                } else if (base.contains("christmas") || base.contains("xmas") || base.contains("holiday")) {
                    variant = "christmas";
                }
                boolean left = base.contains("left");
                boolean right = base.contains("right");
                if (left) {
                    return "assets/minecraft/textures/entity/chest/" + variant + "_left" + ext;
                }
                if (right) {
                    return "assets/minecraft/textures/entity/chest/" + variant + "_right" + ext;
                }
                return "assets/minecraft/textures/entity/chest/" + variant + ext;
            }
            if (base.equals("grass_side")) {
                return "assets/minecraft/textures/block/grass_block_side" + ext;
            }
            if (base.equals("grass_top")) {
                return "assets/minecraft/textures/block/grass_block_top" + ext;
            }
            if (base.startsWith("log_")) {
                String core = base.substring(4);
                if (core.endsWith("_side")) {
                    String wood = core.substring(0, core.length() - 5);
                    return "assets/minecraft/textures/block/" + wood + "_log" + ext;
                }
                if (core.endsWith("_top") || core.endsWith("_bottom")) {
                    String wood = core.substring(0, core.lastIndexOf(95));
                    return "assets/minecraft/textures/block/" + wood + "_log_top" + ext;
                }
            }
            String mappedName = McpackConverter.mapBlockName(base);
            return "assets/minecraft/textures/block/" + mappedName + ext;
        }
        if (rel.startsWith("entity/chest/")) {
            boolean dbl;
            file = rel.substring("entity/chest/".length());
            base = McpackConverter.stripExtension(McpackConverter.filenameOnly(file)).toLowerCase();
            if (base.contains("ender")) {
                return "assets/minecraft/textures/entity/chest/ender" + ext;
            }
            String variant = "normal";
            if (base.contains("trapped") || base.contains("trap")) {
                variant = "trapped";
            } else if (base.contains("christmas") || base.contains("xmas") || base.contains("holiday")) {
                variant = "christmas";
            }
            boolean left = base.contains("left");
            boolean right = base.contains("right");
            boolean bl = dbl = base.contains("double") || base.contains("large");
            if (left) {
                return "assets/minecraft/textures/entity/chest/" + variant + "_left" + ext;
            }
            if (right) {
                return "assets/minecraft/textures/entity/chest/" + variant + "_right" + ext;
            }
            if (dbl) {
                return "assets/minecraft/textures/entity/chest/" + variant + "_left" + ext;
            }
            return "assets/minecraft/textures/entity/chest/" + variant + ext;
        }
        if (rel.startsWith("entity/bed/")) {
            return "assets/minecraft/textures/" + rel;
        }
        return "assets/minecraft/textures/" + rel;
    }

    private static int mapPanoramaIndex(String faceBase) {
        String s = faceBase.toLowerCase();
        if (s.equals("right") || s.equals("px") || s.equals("posx") || s.equals("positive_x")) {
            return 0;
        }
        if (s.equals("left") || s.equals("nx") || s.equals("negx") || s.equals("neg_x") || s.equals("negative_x")) {
            return 1;
        }
        if (s.equals("up") || s.equals("top") || s.equals("py") || s.equals("posy") || s.equals("positive_y")) {
            return 2;
        }
        if (s.equals("down") || s.equals("bottom") || s.equals("ny") || s.equals("negy") || s.equals("neg_y") || s.equals("negative_y")) {
            return 3;
        }
        if (s.equals("front") || s.equals("pz") || s.equals("posz") || s.equals("positive_z")) {
            return 4;
        }
        if (s.equals("back") || s.equals("nz") || s.equals("negz") || s.equals("neg_z") || s.equals("negative_z")) {
            return 5;
        }
        if (s.equals("east")) {
            return 0;
        }
        if (s.equals("west")) {
            return 1;
        }
        if (s.equals("north")) {
            return 4;
        }
        if (s.equals("south")) {
            return 5;
        }
        return -1;
    }

    private static String getExtension(String path) {
        int dot = path.lastIndexOf(46);
        if (dot == -1) {
            return "";
        }
        return path.substring(dot);
    }

    private static String mapFontPathToJava(String fullPath) {
        String rel = fullPath.substring("font/".length());
        String fileName = McpackConverter.filenameOnly(rel).toLowerCase();
        String base = McpackConverter.stripExtension(fileName);
        String ext = McpackConverter.getExtension(fileName);
        if (base.equals("default8") || base.equals("default")) {
            return "assets/minecraft/textures/font/default" + ext;
        }
        if (base.startsWith("glyph_")) {
            String page = base.substring("glyph_".length()).toLowerCase();
            return "assets/minecraft/textures/font/unicode_page_" + page + ext;
        }
        return "assets/minecraft/textures/font/" + fileName;
    }

    private static String mapUiPathToJava(String fullPath) {
        String rel = fullPath.substring("ui/".length());
        String fileName = McpackConverter.filenameOnly(rel).toLowerCase();
        String base = McpackConverter.stripExtension(fileName);
        String ext = McpackConverter.getExtension(fileName);
        if (base.contains("crosshair") || base.contains("cross_hair")) {
            return "assets/minecraft/textures/gui/sprites/crosshair/crosshair" + ext;
        }
        if (base.contains("hotbar")) {
            return "assets/minecraft/textures/gui/sprites/hotbar/hotbar" + ext;
        }
        return null;
    }

    private static String mapTextsPathToJava(String fullPath) {
        String rel = fullPath.substring("texts/".length());
        int slash = rel.indexOf(47);
        if (slash == -1) {
            return null;
        }
        String afterLocale = rel.substring(slash + 1);
        if (!afterLocale.startsWith("font/")) {
            return null;
        }
        String fileName = McpackConverter.filenameOnly(afterLocale.substring("font/".length())).toLowerCase();
        String base = McpackConverter.stripExtension(fileName);
        String ext = McpackConverter.getExtension(fileName);
        if (base.equals("default8") || base.equals("default")) {
            return "assets/minecraft/textures/font/default" + ext;
        }
        if (base.startsWith("glyph_")) {
            String page = base.substring("glyph_".length()).toLowerCase();
            return "assets/minecraft/textures/font/unicode_page_" + page + ext;
        }
        return "assets/minecraft/textures/font/" + fileName;
    }

    private static void safeWrite(ZipOutputStream out, String outPath, ZipFile inZip, ZipEntry source, Set<String> writtenPaths) {
        try {
            if (McpackConverter.registerAndCheck(writtenPaths, outPath)) {
                McpackConverter.writeZipEntry(out, outPath, inZip, source);
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private static boolean registerAndCheck(Set<String> writtenPaths, String path) {
        String norm = McpackConverter.normalize(path);
        return writtenPaths.add(norm);
    }

    private static String filenameOnly(String path) {
        int idx = path.lastIndexOf(47);
        return idx == -1 ? path : path.substring(idx + 1);
    }

    private static String mapItemName(String base) {
        String mapped;
        String norm = base.replace('-', '_');
        if (norm.startsWith("item_")) {
            norm = norm.substring(5);
        }
        if (norm.startsWith("icon_")) {
            norm = norm.substring(5);
        }
        if (norm.endsWith("_item")) {
            norm = norm.substring(0, norm.length() - 5);
        }
        if ((mapped = ITEM_NAME_MAP.get(norm)) != null) {
            return mapped;
        }
        if (norm.startsWith("wood_")) {
            return "wooden_" + norm.substring("wood_".length());
        }
        String gen = McpackConverter.mapGenericTool(norm);
        if (gen != null) {
            return gen;
        }
        return norm;
    }

    private static String mapGenericTool(String norm) {
        if (!norm.contains("_")) {
            return null;
        }
        String[] parts = norm.split("_");
        if (parts.length < 2) {
            return null;
        }
        String a = parts[0];
        String b = parts[1];
        String tool = null;
        String mat = null;
        if (McpackConverter.isToolToken(a) && McpackConverter.isMaterialToken(b)) {
            tool = a;
            mat = b;
        } else if (McpackConverter.isMaterialToken(a) && McpackConverter.isToolToken(b)) {
            tool = b;
            mat = a;
        }
        if (tool == null) {
            return null;
        }
        mat = McpackConverter.normalizeItemMaterial(mat);
        return mat + "_" + tool;
    }

    private static boolean isToolToken(String s) {
        return s.equals("sword") || s.equals("pickaxe") || s.equals("axe") || s.equals("shovel") || s.equals("hoe");
    }

    private static boolean isMaterialToken(String s) {
        return s.equals("wood") || s.equals("wooden") || s.equals("stone") || s.equals("iron") || s.equals("gold") || s.equals("golden") || s.equals("diamond") || s.equals("netherite");
    }

    private static String normalizeItemMaterial(String s) {
        if (s.equals("wood")) {
            return "wooden";
        }
        if (s.equals("gold")) {
            return "golden";
        }
        return s;
    }

    private static String normalizeArmorMaterial(String s) {
        if (s.equals("golden")) {
            return "gold";
        }
        if (s.equals("chain")) {
            return "chainmail";
        }
        return s;
    }

    private static String mapBlockName(String base) {
        String mapped = BLOCK_NAME_MAP.get(base);
        if (mapped != null) {
            return mapped;
        }
        int us = base.indexOf(95);
        if (us > 0 && us < base.length() - 1) {
            String a = base.substring(0, us);
            String b = base.substring(us + 1);
            if (McpackConverter.isBlockTypeToken(a) && McpackConverter.isWoodSetToken(b)) {
                return b + "_" + a;
            }
        }
        return base;
    }

    private static boolean isBlockTypeToken(String s) {
        return s.equals("planks") || s.equals("log") || s.equals("leaves") || s.equals("wood") || s.equals("door") || s.equals("trapdoor") || s.equals("fence") || s.equals("fence_gate") || s.equals("stairs") || s.equals("slab") || s.equals("button") || s.equals("pressure_plate");
    }

    private static boolean isWoodSetToken(String s) {
        return s.equals("oak") || s.equals("spruce") || s.equals("birch") || s.equals("jungle") || s.equals("acacia") || s.equals("dark_oak") || s.equals("mangrove") || s.equals("cherry") || s.equals("bamboo") || s.equals("crimson") || s.equals("warped");
    }

    private static Map<String, String> createItemNameMap() {
        HashMap<String, String> m = new HashMap<String, String>();
        m.put("slimeball", "slime_ball");
        m.put("apple_golden", "golden_apple");
        m.put("apple_golden_enchanted", "enchanted_golden_apple");
        m.put("enchanted_golden_apple", "enchanted_golden_apple");
        m.put("fireball", "fire_charge");
        m.put("fireworks_charge", "firework_star");
        m.put("wood_sword", "wooden_sword");
        m.put("wood_pickaxe", "wooden_pickaxe");
        m.put("wood_axe", "wooden_axe");
        m.put("wood_shovel", "wooden_shovel");
        m.put("wood_hoe", "wooden_hoe");
        m.put("wooden_sword", "wooden_sword");
        m.put("wooden_pickaxe", "wooden_pickaxe");
        m.put("wooden_axe", "wooden_axe");
        m.put("wooden_shovel", "wooden_shovel");
        m.put("wooden_hoe", "wooden_hoe");
        m.put("gold_sword", "golden_sword");
        m.put("sword_gold", "golden_sword");
        m.put("sword_wood", "wooden_sword");
        m.put("sword_stone", "stone_sword");
        m.put("sword_iron", "iron_sword");
        m.put("sword_diamond", "diamond_sword");
        m.put("sword_netherite", "netherite_sword");
        m.put("totem", "totem_of_undying");
        m.put("totem_of_undying", "totem_of_undying");
        m.put("totem_undying", "totem_of_undying");
        m.put("undying_totem", "totem_of_undying");
        return m;
    }

    private static Map<String, String> createBlockNameMap() {
        HashMap<String, String> m = new HashMap<String, String>();
        m.put("stonebrick", "stone_bricks");
        m.put("stonebrick_carved", "chiseled_stone_bricks");
        m.put("stonebrick_cracked", "cracked_stone_bricks");
        m.put("grass_side", "grass_block_side");
        m.put("grass_top", "grass_block_top");
        m.put("furnace_on", "furnace_front_on");
        m.put("furnace_front_lit", "furnace_front_on");
        m.put("lit_furnace_front", "furnace_front_on");
        m.put("furnace_back", "furnace_side");
        m.put("bookshelve", "bookshelf");
        m.put("bookshelves", "bookshelf");
        m.put("planks_oak", "oak_planks");
        m.put("planks_spruce", "spruce_planks");
        m.put("planks_birch", "birch_planks");
        m.put("planks_jungle", "jungle_planks");
        m.put("planks_acacia", "acacia_planks");
        m.put("planks_dark_oak", "dark_oak_planks");
        m.put("log_oak", "oak_log");
        m.put("log_spruce", "spruce_log");
        m.put("log_birch", "birch_log");
        m.put("log_jungle", "jungle_log");
        m.put("log_acacia", "acacia_log");
        m.put("log_dark_oak", "dark_oak_log");
        m.put("leaves_oak", "oak_leaves");
        m.put("leaves_spruce", "spruce_leaves");
        m.put("leaves_birch", "birch_leaves");
        m.put("leaves_jungle", "jungle_leaves");
        m.put("leaves_acacia", "acacia_leaves");
        m.put("leaves_dark_oak", "dark_oak_leaves");
        m.put("reeds", "sugar_cane");
        m.put("sugarcane", "sugar_cane");
        m.put("sugar_cane_top", "sugar_cane");
        m.put("cane", "sugar_cane");
        return m;
    }

    private static void writeZipEntry(ZipOutputStream out, String outPath, ZipFile inZip, ZipEntry source) throws IOException {
        String normalized = McpackConverter.normalize(outPath);
        McpackConverter.ensureDirs(out, normalized);
        ZipEntry outEntry = new ZipEntry(normalized);
        out.putNextEntry(outEntry);
        try (BufferedInputStream is = new BufferedInputStream(inZip.getInputStream(source));){
            ((InputStream)is).transferTo(out);
        }
        out.closeEntry();
    }

    private static void writeStringEntry(ZipOutputStream out, String outPath, String content) throws IOException {
        String normalized = McpackConverter.normalize(outPath);
        McpackConverter.ensureDirs(out, normalized);
        ZipEntry outEntry = new ZipEntry(normalized);
        out.putNextEntry(outEntry);
        out.write(content.getBytes(StandardCharsets.UTF_8));
        out.closeEntry();
    }

    private static void writeZipEntryBytes(ZipOutputStream out, String outPath, byte[] data) throws IOException {
        String normalized = McpackConverter.normalize(outPath);
        McpackConverter.ensureDirs(out, normalized);
        ZipEntry outEntry = new ZipEntry(normalized);
        out.putNextEntry(outEntry);
        out.write(data);
        out.closeEntry();
    }

    private static void ensureDirs(ZipOutputStream out, String path) throws IOException {
        int idx = 0;
        while ((idx = path.indexOf(47, idx)) != -1) {
            String dir = path.substring(0, idx + 1);
            if (!dir.isEmpty()) {
                ZipEntry dirEntry = new ZipEntry(dir);
                try {
                    out.putNextEntry(dirEntry);
                    out.closeEntry();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
            ++idx;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static PackMeta readBedrockManifest(ZipFile inZip, Logger logger) {
        try {
            ZipEntry manifest = inZip.getEntry("manifest.json");
            if (manifest == null) return PackMeta.empty();
            if (manifest.isDirectory()) {
                return PackMeta.empty();
            }
            try (InputStream is = inZip.getInputStream(manifest);){
                JsonObject header;
                JsonObject root = JsonParser.parseReader((Reader)new InputStreamReader(is, StandardCharsets.UTF_8)).getAsJsonObject();
                JsonObject jsonObject = header = root.has("header") && root.get("header").isJsonObject() ? root.getAsJsonObject("header") : null;
                if (header == null) {
                    PackMeta packMeta2 = PackMeta.empty();
                    return packMeta2;
                }
                String name = header.has("name") ? header.get("name").getAsString() : null;
                String description = header.has("description") ? header.get("description").getAsString() : null;
                PackMeta packMeta = new PackMeta(name, description);
                return packMeta;
            }
        }
        catch (Exception e) {
            logger.warn("Unable to read Bedrock manifest.json: {}", (Object)e.toString());
            return PackMeta.empty();
        }
    }

    private static String buildDescription(PackMeta meta) {
        StringBuilder sb = new StringBuilder();
        sb.append("Converted Bedrock pack");
        if (meta.name != null && !meta.name.isBlank()) {
            sb.append(": ").append(meta.name);
        }
        if (meta.description != null && !meta.description.isBlank()) {
            sb.append(" - ").append(meta.description);
        }
        return sb.toString();
    }

    private static String buildPackMcmeta(int packFormat, String description) {
        JsonObject pack = new JsonObject();
        pack.addProperty("pack_format", (Number)packFormat);
        pack.addProperty("description", description);
        JsonObject root = new JsonObject();
        root.add("pack", (JsonElement)pack);
        return GSON.toJson((JsonElement)root);
    }

    private static String sanitize(String name) {
        String n = name.replaceAll("[^a-zA-Z0-9._-]+", "-");
        n = n.replaceAll("-+", "-");
        while (n.startsWith(".")) {
            n = n.substring(1);
        }
        if (n.isEmpty()) {
            n = "pack";
        }
        return n;
    }

    private static String normalize(String path) {
        return path.replace('\\', '/');
    }

    private static String stripExtension(String fileName) {
        int i = fileName.lastIndexOf(46);
        return i == -1 ? fileName : fileName.substring(0, i);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static boolean isLikelyZip(Path file) {
        try (InputStream is = Files.newInputStream(file, new OpenOption[0]);){
            byte[] sig = is.readNBytes(4);
            if (sig.length < 2) {
                boolean bl2 = false;
                return bl2;
            }
            boolean bl = sig[0] == 80 && sig[1] == 75;
            return bl;
        }
        catch (IOException e) {
            return false;
        }
    }

    @Environment(value=EnvType.CLIENT)
    private static class TgaImage {
        int width;
        int height;
        boolean hasAlpha;
        int[] argb;

        private TgaImage() {
        }
    }

    @Environment(value=EnvType.CLIENT)
    private record PackMeta(String name, String description) {
        static PackMeta empty() {
            return new PackMeta(null, null);
        }
    }
}

