package dev.overgrown.sync.factory.power.type;

import dev.overgrown.sync.Sync;
import io.github.apace100.apoli.component.PowerHolderComponent;
import io.github.apace100.apoli.data.ApoliDataTypes;
import io.github.apace100.apoli.power.Power;
import io.github.apace100.apoli.power.PowerType;
import io.github.apace100.apoli.power.factory.PowerFactory;
import io.github.apace100.calio.data.SerializableData;
import io.github.apace100.calio.data.SerializableDataTypes;
import net.minecraft.class_1297;
import net.minecraft.class_1309;
import net.minecraft.class_1657;
import net.minecraft.class_2487;
import net.minecraft.class_2495;
import net.minecraft.class_2499;
import net.minecraft.class_2512;
import net.minecraft.class_2520;
import net.minecraft.class_3218;
import net.minecraft.class_3545;
import net.minecraft.nbt.*;
import net.minecraft.server.MinecraftServer;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.function.Predicate;

public class EntitySetPower extends Power {

    private final Consumer<class_3545<class_1297, class_1297>> actionOnAdd;
    private final Consumer<class_3545<class_1297, class_1297>> actionOnRemove;
    private final int tickRate;

    private final Set<UUID> entityUuids = new HashSet<>();
    private final Map<UUID, class_1297> entities = new HashMap<>();

    private final Set<UUID> tempUuids = new HashSet<>();
    private final Map<UUID, Long> tempEntities = new ConcurrentHashMap<>();

    private Integer startTicks = null;

    private boolean wasActive = false;
    private boolean removedTemps = false;

    public EntitySetPower(PowerType<?> type, class_1309 entity, Consumer<class_3545<class_1297, class_1297>> actionOnAdd, Consumer<class_3545<class_1297, class_1297>> actionOnRemove, int tickRate) {
        super(type, entity);
        if (tickRate <= 0) {
            throw new IllegalArgumentException("Tick rate must be a positive integer");
        }
        this.actionOnAdd = actionOnAdd;
        this.actionOnRemove = actionOnRemove;
        this.tickRate = tickRate;
        this.setTicking(true);
    }

    @Override
    public void onAdded() {
        removedTemps = entityUuids.removeIf(tempUuids::contains);
        tempUuids.clear();
    }

    @Override
    public void tick() {
        if (removedTemps) {
            this.removedTemps = false;
            PowerHolderComponent.syncPower(this.entity, this.type);
            return;
        }

        if (!tempEntities.isEmpty() && this.isActive()) {
            if (startTicks == null) {
                this.startTicks = entity.field_6012 % tickRate;
                return;
            }

            if (entity.field_6012 % tickRate == startTicks) {
                this.tickTempEntities();
            }
            this.wasActive = true;
        } else if (wasActive) {
            this.startTicks = null;
            this.wasActive = false;
        }
    }

    protected void tickTempEntities() {
        Iterator<Map.Entry<UUID, Long>> entryIterator = tempEntities.entrySet().iterator();
        long time = entity.method_37908().method_8510();

        while (entryIterator.hasNext()) {
            Map.Entry<UUID, Long> entry = entryIterator.next();
            if (time < entry.getValue()) {
                continue;
            }

            UUID uuid = entry.getKey();
            class_1297 tempEntity = this.getEntity(uuid);

            entryIterator.remove();
            if (entityUuids.remove(uuid) | entities.remove(uuid) != null | tempUuids.remove(uuid)) {
                if (actionOnRemove != null) {
                    actionOnRemove.accept(new class_3545<>(entity, tempEntity));
                }
                this.removedTemps = true;
            }
        }
    }

    public boolean validateEntities() {
        MinecraftServer server = entity.method_5682();
        if (server == null) {
            return false;
        }

        Iterator<UUID> uuidIterator = entityUuids.iterator();
        boolean valid = true;

        while (uuidIterator.hasNext()) {
            UUID uuid = uuidIterator.next();
            if (getEntityFromAllWorlds(server, uuid) != null) {
                continue;
            }

            uuidIterator.remove();
            entities.remove(uuid);
            tempUuids.remove(uuid);
            tempEntities.remove(uuid);
            valid = false;
        }

        return valid;
    }

    @Nullable
    private static class_1297 getEntityFromAllWorlds(MinecraftServer server, UUID uuid) {
        for (class_3218 world : server.method_3738()) {
            class_1297 entity = world.method_14190(uuid);
            if (entity != null) {
                return entity;
            }
        }
        return null;
    }

    public boolean add(class_1297 entity) {
        return add(entity, null);
    }

    public boolean add(class_1297 entity, @Nullable Integer time) {
        if (entity == null || entity.method_31481() || entity.method_37908().field_9236) {
            return false;
        }

        UUID uuid = entity.method_5667();
        boolean addedToSet = false;

        if (time != null) {
            addedToSet |= tempUuids.add(uuid);
            tempEntities.compute(uuid, (prevUuid, prevTime) -> entity.method_37908().method_8510() + time);
        }

        if (!entityUuids.contains(uuid)) {
            addedToSet |= entityUuids.add(uuid);
            entities.put(uuid, entity);
            if (actionOnAdd != null) {
                actionOnAdd.accept(new class_3545<>(this.entity, entity));
            }
        }

        return addedToSet;
    }

    public boolean remove(@Nullable class_1297 entity) {
        return remove(entity, true);
    }

    public boolean remove(@Nullable class_1297 entity, boolean executeRemoveAction) {
        if (entity == null || entity.method_37908().field_9236) {
            return false;
        }

        UUID uuid = entity.method_5667();
        boolean result = entityUuids.remove(uuid)
                | entities.remove(uuid) != null
                | tempUuids.remove(uuid)
                | tempEntities.remove(uuid) != null;

        if (executeRemoveAction && result && actionOnRemove != null) {
            actionOnRemove.accept(new class_3545<>(this.entity, entity));
        }

        return result;
    }

    public boolean contains(class_1297 entity) {
        if (entity == null) {
            return false;
        }
        return entities.containsValue(entity) || entityUuids.contains(entity.method_5667());
    }

    public int size() {
        return entityUuids.size();
    }

    public void clear() {
        if (actionOnRemove != null) {
            for (UUID entityUuid : entityUuids) {
                actionOnRemove.accept(new class_3545<>(this.entity, this.getEntity(entityUuid)));
            }
        }

        boolean wasNotEmpty = !entityUuids.isEmpty() || !tempUuids.isEmpty();
        tempUuids.clear();
        tempEntities.clear();
        entityUuids.clear();
        entities.clear();

        if (wasNotEmpty) {
            PowerHolderComponent.syncPower(this.entity, this.type);
        }
    }

    public Set<UUID> getIterationSet() {
        return new HashSet<>(entityUuids);
    }

    @Nullable
    public class_1297 getEntity(UUID uuid) {
        if (!entityUuids.contains(uuid)) {
            return null;
        }

        class_1297 entity = entities.get(uuid);
        if (entity != null && !entity.method_31481()) {
            return entity;
        }

        MinecraftServer server = this.entity.method_5682();
        if (server != null) {
            entity = getEntityFromAllWorlds(server, uuid);
            if (entity != null) {
                entities.put(uuid, entity);
                return entity;
            }
        }

        return null;
    }

    @Override
    public class_2520 toTag() {

        class_2487 rootNbt = new class_2487();

        class_2499 entityUuidsNbt = new class_2499();
        class_2499 tempUuidsNbt = new class_2499();

        for (UUID entityUuid : entityUuids) {
            class_2495 entityUuidNbt = class_2512.method_25929(entityUuid);
            entityUuidsNbt.add(entityUuidNbt);
        }

        for (UUID tempUuid : tempUuids) {
            class_2495 tempUuidNbt = class_2512.method_25929(tempUuid);
            tempUuidsNbt.add(tempUuidNbt);
        }

        rootNbt.method_10566("Entities", entityUuidsNbt);
        rootNbt.method_10566("TempEntities", tempUuidsNbt);
        rootNbt.method_10556("RemovedTemps", removedTemps);

        return rootNbt;

    }

    @Override
    public void fromTag(class_2520 tag) {

        if (!(tag instanceof class_2487 rootNbt)) {
            return;
        }

        tempUuids.clear();
        tempEntities.clear();
        entityUuids.clear();
        entities.clear();

        class_2499 tempUuidsNbt = rootNbt.method_10554("TempEntities", class_2520.field_33261);
        for (class_2520 tempUuidNbt : tempUuidsNbt) {
            UUID tempUuid = class_2512.method_25930(tempUuidNbt);
            tempUuids.add(tempUuid);
        }

        class_2499 entityUuidsNbt = rootNbt.method_10554("Entities", class_2520.field_33261);
        for (class_2520 entityUuidNbt : entityUuidsNbt) {
            UUID entityUuid = class_2512.method_25930(entityUuidNbt);
            entityUuids.add(entityUuid);
        }

        removedTemps = rootNbt.method_10577("RemovedTemps");

    }

    public static void integrateLoadCallback(class_1297 loadedEntity, class_3218 world) {
        PowerHolderComponent.KEY.maybeGet(loadedEntity).ifPresent(component -> component.getPowers(EntitySetPower.class, true).stream()
                .filter(Predicate.not(EntitySetPower::validateEntities))
                .map(Power::getType)
                .forEach(powerType -> PowerHolderComponent.syncPower(loadedEntity, powerType)));
    }

    public static void integrateUnloadCallback(class_1297 unloadedEntity, class_3218 world) {

        class_1297.class_5529 removalReason = unloadedEntity.method_35049();
        if (removalReason == null || !removalReason.method_31486() || unloadedEntity instanceof class_1657) {
            return;
        }

        for (class_3218 otherWorld : world.method_8503().method_3738()) {

            for (class_1297 entity : otherWorld.method_27909()) {

                PowerHolderComponent.KEY.maybeGet(entity).ifPresent(component -> component.getPowers(EntitySetPower.class, true).stream()
                        .filter(p -> p.remove(unloadedEntity, false))
                        .map(Power::getType)
                        .forEach(powerType -> PowerHolderComponent.syncPower(entity, powerType)));

            }

        }

    }

    public static PowerFactory<EntitySetPower> getFactory() {
        return new PowerFactory<EntitySetPower>(
                Sync.identifier("entity_set"),
                new SerializableData()
                        .add("action_on_add", ApoliDataTypes.BIENTITY_ACTION, null)
                        .add("action_on_remove", ApoliDataTypes.BIENTITY_ACTION, null)
                        .add("tick_rate", SerializableDataTypes.INT, 1),
                data -> (powerType, livingEntity) -> {
                    int tickRate = data.getInt("tick_rate");
                    return new EntitySetPower(
                            powerType,
                            livingEntity,
                            data.get("action_on_add"),
                            data.get("action_on_remove"),
                            tickRate
                    );
                }
        ).allowCondition();
    }
}