/*
 * Decompiled with CFR 0.152.
 */
package me.koyere.ecoxpert.modules.inflation;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import me.koyere.ecoxpert.EcoXpertPlugin;
import me.koyere.ecoxpert.api.events.WealthTaxAppliedEvent;
import me.koyere.ecoxpert.core.config.ConfigManager;
import me.koyere.ecoxpert.core.education.EducationNotifier;
import me.koyere.ecoxpert.core.translation.TranslationManager;
import me.koyere.ecoxpert.economy.EconomyManager;
import me.koyere.ecoxpert.modules.inflation.EconomicMemory;
import me.koyere.ecoxpert.modules.inflation.EconomicSnapshot;
import me.koyere.ecoxpert.modules.inflation.PlayerEconomicProfile;
import me.koyere.ecoxpert.modules.market.MarketManager;
import org.bukkit.Bukkit;
import org.bukkit.configuration.file.FileConfiguration;
import org.bukkit.entity.Player;
import org.bukkit.event.Event;
import org.bukkit.plugin.Plugin;

public class EconomicIntelligenceEngine {
    private final EcoXpertPlugin plugin;
    private final EconomyManager economyManager;
    private final MarketManager marketManager;
    private EconomicCycle currentCycle = EconomicCycle.STABLE;
    private double economicHealth = 1.0;
    private double inflationRate = 0.0;
    private double velocityOfMoney = 1.0;
    private final Map<UUID, PlayerEconomicProfile> playerProfiles = new ConcurrentHashMap<UUID, PlayerEconomicProfile>();
    private final EconomicMemory economicMemory = new EconomicMemory();
    private double wealthTaxRate = 0.005;
    private double wealthTaxThresholdMultiplier = 2.0;
    private double stimulusFactor = 0.02;
    private double cooldownFactor = 0.02;
    private int interventionMinutes = 10;
    private double biasMax = 0.03;

    public EconomicIntelligenceEngine(EcoXpertPlugin plugin, EconomyManager economyManager, MarketManager marketManager, ConfigManager configManager) {
        this.plugin = plugin;
        this.economyManager = economyManager;
        this.marketManager = marketManager;
        try {
            FileConfiguration cfg = configManager.getModuleConfig("inflation");
            this.wealthTaxRate = cfg.getDouble("policy.wealth_tax.rate", this.wealthTaxRate);
            this.wealthTaxThresholdMultiplier = cfg.getDouble("policy.wealth_tax.threshold_multiplier", this.wealthTaxThresholdMultiplier);
            this.stimulusFactor = cfg.getDouble("policy.market.stimulus_factor", this.stimulusFactor);
            this.cooldownFactor = cfg.getDouble("policy.market.cooldown_factor", this.cooldownFactor);
            this.interventionMinutes = cfg.getInt("policy.intervention.minutes", this.interventionMinutes);
            this.biasMax = cfg.getDouble("policy.market.bias_max", this.biasMax);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    public CompletableFuture<Void> initialize() {
        return CompletableFuture.runAsync(() -> {
            this.plugin.getLogger().info("\ud83e\udde0 Initializing Economic Intelligence Engine...");
            this.economicMemory.loadHistoricalData();
            this.analyzeCurrentEconomicState().thenRun(() -> {
                this.startEconomicHeartbeat();
                this.startPlayerBehaviorAnalysis();
                this.plugin.getLogger().info("\u2705 Economic Intelligence Engine active");
                this.plugin.getLogger().info("\ud83d\udcca Current Cycle: " + String.valueOf((Object)this.currentCycle) + " | Health: " + String.format("%.1f%%", this.economicHealth * 100.0));
            });
        });
    }

    private void startEconomicHeartbeat() {
        Bukkit.getScheduler().runTaskTimerAsynchronously((Plugin)this.plugin, () -> {
            try {
                EconomicSnapshot snapshot = this.captureEconomicSnapshot();
                this.updateEconomicHealth(snapshot);
                this.evaluateCycleTransition(snapshot);
                this.applyEconomicInterventions(snapshot);
                this.economicMemory.recordSnapshot(snapshot);
                this.logEconomicStatus(snapshot);
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Error in Economic Heartbeat: " + e.getMessage());
            }
        }, 6000L, 6000L);
    }

    private EconomicSnapshot captureEconomicSnapshot() {
        double totalMoney = this.calculateTotalMoneySupply();
        double averageBalance = this.calculateAveragePlayerBalance();
        int activeUsers = Bukkit.getOnlinePlayers().size();
        double transactionVolume = this.calculateRecentTransactionVolume();
        double marketActivity = this.calculateMarketActivity();
        return new EconomicSnapshot(LocalDateTime.now(), totalMoney, averageBalance, activeUsers, transactionVolume, marketActivity, this.currentCycle, this.economicHealth, this.inflationRate, this.velocityOfMoney);
    }

    private void evaluateCycleTransition(EconomicSnapshot snapshot) {
        EconomicCycle newCycle = this.currentCycle;
        double healthTrend = this.economicMemory.getHealthTrend();
        double inflationTrend = this.economicMemory.getInflationTrend();
        double activityTrend = this.economicMemory.getActivityTrend();
        if (this.economicHealth > 0.9 && this.inflationRate > 0.04) {
            newCycle = this.getNextCycle(this.currentCycle, false);
            this.plugin.getLogger().info("\ud83d\udd25 Economy overheating detected - initiating cooldown");
        } else if (this.economicHealth < 0.5 && this.inflationRate < -0.02) {
            newCycle = this.getNextCycle(this.currentCycle, true);
            this.plugin.getLogger().info("\u2744\ufe0f Economic crisis detected - initiating stimulus");
        } else if (this.shouldNaturalCycleTransition(snapshot)) {
            boolean shouldGrow = healthTrend > 0.0 && activityTrend > 0.0;
            newCycle = this.getNextCycle(this.currentCycle, shouldGrow);
        }
        if (newCycle != this.currentCycle) {
            this.transitionToNewCycle(newCycle, snapshot);
        }
    }

    private void applyEconomicInterventions(EconomicSnapshot snapshot) {
        if (this.economicHealth < 0.6) {
            this.injectLiquidity(snapshot);
        } else if (this.economicHealth > 0.9 && this.inflationRate > 0.05) {
            this.reduceLiquidity(snapshot);
        }
        if (snapshot.getMarketActivity() < 0.3) {
            this.stimulateMarketActivity();
        }
        this.adjustInterestRates(snapshot);
        this.adjustMarketPricingFactors(snapshot);
    }

    private void injectLiquidity(EconomicSnapshot snapshot) {
        double injectionAmount = this.calculateOptimalInjectionAmount(snapshot);
        this.plugin.getLogger().info("\ud83d\udc89 Injecting liquidity: $" + String.format("%.2f", injectionAmount) + " into economy");
        this.distributeEconomicStimulus(injectionAmount);
    }

    private void distributeEconomicStimulus(double totalAmount) {
        double activePlayerPool = totalAmount * 0.4;
        double marketMakerPool = totalAmount * 0.3;
        double newPlayerPool = totalAmount * 0.2;
        Bukkit.getOnlinePlayers().forEach(player -> {
            PlayerEconomicProfile profile = this.getPlayerProfile(player.getUniqueId());
            double playerStimulus = this.calculatePlayerStimulus(profile, activePlayerPool);
            if (playerStimulus > 0.0) {
                this.economyManager.addMoney(player.getUniqueId(), BigDecimal.valueOf(playerStimulus), "Economic Stimulus - Cycle: " + String.valueOf((Object)this.currentCycle));
            }
        });
    }

    private PlayerEconomicProfile getPlayerProfile(UUID playerId) {
        return this.playerProfiles.computeIfAbsent(playerId, id -> new PlayerEconomicProfile((UUID)id));
    }

    private void startPlayerBehaviorAnalysis() {
        Bukkit.getScheduler().runTaskTimerAsynchronously((Plugin)this.plugin, () -> this.analyzePlayerBehaviorPatterns(), 72000L, 72000L);
    }

    private void analyzePlayerBehaviorPatterns() {
        this.plugin.getLogger().info("\ud83d\udd0d Analyzing player economic behavior patterns...");
        this.playerProfiles.values().forEach(profile -> {
            profile.updateBehaviorMetrics();
            profile.calculateEconomicImpact();
        });
        this.updateVelocityOfMoney();
    }

    private CompletableFuture<Void> analyzeCurrentEconomicState() {
        return CompletableFuture.runAsync(() -> {
            EconomicSnapshot snapshot = this.captureEconomicSnapshot();
            this.economicHealth = this.calculateEconomicHealth(snapshot);
            this.inflationRate = this.calculateCurrentInflationRate(snapshot);
            this.velocityOfMoney = this.calculateVelocityOfMoney(snapshot);
        });
    }

    private double calculateTotalMoneySupply() {
        return (double)Bukkit.getOnlinePlayers().size() * 50000.0;
    }

    private double calculateAveragePlayerBalance() {
        return 25000.0;
    }

    private double calculateRecentTransactionVolume() {
        return 100000.0;
    }

    private double calculateMarketActivity() {
        return 0.6;
    }

    private double calculateEconomicHealth(EconomicSnapshot snapshot) {
        double balanceDistribution = 0.8;
        double transactionHealth = 0.9;
        double marketHealth = snapshot.getMarketActivity();
        return balanceDistribution * 0.4 + transactionHealth * 0.4 + marketHealth * 0.2;
    }

    private double calculateCurrentInflationRate(EconomicSnapshot snapshot) {
        return this.currentCycle.getBaseInflationRate() + snapshot.getTransactionVolume() / 1000000.0 * 0.01;
    }

    private double calculateVelocityOfMoney(EconomicSnapshot snapshot) {
        return snapshot.getTransactionVolume() / snapshot.getTotalMoney();
    }

    private void updateEconomicHealth(EconomicSnapshot snapshot) {
        double newHealth = this.calculateEconomicHealth(snapshot);
        this.economicHealth = this.economicHealth * 0.7 + newHealth * 0.3;
        this.inflationRate = this.calculateCurrentInflationRate(snapshot);
    }

    private boolean shouldNaturalCycleTransition(EconomicSnapshot snapshot) {
        return this.economicMemory.getTimeSinceLastCycleChange() > 3600L;
    }

    private EconomicCycle getNextCycle(EconomicCycle current, boolean shouldGrow) {
        EconomicCycle[] cycles = EconomicCycle.values();
        int currentIndex = current.ordinal();
        if (shouldGrow && currentIndex < cycles.length - 1) {
            return cycles[currentIndex + 1];
        }
        if (!shouldGrow && currentIndex > 0) {
            return cycles[currentIndex - 1];
        }
        return current;
    }

    private void transitionToNewCycle(EconomicCycle newCycle, EconomicSnapshot snapshot) {
        this.plugin.getLogger().info("\ud83d\udd04 Economic cycle transition: " + String.valueOf((Object)this.currentCycle) + " \u2192 " + String.valueOf((Object)newCycle));
        this.currentCycle = newCycle;
        this.economicMemory.recordCycleChange(newCycle, snapshot);
        String message = this.getEconomicCycleMessage(newCycle);
        Bukkit.broadcastMessage((String)("\u00a76[EcoXpert] \u00a7e" + message));
    }

    private String getEconomicCycleMessage(EconomicCycle cycle) {
        return switch (cycle) {
            default -> throw new IncompatibleClassChangeError();
            case EconomicCycle.DEPRESSION -> "Economic depression detected. Government stimulus programs activated.";
            case EconomicCycle.RECESSION -> "Economic slowdown. Consider saving and investing wisely.";
            case EconomicCycle.STABLE -> "Economic conditions stabilized. Normal market activity resumed.";
            case EconomicCycle.GROWTH -> "Economic growth period. Great time for investments and expansion!";
            case EconomicCycle.BOOM -> "Economic boom! High market activity and opportunities.";
            case EconomicCycle.BUBBLE -> "Economic bubble warning! Exercise caution with large investments.";
        };
    }

    private void logEconomicStatus(EconomicSnapshot snapshot) {
        if (this.plugin.getLogger().isLoggable(Level.INFO)) {
            this.plugin.getLogger().info(String.format("\ud83d\udcca Economic Status | Cycle: %s | Health: %.1f%% | Inflation: %.2f%% | Activity: %.1f", new Object[]{this.currentCycle, this.economicHealth * 100.0, this.inflationRate * 100.0, snapshot.getMarketActivity() * 100.0}));
        }
    }

    private void reduceLiquidity(EconomicSnapshot snapshot) {
        try {
            BigDecimal rate = BigDecimal.valueOf(this.wealthTaxRate);
            BigDecimal threshold = BigDecimal.valueOf(snapshot.getAverageBalance() * this.wealthTaxThresholdMultiplier).setScale(2, RoundingMode.HALF_UP);
            String reason = "Overheating liquidity reduction";
            this.economyManager.applyWealthTax(rate, threshold, reason).thenAccept(affected -> {
                try {
                    Bukkit.getScheduler().runTask((Plugin)this.plugin, () -> Bukkit.getPluginManager().callEvent((Event)new WealthTaxAppliedEvent(rate, threshold, (int)affected, reason, Instant.now())));
                }
                catch (Exception exception) {
                    // empty catch block
                }
            });
            try {
                TranslationManager tm = this.plugin.getServiceRegistry().getInstance(TranslationManager.class);
                for (Player op : Bukkit.getOnlinePlayers()) {
                    try {
                        BigDecimal bal = this.economyManager.getBalance(op.getUniqueId()).join();
                        if (bal == null || bal.compareTo(threshold) <= 0) continue;
                        EducationNotifier.notifyWealthTaxApplied(this.plugin, tm, op.getUniqueId(), threshold);
                    }
                    catch (Exception exception) {}
                }
            }
            catch (Exception exception) {
                // empty catch block
            }
            this.marketManager.setGlobalPriceFactors(1.0 + this.cooldownFactor, 1.0 - this.cooldownFactor);
            Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, () -> this.marketManager.setGlobalPriceFactors(1.0, 1.0), 1200L * (long)this.interventionMinutes);
            this.plugin.getLogger().info("Applied liquidity reduction: wealth tax and market factors (10m)");
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("reduceLiquidity failed: " + e.getMessage());
        }
    }

    private void stimulateMarketActivity() {
        try {
            this.marketManager.setGlobalPriceFactors(1.0 - this.stimulusFactor, 1.0 + this.stimulusFactor);
            Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, () -> this.marketManager.setGlobalPriceFactors(1.0, 1.0), 1200L * (long)this.interventionMinutes);
            this.plugin.getLogger().info("Stimulating market activity: adjusted price factors for 10m");
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("stimulateMarketActivity failed: " + e.getMessage());
        }
    }

    private void adjustInterestRates(EconomicSnapshot snapshot) {
        double base = 0.01;
        double adjust = (0.5 - snapshot.getEconomicHealth()) * 0.02;
        double recommended = Math.max(0.0, Math.min(0.05, base + adjust));
        this.plugin.getLogger().info(String.format("Recommended bank interest rate: %.2f%%", recommended * 100.0));
    }

    private void adjustMarketPricingFactors(EconomicSnapshot snapshot) {
        double infl = snapshot.getInflationRate();
        double health = snapshot.getEconomicHealth();
        double buyBias = 1.0 + this.clamp(infl * 0.5 - (health - 0.5) * 0.02, -this.biasMax, this.biasMax);
        double sellBias = 1.0 + this.clamp(-(infl * 0.5) + (health - 0.5) * 0.02, -this.biasMax, this.biasMax);
        this.marketManager.setGlobalPriceFactors(buyBias, sellBias);
    }

    private double clamp(double v, double min, double max) {
        return Math.max(min, Math.min(max, v));
    }

    public String getPolicyInfo() {
        return String.format("Policy{wealth_tax_rate=%.3f, threshold_mult=%.2f, stim=%.3f, cool=%.3f, minutes=%d, bias_max=%.3f}", this.wealthTaxRate, this.wealthTaxThresholdMultiplier, this.stimulusFactor, this.cooldownFactor, this.interventionMinutes, this.biasMax);
    }

    public boolean setPolicyParam(String name, double value) {
        switch (name.toLowerCase()) {
            case "wealth_tax_rate": {
                this.wealthTaxRate = this.clamp(value, 0.0, 0.05);
                break;
            }
            case "wealth_tax_threshold_multiplier": {
                this.wealthTaxThresholdMultiplier = this.clamp(value, 1.0, 10.0);
                break;
            }
            case "stimulus_factor": {
                this.stimulusFactor = this.clamp(value, 0.0, 0.2);
                break;
            }
            case "cooldown_factor": {
                this.cooldownFactor = this.clamp(value, 0.0, 0.2);
                break;
            }
            case "intervention_minutes": {
                this.interventionMinutes = (int)Math.max(1.0, Math.min(120.0, value));
                break;
            }
            case "bias_max": {
                this.biasMax = this.clamp(value, 0.0, 0.2);
                break;
            }
            default: {
                return false;
            }
        }
        return true;
    }

    public void reloadPolicy(ConfigManager configManager) {
        try {
            FileConfiguration cfg = configManager.getModuleConfig("inflation");
            this.wealthTaxRate = cfg.getDouble("policy.wealth_tax.rate", this.wealthTaxRate);
            this.wealthTaxThresholdMultiplier = cfg.getDouble("policy.wealth_tax.threshold_multiplier", this.wealthTaxThresholdMultiplier);
            this.stimulusFactor = cfg.getDouble("policy.market.stimulus_factor", this.stimulusFactor);
            this.cooldownFactor = cfg.getDouble("policy.market.cooldown_factor", this.cooldownFactor);
            this.interventionMinutes = cfg.getInt("policy.intervention.minutes", this.interventionMinutes);
            this.biasMax = cfg.getDouble("policy.market.bias_max", this.biasMax);
            this.plugin.getLogger().info("Reloaded economic policy from configuration");
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Failed to reload policy: " + e.getMessage());
        }
    }

    private double calculateOptimalInjectionAmount(EconomicSnapshot snapshot) {
        return 10000.0;
    }

    private double calculatePlayerStimulus(PlayerEconomicProfile profile, double pool) {
        return 100.0;
    }

    private void updateVelocityOfMoney() {
    }

    public EconomicCycle getCurrentCycle() {
        return this.currentCycle;
    }

    public double getEconomicHealth() {
        return this.economicHealth;
    }

    public double getInflationRate() {
        return this.inflationRate;
    }

    public double getVelocityOfMoney() {
        return this.velocityOfMoney;
    }

    public static enum EconomicCycle {
        DEPRESSION(0.7, -0.02),
        RECESSION(0.8, -0.01),
        STABLE(1.0, 0.0),
        GROWTH(1.2, 0.01),
        BOOM(1.4, 0.03),
        BUBBLE(1.6, 0.05);

        private final double activityMultiplier;
        private final double baseInflationRate;

        private EconomicCycle(double activityMultiplier, double baseInflationRate) {
            this.activityMultiplier = activityMultiplier;
            this.baseInflationRate = baseInflationRate;
        }

        public double getActivityMultiplier() {
            return this.activityMultiplier;
        }

        public double getBaseInflationRate() {
            return this.baseInflationRate;
        }
    }
}

