/*
 * Decompiled with CFR 0.152.
 */
package me.koyere.ecoxpert.modules.inflation;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

public class PlayerEconomicProfile {
    private final UUID playerId;
    private final LocalDateTime createdAt;
    private LocalDateTime lastUpdated;
    private double totalTransactionVolume = 0.0;
    private int transactionCount = 0;
    private double averageTransactionSize = 0.0;
    private double balanceVolatility = 0.0;
    private int marketTransactions = 0;
    private double marketVolume = 0.0;
    private double marketProfitability = 0.0;
    private boolean isMarketMaker = false;
    private double totalSaved = 0.0;
    private double interestEarned = 0.0;
    private int loanHistory = 0;
    private double creditScore = 700.0;
    private double economicContribution = 0.0;
    private double inflationContribution = 0.0;
    private double velocityContribution = 1.0;
    private EconomicPersonality personality = EconomicPersonality.UNKNOWN;
    private List<EconomicBehaviorPattern> behaviorPatterns = new ArrayList<EconomicBehaviorPattern>();
    private double predictedFutureBalance = 0.0;
    private double riskScore = 0.5;
    private double trustScore = 0.5;

    public PlayerEconomicProfile(UUID playerId) {
        this.playerId = playerId;
        this.createdAt = LocalDateTime.now();
        this.lastUpdated = LocalDateTime.now();
    }

    public void updateBehaviorMetrics() {
        this.lastUpdated = LocalDateTime.now();
        if (this.transactionCount > 0) {
            this.averageTransactionSize = this.totalTransactionVolume / (double)this.transactionCount;
        }
        if (this.marketTransactions > 0) {
            this.isMarketMaker = this.marketTransactions > 50 && this.marketVolume > this.averageTransactionSize * 20.0;
        }
        this.updateBehaviorPatterns();
        this.updatePersonalityClassification();
        this.updatePredictiveMetrics();
    }

    public void calculateEconomicImpact() {
        double positiveImpact = 0.0;
        positiveImpact += this.marketVolume * 0.1;
        positiveImpact += Math.min(this.velocityContribution, 2.0) * 1000.0;
        double negativeImpact = 0.0;
        negativeImpact += Math.max(0.0, this.inflationContribution) * 1000.0;
        this.economicContribution = (positiveImpact += this.totalSaved * 0.05) - (negativeImpact += Math.max(0.0, this.riskScore - 0.5) * 500.0);
    }

    private void updateBehaviorPatterns() {
        this.behaviorPatterns.clear();
        if (this.averageTransactionSize > 10000.0) {
            this.behaviorPatterns.add(EconomicBehaviorPattern.WHALE);
        } else if (this.averageTransactionSize < 100.0 && this.transactionCount > 100) {
            this.behaviorPatterns.add(EconomicBehaviorPattern.MICRO_TRADER);
        }
        if (this.totalSaved > this.totalTransactionVolume * 0.5) {
            this.behaviorPatterns.add(EconomicBehaviorPattern.CONSISTENT_SAVER);
        }
        if (this.isMarketMaker) {
            this.behaviorPatterns.add(EconomicBehaviorPattern.MARKET_MAKER);
        }
        if (Math.abs(this.economicContribution) > 5000.0) {
            this.behaviorPatterns.add(EconomicBehaviorPattern.ECONOMIC_CATALYST);
        }
    }

    private void updatePersonalityClassification() {
        double hoarderScore;
        double speculatorScore;
        double investorScore;
        double spenderScore;
        double traderScore;
        if (this.transactionCount < 10) {
            this.personality = EconomicPersonality.UNKNOWN;
            return;
        }
        double saverScore = this.calculateSaverScore();
        double maxScore = Math.max(saverScore, Math.max(traderScore = this.calculateTraderScore(), Math.max(spenderScore = this.calculateSpenderScore(), Math.max(investorScore = this.calculateInvestorScore(), Math.max(speculatorScore = this.calculateSpeculatorScore(), hoarderScore = this.calculateHoarderScore())))));
        if (maxScore == saverScore) {
            this.personality = EconomicPersonality.SAVER;
        } else if (maxScore == traderScore) {
            this.personality = EconomicPersonality.TRADER;
        } else if (maxScore == spenderScore) {
            this.personality = EconomicPersonality.SPENDER;
        } else if (maxScore == investorScore) {
            this.personality = EconomicPersonality.INVESTOR;
        } else if (maxScore == speculatorScore) {
            this.personality = EconomicPersonality.SPECULATOR;
        } else if (maxScore == hoarderScore) {
            this.personality = EconomicPersonality.HOARDER;
        }
        if (this.riskScore > 0.8 && this.trustScore < 0.3) {
            this.personality = EconomicPersonality.EXPLOITER;
        } else if (this.economicContribution > 10000.0) {
            this.personality = EconomicPersonality.PHILANTHROPIST;
        }
    }

    private void updatePredictiveMetrics() {
        double growthRate = this.calculateExpectedGrowthRate();
        this.predictedFutureBalance = this.getCurrentBalance() * (1.0 + growthRate);
        this.riskScore = this.calculateRiskScore();
        this.trustScore = this.calculateTrustScore();
        this.updateCreditScore();
    }

    private double calculateSaverScore() {
        double savingsRatio = this.totalSaved / Math.max(this.totalTransactionVolume, 1.0);
        double lowSpendingBonus = Math.max(0.0, 1.0 - this.totalTransactionVolume / 100000.0);
        return savingsRatio * 0.7 + lowSpendingBonus * 0.3;
    }

    private double calculateTraderScore() {
        double marketActivityRatio = this.marketVolume / Math.max(this.totalTransactionVolume, 1.0);
        double frequencyBonus = Math.min(1.0, (double)this.marketTransactions / 50.0);
        return marketActivityRatio * 0.6 + frequencyBonus * 0.4;
    }

    private double calculateSpenderScore() {
        double spendingRatio = this.totalTransactionVolume / Math.max(this.totalSaved + this.totalTransactionVolume, 1.0);
        double velocityBonus = Math.min(1.0, this.velocityContribution);
        return spendingRatio * 0.7 + velocityBonus * 0.3;
    }

    private double calculateInvestorScore() {
        double longTermSavings = this.totalSaved > 50000.0 ? 1.0 : this.totalSaved / 50000.0;
        double consistencyBonus = 1.0 - this.balanceVolatility;
        return longTermSavings * 0.6 + consistencyBonus * 0.4;
    }

    private double calculateSpeculatorScore() {
        double volatilityScore = this.balanceVolatility;
        double riskTakingScore = this.riskScore;
        return volatilityScore * 0.5 + riskTakingScore * 0.5;
    }

    private double calculateHoarderScore() {
        if (this.totalTransactionVolume < 1000.0) {
            return 1.0;
        }
        double lowActivityScore = 1.0 - Math.min(1.0, (double)this.transactionCount / 100.0);
        double highBalanceScore = Math.min(1.0, this.getCurrentBalance() / 100000.0);
        return lowActivityScore * 0.6 + highBalanceScore * 0.4;
    }

    private double calculateExpectedGrowthRate() {
        switch (this.personality) {
            case INVESTOR: {
                double d = 0.1;
                break;
            }
            case TRADER: {
                double d = 0.05;
                break;
            }
            case SPECULATOR: {
                double d = 0.15;
                break;
            }
            case SAVER: {
                double d = 0.02;
                break;
            }
            case SPENDER: {
                double d = -0.05;
                break;
            }
            case HOARDER: {
                double d = 0.01;
                break;
            }
            default: {
                double baseGrowth;
                double d = baseGrowth = 0.0;
            }
        }
        if (this.isMarketMaker) {
            baseGrowth += 0.03;
        }
        return Math.max(-0.5, Math.min(0.5, baseGrowth += this.economicContribution / 100000.0));
    }

    private double calculateRiskScore() {
        double behaviorRisk = this.balanceVolatility;
        double marketRisk = this.marketTransactions > 0 ? Math.min(1.0, this.marketVolume / this.totalTransactionVolume) : 0.0;
        double creditRisk = this.loanHistory > 5 ? 0.3 : 0.0;
        return Math.max(0.0, Math.min(1.0, behaviorRisk * 0.5 + marketRisk * 0.3 + creditRisk * 0.2));
    }

    private double calculateTrustScore() {
        double contributionScore = Math.max(0.0, Math.min(1.0, this.economicContribution / 10000.0));
        double consistencyScore = 1.0 - this.balanceVolatility;
        double longevityScore = Math.min(1.0, (double)Duration.between(this.createdAt, LocalDateTime.now()).toDays() / 365.0);
        return contributionScore * 0.4 + consistencyScore * 0.3 + longevityScore * 0.3;
    }

    private void updateCreditScore() {
        double newScore = 700.0;
        newScore += (this.trustScore - 0.5) * 200.0;
        newScore -= (this.riskScore - 0.5) * 150.0;
        newScore += Math.min(100.0, this.economicContribution / 100.0);
        newScore -= (double)(this.loanHistory * 10);
        this.creditScore = Math.max(300.0, Math.min(850.0, newScore += Math.min(50.0, this.interestEarned / 100.0)));
    }

    private double getCurrentBalance() {
        return 25000.0;
    }

    public void recordTransaction(double amount, String type) {
        this.totalTransactionVolume += Math.abs(amount);
        ++this.transactionCount;
        if ("MARKET".equals(type)) {
            ++this.marketTransactions;
            this.marketVolume += Math.abs(amount);
        }
        this.balanceVolatility = Math.min(1.0, this.balanceVolatility * 0.9 + Math.abs(amount) / 10000.0 * 0.1);
        this.lastUpdated = LocalDateTime.now();
    }

    public List<String> getRecommendedActions() {
        ArrayList<String> recommendations = new ArrayList<String>();
        if (this.personality == EconomicPersonality.SPENDER && this.totalSaved < 1000.0) {
            recommendations.add("Consider saving some money in the bank for future opportunities");
        }
        if (this.personality == EconomicPersonality.HOARDER && this.transactionCount < 10) {
            recommendations.add("Try participating in the market to grow your wealth");
        }
        if (this.creditScore < 600.0) {
            recommendations.add("Improve your credit score by making regular transactions and contributing to the economy");
        }
        if (this.isMarketMaker) {
            recommendations.add("Great job providing market liquidity! You're helping the server economy");
        }
        return recommendations;
    }

    public UUID getPlayerId() {
        return this.playerId;
    }

    public EconomicPersonality getPersonality() {
        return this.personality;
    }

    public List<EconomicBehaviorPattern> getBehaviorPatterns() {
        return this.behaviorPatterns;
    }

    public double getEconomicContribution() {
        return this.economicContribution;
    }

    public double getCreditScore() {
        return this.creditScore;
    }

    public double getRiskScore() {
        return this.riskScore;
    }

    public double getTrustScore() {
        return this.trustScore;
    }

    public double getPredictedFutureBalance() {
        return this.predictedFutureBalance;
    }

    public double getTotalTransactionVolume() {
        return this.totalTransactionVolume;
    }

    public int getTransactionCount() {
        return this.transactionCount;
    }

    public boolean isMarketMaker() {
        return this.isMarketMaker;
    }

    public LocalDateTime getLastUpdated() {
        return this.lastUpdated;
    }

    public static enum EconomicPersonality {
        UNKNOWN,
        SAVER,
        SPENDER,
        TRADER,
        INVESTOR,
        SPECULATOR,
        HOARDER,
        PHILANTHROPIST,
        EXPLOITER;

    }

    public static enum EconomicBehaviorPattern {
        CONSISTENT_SAVER,
        PANIC_SELLER,
        MOMENTUM_TRADER,
        CONTRARIAN,
        WHALE,
        MICRO_TRADER,
        SEASONAL_SPENDER,
        DEBT_ACCUMULATOR,
        MARKET_MAKER,
        ECONOMIC_CATALYST;

    }
}

