/*
 * Decompiled with CFR 0.152.
 */
package me.koyere.ecoxpert.modules.market;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.List;
import me.koyere.ecoxpert.modules.market.MarketItem;
import me.koyere.ecoxpert.modules.market.MarketTransaction;
import me.koyere.ecoxpert.modules.market.MarketTrend;
import org.bukkit.Material;

public class PriceCalculator {
    private BigDecimal maxPriceChange = BigDecimal.valueOf(0.2);
    private static final BigDecimal MIN_PRICE_RATIO = BigDecimal.valueOf(0.1);
    private static final BigDecimal MAX_PRICE_RATIO = BigDecimal.valueOf(10.0);
    private BigDecimal volatilityDamping = BigDecimal.valueOf(0.85);
    private static final BigDecimal MOMENTUM_FACTOR = BigDecimal.valueOf(0.3);
    private int trendAnalysisHours = 24;

    public void configure(double maxChangeFrac, double dampingFrac, int trendHours) {
        double mc = Math.max(0.01, Math.min(0.5, maxChangeFrac));
        double vd = Math.max(0.1, Math.min(0.99, dampingFrac));
        int th = Math.max(1, Math.min(168, trendHours));
        this.maxPriceChange = BigDecimal.valueOf(mc);
        this.volatilityDamping = BigDecimal.valueOf(vd);
        this.trendAnalysisHours = th;
    }

    public MarketPriceUpdate calculatePriceUpdate(MarketItem item, List<MarketTransaction> recentTransactions) {
        BigDecimal currentSellPrice;
        BigDecimal currentBuyPrice;
        BigDecimal basePrice = item.getBasePrice();
        if (basePrice == null || basePrice.compareTo(BigDecimal.ZERO) <= 0) {
            basePrice = BigDecimal.ONE;
        }
        if ((currentBuyPrice = item.getCurrentBuyPrice()) == null || currentBuyPrice.compareTo(BigDecimal.ZERO) <= 0) {
            currentBuyPrice = basePrice;
        }
        if ((currentSellPrice = item.getCurrentSellPrice()) == null || currentSellPrice.compareTo(BigDecimal.ZERO) <= 0) {
            currentSellPrice = basePrice.multiply(BigDecimal.valueOf(0.8));
        }
        SupplyDemandAnalysis analysis = this.analyzeSupplyDemand(recentTransactions, item.getMaterial());
        BigDecimal demandMultiplier = this.calculateDemandMultiplier(analysis);
        BigDecimal volatilityAdjustment = this.calculateVolatilityAdjustment(item, analysis);
        BigDecimal momentumAdjustment = this.calculateMomentumAdjustment(recentTransactions);
        BigDecimal newBuyPrice = this.calculateNewBuyPrice(basePrice, currentBuyPrice, demandMultiplier, volatilityAdjustment, momentumAdjustment);
        BigDecimal newSellPrice = this.calculateNewSellPrice(newBuyPrice, analysis);
        newBuyPrice = this.applySafetyConstraints(basePrice, newBuyPrice);
        newSellPrice = this.applySafetyConstraints(basePrice.multiply(BigDecimal.valueOf(0.8)), newSellPrice);
        MarketTrend.TrendDirection direction = this.determineTrendDirection(currentBuyPrice, newBuyPrice, analysis);
        BigDecimal priceChange = newBuyPrice.subtract(currentBuyPrice);
        BigDecimal priceChangePercentage = this.calculatePercentageChange(currentBuyPrice, newBuyPrice);
        return new MarketPriceUpdate(item.getMaterial(), newBuyPrice, newSellPrice, priceChange, priceChangePercentage, direction, analysis.getVolatility(), LocalDateTime.now());
    }

    private SupplyDemandAnalysis analyzeSupplyDemand(List<MarketTransaction> transactions, Material material) {
        int buyTransactions = 0;
        int sellTransactions = 0;
        BigDecimal buyVolume = BigDecimal.ZERO;
        BigDecimal sellVolume = BigDecimal.ZERO;
        BigDecimal totalVolume = BigDecimal.ZERO;
        LocalDateTime cutoff = LocalDateTime.now().minusHours(this.trendAnalysisHours);
        for (MarketTransaction transaction : transactions) {
            if (!transaction.getMaterial().equals((Object)material) || transaction.getTimestamp().isBefore(cutoff)) continue;
            BigDecimal transactionVolume = transaction.getTotalAmount();
            totalVolume = totalVolume.add(transactionVolume);
            if (transaction.isBuyTransaction()) {
                ++buyTransactions;
                buyVolume = buyVolume.add(transactionVolume);
                continue;
            }
            ++sellTransactions;
            sellVolume = sellVolume.add(transactionVolume);
        }
        double demandRatio = this.calculateDemandRatio(buyTransactions, sellTransactions);
        double volumeRatio = this.calculateVolumeRatio(buyVolume, sellVolume);
        double volatility = this.calculateVolatility(transactions, material);
        double activityLevel = this.calculateActivityLevel(buyTransactions + sellTransactions);
        return new SupplyDemandAnalysis(buyTransactions, sellTransactions, buyVolume, sellVolume, demandRatio, volumeRatio, volatility, activityLevel);
    }

    private BigDecimal calculateDemandMultiplier(SupplyDemandAnalysis analysis) {
        double demandRatio = analysis.getDemandRatio();
        double volumeRatio = analysis.getVolumeRatio();
        double activityLevel = analysis.getActivityLevel();
        double combinedRatio = demandRatio * 0.6 + volumeRatio * 0.4;
        double activityMultiplier = 0.5 + activityLevel * 0.5;
        double multiplier = 1.0 + ((combinedRatio *= activityMultiplier) - 1.0) * 0.1;
        return BigDecimal.valueOf(Math.max(0.9, Math.min(1.1, multiplier)));
    }

    private BigDecimal calculateVolatilityAdjustment(MarketItem item, SupplyDemandAnalysis analysis) {
        double currentVolatility = item.getPriceVolatility().doubleValue();
        double marketVolatility = analysis.getVolatility();
        double newVolatility = currentVolatility * 0.7 + marketVolatility * 0.3;
        return BigDecimal.valueOf(Math.max(0.01, Math.min(0.5, newVolatility *= this.volatilityDamping.doubleValue())));
    }

    private BigDecimal calculateMomentumAdjustment(List<MarketTransaction> transactions) {
        if (transactions.size() < 2) {
            return BigDecimal.ZERO;
        }
        LocalDateTime now = LocalDateTime.now();
        LocalDateTime oneHourAgo = now.minusHours(1L);
        long recentTransactions = transactions.stream().filter(t -> t.getTimestamp().isAfter(oneHourAgo)).count();
        double momentum = Math.min(1.0, (double)recentTransactions / 10.0);
        return BigDecimal.valueOf(momentum * MOMENTUM_FACTOR.doubleValue());
    }

    private BigDecimal calculateNewBuyPrice(BigDecimal basePrice, BigDecimal currentPrice, BigDecimal demandMultiplier, BigDecimal volatilityAdjustment, BigDecimal momentumAdjustment) {
        BigDecimal newPrice = currentPrice;
        newPrice = newPrice.multiply(demandMultiplier);
        BigDecimal volatilityChange = BigDecimal.valueOf(Math.random() - 0.5).multiply(volatilityAdjustment).multiply(basePrice);
        newPrice = newPrice.add(volatilityChange);
        BigDecimal momentumChange = momentumAdjustment.multiply(basePrice);
        newPrice = newPrice.add(momentumChange);
        BigDecimal maxChange = currentPrice.multiply(this.maxPriceChange);
        BigDecimal change = newPrice.subtract(currentPrice);
        if (change.abs().compareTo(maxChange) > 0) {
            BigDecimal limitedChange = change.signum() == 1 ? maxChange : maxChange.negate();
            newPrice = currentPrice.add(limitedChange);
        }
        return newPrice.setScale(2, RoundingMode.HALF_UP);
    }

    private BigDecimal calculateNewSellPrice(BigDecimal newBuyPrice, SupplyDemandAnalysis analysis) {
        BigDecimal baseSellRatio = BigDecimal.valueOf(0.8);
        double activityLevel = analysis.getActivityLevel();
        double volatility = analysis.getVolatility();
        double spreadAdjustment = 1.0 - activityLevel * 0.1 + volatility * 0.1;
        BigDecimal adjustedRatio = baseSellRatio.multiply(BigDecimal.valueOf(spreadAdjustment));
        adjustedRatio = adjustedRatio.max(BigDecimal.valueOf(0.6));
        adjustedRatio = adjustedRatio.min(BigDecimal.valueOf(0.95));
        return newBuyPrice.multiply(adjustedRatio).setScale(2, RoundingMode.HALF_UP);
    }

    private BigDecimal applySafetyConstraints(BigDecimal basePrice, BigDecimal newPrice) {
        BigDecimal minPrice = basePrice.multiply(MIN_PRICE_RATIO);
        BigDecimal maxPrice = basePrice.multiply(MAX_PRICE_RATIO);
        if (newPrice.compareTo(minPrice) < 0) {
            return minPrice;
        }
        if (newPrice.compareTo(maxPrice) > 0) {
            return maxPrice;
        }
        return newPrice;
    }

    private double calculateDemandRatio(int buyTransactions, int sellTransactions) {
        if (sellTransactions == 0) {
            return buyTransactions > 0 ? 2.0 : 1.0;
        }
        return (double)buyTransactions / (double)sellTransactions;
    }

    private double calculateVolumeRatio(BigDecimal buyVolume, BigDecimal sellVolume) {
        if (sellVolume.equals(BigDecimal.ZERO)) {
            return buyVolume.compareTo(BigDecimal.ZERO) > 0 ? 2.0 : 1.0;
        }
        return buyVolume.divide(sellVolume, 4, RoundingMode.HALF_UP).doubleValue();
    }

    private double calculateVolatility(List<MarketTransaction> transactions, Material material) {
        if (transactions.size() < 2) {
            return 0.1;
        }
        List<BigDecimal> prices = transactions.stream().filter(t -> t.getMaterial().equals((Object)material)).map(MarketTransaction::getUnitPrice).toList();
        if (prices.size() < 2) {
            return 0.1;
        }
        BigDecimal average = prices.stream().reduce(BigDecimal.ZERO, BigDecimal::add).divide(BigDecimal.valueOf(prices.size()), 4, RoundingMode.HALF_UP);
        BigDecimal variance = prices.stream().map(price -> price.subtract(average).pow(2)).reduce(BigDecimal.ZERO, BigDecimal::add).divide(BigDecimal.valueOf(prices.size()), 4, RoundingMode.HALF_UP);
        double standardDeviation = Math.sqrt(variance.doubleValue());
        double volatility = standardDeviation / average.doubleValue();
        return Math.min(0.5, Math.max(0.01, volatility));
    }

    private double calculateActivityLevel(int totalTransactions) {
        return Math.min(1.0, (double)totalTransactions / 50.0);
    }

    private MarketTrend.TrendDirection determineTrendDirection(BigDecimal oldPrice, BigDecimal newPrice, SupplyDemandAnalysis analysis) {
        BigDecimal changePercentage = this.calculatePercentageChange(oldPrice, newPrice);
        double volatility = analysis.getVolatility();
        if (volatility > 0.3) {
            return MarketTrend.TrendDirection.VOLATILE;
        }
        double changePercent = changePercentage.doubleValue();
        if (changePercent > 5.0) {
            return MarketTrend.TrendDirection.STRONG_UPWARD;
        }
        if (changePercent > 1.0) {
            return MarketTrend.TrendDirection.UPWARD;
        }
        if (changePercent < -5.0) {
            return MarketTrend.TrendDirection.STRONG_DOWNWARD;
        }
        if (changePercent < -1.0) {
            return MarketTrend.TrendDirection.DOWNWARD;
        }
        return MarketTrend.TrendDirection.STABLE;
    }

    private BigDecimal calculatePercentageChange(BigDecimal oldValue, BigDecimal newValue) {
        if (oldValue.equals(BigDecimal.ZERO)) {
            return BigDecimal.ZERO;
        }
        return newValue.subtract(oldValue).divide(oldValue, 4, RoundingMode.HALF_UP).multiply(BigDecimal.valueOf(100L));
    }

    private static class SupplyDemandAnalysis {
        private final int buyTransactions;
        private final int sellTransactions;
        private final BigDecimal buyVolume;
        private final BigDecimal sellVolume;
        private final double demandRatio;
        private final double volumeRatio;
        private final double volatility;
        private final double activityLevel;

        public SupplyDemandAnalysis(int buyTransactions, int sellTransactions, BigDecimal buyVolume, BigDecimal sellVolume, double demandRatio, double volumeRatio, double volatility, double activityLevel) {
            this.buyTransactions = buyTransactions;
            this.sellTransactions = sellTransactions;
            this.buyVolume = buyVolume;
            this.sellVolume = sellVolume;
            this.demandRatio = demandRatio;
            this.volumeRatio = volumeRatio;
            this.volatility = volatility;
            this.activityLevel = activityLevel;
        }

        public double getDemandRatio() {
            return this.demandRatio;
        }

        public double getVolumeRatio() {
            return this.volumeRatio;
        }

        public double getVolatility() {
            return this.volatility;
        }

        public double getActivityLevel() {
            return this.activityLevel;
        }
    }

    public static class MarketPriceUpdate {
        private final Material material;
        private final BigDecimal newBuyPrice;
        private final BigDecimal newSellPrice;
        private final BigDecimal priceChange;
        private final BigDecimal priceChangePercentage;
        private final MarketTrend.TrendDirection trendDirection;
        private final double volatility;
        private final LocalDateTime updateTime;

        public MarketPriceUpdate(Material material, BigDecimal newBuyPrice, BigDecimal newSellPrice, BigDecimal priceChange, BigDecimal priceChangePercentage, MarketTrend.TrendDirection trendDirection, double volatility, LocalDateTime updateTime) {
            this.material = material;
            this.newBuyPrice = newBuyPrice;
            this.newSellPrice = newSellPrice;
            this.priceChange = priceChange;
            this.priceChangePercentage = priceChangePercentage;
            this.trendDirection = trendDirection;
            this.volatility = volatility;
            this.updateTime = updateTime;
        }

        public Material getMaterial() {
            return this.material;
        }

        public BigDecimal getNewBuyPrice() {
            return this.newBuyPrice;
        }

        public BigDecimal getNewSellPrice() {
            return this.newSellPrice;
        }

        public BigDecimal getPriceChange() {
            return this.priceChange;
        }

        public BigDecimal getPriceChangePercentage() {
            return this.priceChangePercentage;
        }

        public MarketTrend.TrendDirection getTrendDirection() {
            return this.trendDirection;
        }

        public double getVolatility() {
            return this.volatility;
        }

        public LocalDateTime getUpdateTime() {
            return this.updateTime;
        }
    }
}

