/*
 * Decompiled with CFR 0.152.
 */
package me.koyere.ecoxpert.core.data;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import me.koyere.ecoxpert.EcoXpertPlugin;
import me.koyere.ecoxpert.core.config.ConfigManager;
import me.koyere.ecoxpert.core.config.DatabasePoolSettings;
import me.koyere.ecoxpert.core.config.MySqlConfig;
import me.koyere.ecoxpert.core.data.DataManager;
import me.koyere.ecoxpert.core.data.DatabaseStatus;
import me.koyere.ecoxpert.core.data.DatabaseTransaction;
import me.koyere.ecoxpert.core.data.DatabaseTransactionImpl;
import me.koyere.ecoxpert.core.data.QueryResult;
import me.koyere.ecoxpert.core.data.QueryResultImpl;
import me.koyere.ecoxpert.libs.hikari.HikariConfig;
import me.koyere.ecoxpert.libs.hikari.HikariDataSource;

public class DataManagerImpl
implements DataManager {
    private final EcoXpertPlugin plugin;
    private final ConfigManager configManager;
    private final ExecutorService databaseExecutor;
    private HikariDataSource dataSource;
    private boolean connected = false;
    private DatabaseType databaseType;

    public DataManagerImpl(EcoXpertPlugin plugin, ConfigManager configManager) {
        this.plugin = plugin;
        this.configManager = configManager;
        this.databaseExecutor = Executors.newFixedThreadPool(4, r -> {
            Thread thread = new Thread(r, "EcoXpert-Database-" + Thread.currentThread().getId());
            thread.setDaemon(true);
            return thread;
        });
    }

    @Override
    public void initialize() {
        this.plugin.getLogger().info("Initializing data management system...");
        try {
            String dbTypeStr = this.configManager.getDatabaseType().toLowerCase();
            this.databaseType = dbTypeStr.equals("mysql") ? DatabaseType.MYSQL : DatabaseType.SQLITE;
            this.plugin.getLogger().info("Database type: " + String.valueOf((Object)this.databaseType));
            this.initializeDataSource();
            this.testConnection();
            this.createTables();
            this.connected = true;
            this.plugin.getLogger().info("Data management system initialized successfully");
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Failed to initialize data management system: " + e.getMessage());
            throw new RuntimeException("Database initialization failed", e);
        }
    }

    @Override
    public void shutdown() {
        if (this.connected) {
            this.plugin.getLogger().info("Shutting down data management system...");
            try {
                if (this.dataSource != null && !this.dataSource.isClosed()) {
                    this.dataSource.close();
                }
                this.databaseExecutor.shutdown();
            }
            catch (Exception e) {
                this.plugin.getLogger().warning("Error during database shutdown: " + e.getMessage());
            }
            finally {
                this.connected = false;
                this.plugin.getLogger().info("Data management system shutdown complete");
            }
        }
    }

    @Override
    public boolean isConnected() {
        return this.connected;
    }

    @Override
    public String getDatabaseType() {
        return this.databaseType != null ? this.databaseType.name().toLowerCase() : "unknown";
    }

    @Override
    public CompletableFuture<Integer> executeUpdate(String sql, Object ... params) {
        return CompletableFuture.supplyAsync(() -> {
            this.checkConnection();
            try (Connection conn = this.dataSource.getConnection();){
                Integer n;
                block14: {
                    PreparedStatement stmt = conn.prepareStatement(sql);
                    try {
                        this.setParameters(stmt, params);
                        n = stmt.executeUpdate();
                        if (stmt == null) break block14;
                    }
                    catch (Throwable throwable) {
                        if (stmt != null) {
                            try {
                                stmt.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    stmt.close();
                }
                return n;
            }
            catch (SQLException e) {
                throw new RuntimeException("Failed to execute update: " + sql, e);
            }
        }, this.databaseExecutor);
    }

    @Override
    public CompletableFuture<QueryResult> executeQuery(String sql, Object ... params) {
        return CompletableFuture.supplyAsync(() -> {
            this.checkConnection();
            try {
                Connection conn = this.dataSource.getConnection();
                PreparedStatement stmt = conn.prepareStatement(sql);
                this.setParameters(stmt, params);
                return new QueryResultImpl(stmt.executeQuery(), conn, stmt);
            }
            catch (SQLException e) {
                throw new RuntimeException("Failed to execute query: " + sql, e);
            }
        }, this.databaseExecutor);
    }

    @Override
    public CompletableFuture<int[]> executeBatch(String sql, Object[] ... paramsList) {
        return CompletableFuture.supplyAsync(() -> {
            this.checkConnection();
            try (Connection conn = this.dataSource.getConnection();){
                int[] nArray;
                block15: {
                    PreparedStatement stmt = conn.prepareStatement(sql);
                    try {
                        for (Object[] params : paramsList) {
                            this.setParameters(stmt, params);
                            stmt.addBatch();
                        }
                        nArray = stmt.executeBatch();
                        if (stmt == null) break block15;
                    }
                    catch (Throwable throwable) {
                        if (stmt != null) {
                            try {
                                stmt.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    stmt.close();
                }
                return nArray;
            }
            catch (SQLException e) {
                throw new RuntimeException("Failed to execute batch: " + sql, e);
            }
        }, this.databaseExecutor);
    }

    @Override
    public CompletableFuture<DatabaseTransaction> beginTransaction() {
        return CompletableFuture.supplyAsync(() -> {
            this.checkConnection();
            try {
                Connection conn = this.dataSource.getConnection();
                return new DatabaseTransactionImpl(conn, this.databaseExecutor);
            }
            catch (SQLException e) {
                throw new RuntimeException("Failed to begin transaction", e);
            }
        }, this.databaseExecutor);
    }

    @Override
    public void createTables() {
        this.plugin.getLogger().info("Creating database tables...");
        try {
            this.createEconomyTables();
            this.plugin.getLogger().info("Database tables created successfully");
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Failed to create database tables: " + e.getMessage());
            throw new RuntimeException("Table creation failed", e);
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    @Override
    public boolean needsMigration() {
        try (Connection conn = this.dataSource.getConnection();){
            boolean bl;
            block18: {
                PreparedStatement stmt;
                block16: {
                    boolean bl2;
                    block17: {
                        stmt = conn.prepareStatement("SELECT version FROM ecoxpert_meta WHERE key = 'schema_version'");
                        try {
                            ResultSet result = stmt.executeQuery();
                            if (!result.next()) break block16;
                            int currentVersion = result.getInt("version");
                            boolean bl3 = bl2 = currentVersion < this.getCurrentSchemaVersion();
                            if (stmt == null) break block17;
                        }
                        catch (Throwable throwable) {
                            if (stmt != null) {
                                try {
                                    stmt.close();
                                }
                                catch (Throwable throwable2) {
                                    throwable.addSuppressed(throwable2);
                                }
                            }
                            throw throwable;
                        }
                        stmt.close();
                    }
                    return bl2;
                }
                bl = true;
                if (stmt == null) break block18;
                stmt.close();
            }
            return bl;
        }
        catch (SQLException e) {
            return true;
        }
    }

    @Override
    public CompletableFuture<Void> migrate() {
        return CompletableFuture.runAsync(() -> {
            this.plugin.getLogger().info("Starting database migration...");
            try {
                this.updateSchemaVersion();
                this.plugin.getLogger().info("Database migration completed successfully");
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Database migration failed: " + e.getMessage());
                throw new RuntimeException("Migration failed", e);
            }
        }, this.databaseExecutor);
    }

    private void initializeDataSource() {
        HikariConfig config = new HikariConfig();
        config.setMaximumPoolSize(10);
        config.setMinimumIdle(2);
        config.setConnectionTimeout(30000L);
        config.setIdleTimeout(600000L);
        config.setMaxLifetime(1800000L);
        config.setLeakDetectionThreshold(60000L);
        if (this.databaseType == DatabaseType.SQLITE) {
            this.configureSQLite(config);
        } else {
            this.configureMySQL(config);
        }
        this.dataSource = new HikariDataSource(config);
    }

    private void configureSQLite(HikariConfig config) {
        File dataFolder = this.plugin.getDataFolder();
        if (!dataFolder.exists()) {
            dataFolder.mkdirs();
        }
        File dbFile = new File(dataFolder, "ecoxpert.db");
        config.setJdbcUrl("jdbc:sqlite:" + dbFile.getAbsolutePath() + "?busy_timeout=5000");
        config.setDriverClassName("org.sqlite.JDBC");
        config.addDataSourceProperty("cache_size", "8192");
        config.addDataSourceProperty("synchronous", "NORMAL");
        config.addDataSourceProperty("journal_mode", "WAL");
        try {
            config.setConnectionInitSql("PRAGMA busy_timeout=5000;");
        }
        catch (Throwable throwable) {
            // empty catch block
        }
    }

    private void configureMySQL(HikariConfig config) {
        MySqlConfig mySql = this.configManager.getMySqlConfig();
        if (mySql == null) {
            throw new IllegalStateException("MySQL configuration not available");
        }
        this.validateMySqlConfig(mySql);
        String jdbcUrl = String.format("jdbc:mysql://%s:%d/%s", mySql.getHost().trim(), mySql.getPort(), mySql.getDatabase().trim());
        config.setJdbcUrl(jdbcUrl);
        config.setUsername(mySql.getUsername());
        config.setPassword(mySql.getPassword());
        config.setDriverClassName("com.mysql.cj.jdbc.Driver");
        config.setPoolName("EcoXpert-MySQL");
        config.addDataSourceProperty("cachePrepStmts", "true");
        config.addDataSourceProperty("prepStmtCacheSize", "250");
        config.addDataSourceProperty("prepStmtCacheSqlLimit", "2048");
        config.addDataSourceProperty("useServerPrepStmts", "true");
        config.addDataSourceProperty("useLocalSessionState", "true");
        config.addDataSourceProperty("rewriteBatchedStatements", "true");
        config.addDataSourceProperty("cacheResultSetMetadata", "true");
        config.addDataSourceProperty("cacheServerConfiguration", "true");
        config.addDataSourceProperty("maintainTimeStats", "false");
        config.addDataSourceProperty("useUnicode", "true");
        config.addDataSourceProperty("characterEncoding", "utf8mb4");
        config.addDataSourceProperty("connectionCollation", "utf8mb4_unicode_ci");
        config.addDataSourceProperty("characterSetResults", "utf8mb4");
        config.addDataSourceProperty("serverTimezone", "UTC");
        config.addDataSourceProperty("useSSL", Boolean.toString(mySql.isUseSsl()));
        config.addDataSourceProperty("allowPublicKeyRetrieval", Boolean.toString(mySql.isAllowPublicKeyRetrieval()));
        this.applyPoolOverrides(config, mySql.getPoolSettings());
    }

    private void applyPoolOverrides(HikariConfig config, DatabasePoolSettings overrides) {
        if (overrides == null) {
            return;
        }
        overrides.getMaximumPoolSize().ifPresent(config::setMaximumPoolSize);
        overrides.getMinimumIdle().ifPresent(config::setMinimumIdle);
        overrides.getConnectionTimeout().ifPresent(config::setConnectionTimeout);
        overrides.getIdleTimeout().ifPresent(config::setIdleTimeout);
        overrides.getMaxLifetime().ifPresent(config::setMaxLifetime);
    }

    private void validateMySqlConfig(MySqlConfig config) {
        if (config.getHost().trim().isEmpty()) {
            throw new IllegalStateException("database.mysql.host must not be empty");
        }
        if (config.getDatabase().trim().isEmpty()) {
            throw new IllegalStateException("database.mysql.database must not be empty");
        }
        if (config.getUsername().trim().isEmpty()) {
            throw new IllegalStateException("database.mysql.username must not be empty");
        }
        if (config.getPort() <= 0 || config.getPort() > 65535) {
            throw new IllegalStateException("database.mysql.port must be a valid TCP port");
        }
    }

    private void testConnection() throws SQLException {
        try (Connection conn = this.dataSource.getConnection();){
            if (!conn.isValid(5)) {
                throw new SQLException("Connection validation failed");
            }
            this.plugin.getLogger().info("Database connection test successful");
        }
    }

    private void createEconomyTables() throws SQLException {
        String[] tables = new String[]{"CREATE TABLE IF NOT EXISTS ecoxpert_meta (\n    key VARCHAR(255) PRIMARY KEY,\n    version INTEGER NOT NULL,\n    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n)\n", "CREATE TABLE IF NOT EXISTS ecoxpert_accounts (\n    player_uuid VARCHAR(36) PRIMARY KEY,\n    balance DECIMAL(20,2) NOT NULL DEFAULT 0.00,\n    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n)\n", "CREATE TABLE IF NOT EXISTS ecoxpert_transactions (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    from_uuid VARCHAR(36),\n    to_uuid VARCHAR(36),\n    amount DECIMAL(20,2) NOT NULL,\n    type VARCHAR(50) NOT NULL,\n    description TEXT,\n    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n)\n", "CREATE TABLE IF NOT EXISTS ecoxpert_market_items (\n    material VARCHAR(100) PRIMARY KEY,\n    base_price DECIMAL(20,2) NOT NULL,\n    current_buy_price DECIMAL(20,2) NOT NULL,\n    current_sell_price DECIMAL(20,2) NOT NULL,\n    buyable BOOLEAN DEFAULT TRUE,\n    sellable BOOLEAN DEFAULT TRUE,\n    total_sold INTEGER DEFAULT 0,\n    total_bought INTEGER DEFAULT 0,\n    price_volatility DECIMAL(5,4) DEFAULT 0.1000,\n    last_price_update TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n)\n", "CREATE TABLE IF NOT EXISTS ecoxpert_market_transactions (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    player_uuid VARCHAR(36) NOT NULL,\n    player_name VARCHAR(50) NOT NULL,\n    material VARCHAR(100) NOT NULL,\n    transaction_type VARCHAR(10) NOT NULL,\n    quantity INTEGER NOT NULL,\n    unit_price DECIMAL(20,2) NOT NULL,\n    total_amount DECIMAL(20,2) NOT NULL,\n    description TEXT,\n    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n)\n", "CREATE TABLE IF NOT EXISTS ecoxpert_market_price_history (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    material VARCHAR(100) NOT NULL,\n    buy_price DECIMAL(20,2) NOT NULL,\n    sell_price DECIMAL(20,2) NOT NULL,\n    transaction_count INTEGER DEFAULT 0,\n    volume DECIMAL(20,2) DEFAULT 0.00,\n    snapshot_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n)\n", "CREATE TABLE IF NOT EXISTS ecoxpert_bank_accounts (\n    player_uuid VARCHAR(36) PRIMARY KEY,\n    account_number VARCHAR(20) UNIQUE NOT NULL,\n    balance DECIMAL(20,2) NOT NULL DEFAULT 0.00,\n    tier VARCHAR(20) NOT NULL DEFAULT 'BASIC',\n    total_interest_earned DECIMAL(20,2) DEFAULT 0.00,\n    frozen BOOLEAN DEFAULT FALSE,\n    frozen_reason TEXT,\n    daily_deposit_used DECIMAL(20,2) DEFAULT 0.00,\n    daily_withdraw_used DECIMAL(20,2) DEFAULT 0.00,\n    daily_transfer_used DECIMAL(20,2) DEFAULT 0.00,\n    last_reset_date DATE DEFAULT CURRENT_DATE,\n    last_interest_calculation TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n    failed_transaction_count INTEGER DEFAULT 0,\n    last_failed_transaction TIMESTAMP,\n    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n)\n", "CREATE TABLE IF NOT EXISTS ecoxpert_bank_transactions (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    account_uuid VARCHAR(36) NOT NULL,\n    transaction_type VARCHAR(30) NOT NULL,\n    amount DECIMAL(20,2) NOT NULL,\n    balance_before DECIMAL(20,2) NOT NULL,\n    balance_after DECIMAL(20,2) NOT NULL,\n    description TEXT,\n    reference VARCHAR(100),\n    related_account_uuid VARCHAR(36),\n    admin_id VARCHAR(100),\n    ip_address VARCHAR(45),\n    reason TEXT,\n    transaction_hash VARCHAR(100) NOT NULL,\n    verified BOOLEAN DEFAULT TRUE,\n    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n)\n", "CREATE TABLE IF NOT EXISTS ecoxpert_bank_statements (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    account_uuid VARCHAR(36) NOT NULL,\n    statement_year INTEGER NOT NULL,\n    statement_month INTEGER NOT NULL,\n    opening_balance DECIMAL(20,2) NOT NULL,\n    closing_balance DECIMAL(20,2) NOT NULL,\n    total_deposits DECIMAL(20,2) DEFAULT 0.00,\n    total_withdrawals DECIMAL(20,2) DEFAULT 0.00,\n    total_transfers_in DECIMAL(20,2) DEFAULT 0.00,\n    total_transfers_out DECIMAL(20,2) DEFAULT 0.00,\n    total_interest DECIMAL(20,2) DEFAULT 0.00,\n    total_fees DECIMAL(20,2) DEFAULT 0.00,\n    transaction_count INTEGER DEFAULT 0,\n    generated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n    UNIQUE(account_uuid, statement_year, statement_month)\n)\n", "CREATE TABLE IF NOT EXISTS ecoxpert_economic_events (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    event_id VARCHAR(100) NOT NULL,\n    type VARCHAR(50) NOT NULL,\n    status VARCHAR(20) NOT NULL,\n    parameters TEXT,\n    start_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n    end_time TIMESTAMP\n)\n", "CREATE TABLE IF NOT EXISTS ecoxpert_loans (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    player_uuid VARCHAR(36) NOT NULL,\n    principal DECIMAL(20,2) NOT NULL,\n    outstanding DECIMAL(20,2) NOT NULL,\n    interest_rate DECIMAL(5,4) NOT NULL DEFAULT 0.0000,\n    status VARCHAR(20) NOT NULL DEFAULT 'ACTIVE',\n    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n    last_payment_at TIMESTAMP\n)\n", "CREATE TABLE IF NOT EXISTS ecoxpert_loan_schedules (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    loan_id INTEGER NOT NULL,\n    installment_no INTEGER NOT NULL,\n    due_date DATE NOT NULL,\n    amount_due DECIMAL(20,2) NOT NULL,\n    paid_amount DECIMAL(20,2) NOT NULL DEFAULT 0.00,\n    status VARCHAR(20) NOT NULL DEFAULT 'PENDING',\n    paid_at TIMESTAMP,\n    UNIQUE(loan_id, installment_no)\n)\n", "CREATE TABLE IF NOT EXISTS ecoxpert_professions (\n    player_uuid VARCHAR(36) PRIMARY KEY,\n    role VARCHAR(40) NOT NULL,\n    level INTEGER NOT NULL DEFAULT 1,\n    selected_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n)\n", "CREATE TABLE IF NOT EXISTS ecoxpert_profession_xp (\n    player_uuid VARCHAR(36) PRIMARY KEY,\n    xp INTEGER NOT NULL DEFAULT 0,\n    last_gain_at TIMESTAMP\n)\n", "CREATE TABLE IF NOT EXISTS ecoxpert_market_orders (\n    id INTEGER PRIMARY KEY AUTOINCREMENT,\n    seller_uuid VARCHAR(36) NOT NULL,\n    material VARCHAR(64) NOT NULL,\n    unit_price DECIMAL(20,2) NOT NULL,\n    remaining_quantity INTEGER NOT NULL,\n    status VARCHAR(16) NOT NULL DEFAULT 'OPEN',\n    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n    expires_at TIMESTAMP\n)\n"};
        try (Connection conn = this.dataSource.getConnection();){
            String[] indexes;
            for (String sql : tables) {
                try (PreparedStatement stmt = conn.prepareStatement(sql);){
                    stmt.executeUpdate();
                }
            }
            for (String idx : indexes = new String[]{"CREATE INDEX IF NOT EXISTS idx_loans_player_status ON ecoxpert_loans(player_uuid, status)", "CREATE INDEX IF NOT EXISTS idx_loan_sched_status_due ON ecoxpert_loan_schedules(status, due_date)", "CREATE INDEX IF NOT EXISTS idx_loan_sched_loan ON ecoxpert_loan_schedules(loan_id)", "CREATE INDEX IF NOT EXISTS idx_orders_status_material_created ON ecoxpert_market_orders(status, material, created_at)", "CREATE INDEX IF NOT EXISTS idx_orders_status_created ON ecoxpert_market_orders(status, created_at)", "CREATE INDEX IF NOT EXISTS idx_prof_xp_player ON ecoxpert_profession_xp(player_uuid)"}) {
                try (PreparedStatement stmt = conn.prepareStatement(idx);){
                    stmt.executeUpdate();
                }
                catch (Exception e) {
                    this.plugin.getLogger().fine("Index creation skipped/failed: " + idx + " -> " + e.getMessage());
                }
            }
            this.updateSchemaVersion();
        }
    }

    private void updateSchemaVersion() throws SQLException {
        String sql = "INSERT OR REPLACE INTO ecoxpert_meta (key, version)\nVALUES ('schema_version', ?)\n";
        try (Connection conn = this.dataSource.getConnection();
             PreparedStatement stmt = conn.prepareStatement(sql);){
            stmt.setInt(1, this.getCurrentSchemaVersion());
            stmt.executeUpdate();
        }
    }

    private int getCurrentSchemaVersion() {
        return 1;
    }

    private void setParameters(PreparedStatement stmt, Object ... params) throws SQLException {
        for (int i = 0; i < params.length; ++i) {
            stmt.setObject(i + 1, params[i]);
        }
    }

    private void checkConnection() {
        if (!this.connected) {
            throw new IllegalStateException("Database is not connected");
        }
    }

    @Override
    public boolean isHealthy() {
        boolean bl;
        block9: {
            if (!this.connected || this.dataSource == null) {
                return false;
            }
            Connection conn = this.dataSource.getConnection();
            try {
                bl = conn.isValid(5);
                if (conn == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (conn != null) {
                        try {
                            conn.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (SQLException e) {
                    return false;
                }
            }
            conn.close();
        }
        return bl;
    }

    @Override
    public CompletableFuture<Void> exportDatabase(Path backupPath) {
        return CompletableFuture.runAsync(() -> {
            try {
                this.plugin.getLogger().info("Creating database backup to: " + String.valueOf(backupPath));
                if (this.databaseType == DatabaseType.SQLITE) {
                    Files.copy(new File(this.plugin.getDataFolder(), "ecoxpert.db").toPath(), backupPath, StandardCopyOption.REPLACE_EXISTING);
                } else {
                    this.plugin.getLogger().warning("MySQL backup not fully implemented - using connection test");
                    if (!this.isHealthy()) {
                        throw new RuntimeException("Database is not healthy for backup");
                    }
                }
                this.plugin.getLogger().info("Database backup completed successfully");
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Database backup failed: " + e.getMessage());
                throw new RuntimeException("Database backup failed", e);
            }
        }, this.databaseExecutor);
    }

    @Override
    public CompletableFuture<Void> importDatabase(Path backupPath) {
        return CompletableFuture.runAsync(() -> {
            try {
                this.plugin.getLogger().info("Restoring database from: " + String.valueOf(backupPath));
                if (this.databaseType == DatabaseType.SQLITE) {
                    if (this.dataSource != null) {
                        this.dataSource.close();
                    }
                    Files.copy(backupPath, new File(this.plugin.getDataFolder(), "ecoxpert.db").toPath(), StandardCopyOption.REPLACE_EXISTING);
                    this.initializeSQLiteConnection();
                    this.testConnection();
                }
                this.plugin.getLogger().info("Database restore completed successfully");
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Database restore failed: " + e.getMessage());
                throw new RuntimeException("Database restore failed", e);
            }
        }, this.databaseExecutor);
    }

    @Override
    public CompletableFuture<Boolean> switchToFallback(String fallbackType) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                this.plugin.getLogger().warning("Switching to fallback database: " + fallbackType);
                if (this.dataSource != null) {
                    this.dataSource.close();
                    this.connected = false;
                }
                switch (fallbackType.toLowerCase()) {
                    case "sqlite": {
                        this.initializeSQLiteConnection();
                        break;
                    }
                    case "h2": {
                        this.initializeH2();
                        break;
                    }
                    case "memory": {
                        this.initializeMemoryDatabase();
                        break;
                    }
                    default: {
                        throw new IllegalArgumentException("Unknown fallback type: " + fallbackType);
                    }
                }
                this.createTables();
                this.connected = true;
                this.plugin.getLogger().info("Successfully switched to fallback database: " + fallbackType);
                return true;
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Failed to switch to fallback database: " + e.getMessage());
                return false;
            }
        }, this.databaseExecutor);
    }

    @Override
    public DatabaseStatus getStatus() {
        return DatabaseStatus.builder().currentType(this.databaseType != null ? this.databaseType.name().toLowerCase() : "unknown").originalType(this.databaseType != null ? this.databaseType.name().toLowerCase() : "unknown").connected(this.connected).healthy(this.isHealthy()).usingFallback(false).connectionPoolSize(this.dataSource != null ? this.dataSource.getHikariPoolMXBean().getTotalConnections() : 0).activeConnections(this.dataSource != null ? this.dataSource.getHikariPoolMXBean().getActiveConnections() : 0).build();
    }

    private void initializeSQLiteConnection() throws SQLException {
        HikariConfig config = new HikariConfig();
        this.configureSQLite(config);
        this.dataSource = new HikariDataSource(config);
        this.databaseType = DatabaseType.SQLITE;
    }

    private void initializeH2() throws SQLException {
        HikariConfig config = new HikariConfig();
        config.setJdbcUrl("jdbc:h2:" + this.plugin.getDataFolder().getAbsolutePath() + "/database_h2");
        config.setDriverClassName("org.h2.Driver");
        config.setMaximumPoolSize(10);
        config.setMinimumIdle(1);
        config.setConnectionTimeout(30000L);
        config.setIdleTimeout(600000L);
        config.setMaxLifetime(1800000L);
        this.dataSource = new HikariDataSource(config);
        this.databaseType = DatabaseType.H2;
    }

    private void initializeMemoryDatabase() throws SQLException {
        HikariConfig config = new HikariConfig();
        config.setJdbcUrl("jdbc:h2:mem:ecoxpert;DB_CLOSE_DELAY=-1");
        config.setDriverClassName("org.h2.Driver");
        config.setMaximumPoolSize(5);
        config.setMinimumIdle(1);
        this.dataSource = new HikariDataSource(config);
        this.databaseType = DatabaseType.MEMORY;
    }

    private static enum DatabaseType {
        SQLITE,
        MYSQL,
        H2,
        MEMORY;

    }
}

