/*
 * Decompiled with CFR 0.152.
 */
package me.koyere.ecoxpert.core.dependencies;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import me.koyere.ecoxpert.EcoXpertPlugin;
import me.koyere.ecoxpert.core.dependencies.DependencyInfo;
import me.koyere.ecoxpert.core.dependencies.DependencyManager;
import me.koyere.ecoxpert.core.dependencies.DependencyResult;
import me.koyere.ecoxpert.core.dependencies.DependencyState;

public class DependencyManagerImpl
implements DependencyManager {
    private final EcoXpertPlugin plugin;
    private final ExecutorService executorService;
    private final ScheduledExecutorService healthCheckService;
    private final Map<String, CircuitBreakerState> circuitBreakers = new ConcurrentHashMap<String, CircuitBreakerState>();
    private final Map<String, DependencyState> dependencyStates = new ConcurrentHashMap<String, DependencyState>();
    private final Map<String, ClassLoader> dependencyClassLoaders = new ConcurrentHashMap<String, ClassLoader>();
    private final Map<String, DependencyInfo> dependencyRegistry = new ConcurrentHashMap<String, DependencyInfo>();
    private final Map<String, Instant> lastFailureTime = new ConcurrentHashMap<String, Instant>();
    private static final int MAX_RETRIES = 3;
    private static final Duration CIRCUIT_BREAKER_TIMEOUT = Duration.ofMinutes(5L);
    private static final Duration HEALTH_CHECK_INTERVAL = Duration.ofSeconds(30L);
    private static final Duration DOWNLOAD_TIMEOUT = Duration.ofMinutes(2L);
    private Path libsDirectory;
    private volatile boolean healthy = true;

    public DependencyManagerImpl(EcoXpertPlugin plugin) {
        this.plugin = plugin;
        this.executorService = Executors.newFixedThreadPool(3, r -> {
            Thread t = new Thread(r, "EcoXpert-DependencyManager");
            t.setDaemon(true);
            return t;
        });
        this.healthCheckService = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "EcoXpert-HealthCheck");
            t.setDaemon(true);
            return t;
        });
    }

    @Override
    public void initialize() {
        this.plugin.getLogger().info("Initializing dependency management system...");
        this.libsDirectory = this.plugin.getDataFolder().toPath().resolve("libs");
        try {
            Files.createDirectories(this.libsDirectory, new FileAttribute[0]);
        }
        catch (IOException e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to create libs directory", e);
            throw new RuntimeException("Cannot initialize dependency manager", e);
        }
        this.registerKnownDependencies();
        this.healthCheckService.scheduleWithFixedDelay(this::runHealthCheck, HEALTH_CHECK_INTERVAL.toSeconds(), HEALTH_CHECK_INTERVAL.toSeconds(), TimeUnit.SECONDS);
        this.plugin.getLogger().info("Dependency management system initialized successfully");
    }

    @Override
    public void shutdown() {
        this.plugin.getLogger().info("Shutting down dependency management system...");
        this.healthCheckService.shutdown();
        this.executorService.shutdown();
        try {
            if (!this.executorService.awaitTermination(10L, TimeUnit.SECONDS)) {
                this.executorService.shutdownNow();
            }
            if (!this.healthCheckService.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.healthCheckService.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.executorService.shutdownNow();
            this.healthCheckService.shutdownNow();
        }
        this.dependencyClassLoaders.values().forEach(classLoader -> {
            if (classLoader instanceof URLClassLoader) {
                try {
                    ((URLClassLoader)classLoader).close();
                }
                catch (IOException e) {
                    this.plugin.getLogger().log(Level.WARNING, "Failed to close class loader", e);
                }
            }
        });
        this.plugin.getLogger().info("Dependency management system shut down successfully");
    }

    @Override
    public CompletableFuture<DependencyResult> ensureDependency(String dependencyName) {
        DependencyInfo info = this.dependencyRegistry.get(dependencyName);
        if (info == null) {
            return CompletableFuture.completedFuture(DependencyResult.failure(DependencyState.MISSING, "Unknown dependency: " + dependencyName));
        }
        CircuitBreakerState circuitState = this.circuitBreakers.get(dependencyName);
        if (circuitState == CircuitBreakerState.OPEN) {
            Instant lastFailure = this.lastFailureTime.get(dependencyName);
            if (lastFailure != null && Instant.now().isBefore(lastFailure.plus(CIRCUIT_BREAKER_TIMEOUT))) {
                return this.attemptFallback(info);
            }
            this.circuitBreakers.put(dependencyName, CircuitBreakerState.HALF_OPEN);
        }
        return CompletableFuture.supplyAsync(() -> {
            try {
                return this.downloadAndVerifyDependency(info);
            }
            catch (Exception e) {
                this.recordFailure(dependencyName);
                this.plugin.getLogger().log(Level.WARNING, "Failed to ensure dependency " + dependencyName + ", attempting fallback", e);
                return this.attemptFallback(info).join();
            }
        }, this.executorService);
    }

    @Override
    public CompletableFuture<Void> ensureAllDependencies(Set<String> dependencyNames) {
        List<CompletableFuture> futures = dependencyNames.stream().map(this::ensureDependency).toList();
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).thenRun(() -> {
            ArrayList<String> failures = new ArrayList<String>();
            for (int i = 0; i < futures.size(); ++i) {
                String depName;
                DependencyInfo info;
                DependencyResult result = (DependencyResult)((CompletableFuture)futures.get(i)).join();
                if (result.isSuccess() || (info = this.dependencyRegistry.get(depName = dependencyNames.toArray(new String[0])[i])) == null || !info.isRequired()) continue;
                failures.add(depName);
            }
            if (!failures.isEmpty()) {
                this.healthy = false;
                this.plugin.getLogger().severe("Critical dependencies failed: " + String.valueOf(failures));
                throw new RuntimeException("Critical dependencies unavailable: " + String.valueOf(failures));
            }
        });
    }

    @Override
    public DependencyState getDependencyState(String dependencyName) {
        return this.dependencyStates.getOrDefault(dependencyName, DependencyState.MISSING);
    }

    @Override
    public boolean isDependencyAvailable(String dependencyName) {
        DependencyState state = this.getDependencyState(dependencyName);
        return state == DependencyState.LOADED || state == DependencyState.FALLBACK;
    }

    @Override
    public boolean isUsingFallback(String dependencyName) {
        return this.getDependencyState(dependencyName) == DependencyState.FALLBACK;
    }

    @Override
    public ClassLoader getDependencyClassLoader(String dependencyName) {
        return this.dependencyClassLoaders.get(dependencyName);
    }

    @Override
    public CompletableFuture<Void> performHealthCheck() {
        return CompletableFuture.runAsync(this::runHealthCheck, this.executorService);
    }

    @Override
    public boolean isHealthy() {
        return this.healthy;
    }

    @Override
    public CompletableFuture<DependencyResult> forceRefresh(String dependencyName) {
        DependencyInfo info;
        this.dependencyStates.remove(dependencyName);
        CircuitBreakerState circuitState = this.circuitBreakers.get(dependencyName);
        if (circuitState != null) {
            this.circuitBreakers.put(dependencyName, CircuitBreakerState.CLOSED);
        }
        if ((info = this.dependencyRegistry.get(dependencyName)) != null) {
            Path cachedFile = this.libsDirectory.resolve(info.getFileName());
            try {
                Files.deleteIfExists(cachedFile);
            }
            catch (IOException e) {
                this.plugin.getLogger().log(Level.WARNING, "Failed to delete cached dependency", e);
            }
        }
        return this.ensureDependency(dependencyName);
    }

    private void registerKnownDependencies() {
        this.dependencyRegistry.put("sqlite", new DependencyInfo("sqlite", "org.xerial", "sqlite-jdbc", "3.44.1.0", "https://repo1.maven.org/maven2/org/xerial/sqlite-jdbc/3.44.1.0/sqlite-jdbc-3.44.1.0.jar", "ab0b5e04e81bb0b4eb3bbad624a75dde4cea52cfadf4e4c9f94bd6570147bcd3", false, "org.h2.Driver"));
        this.dependencyRegistry.put("mysql", new DependencyInfo("mysql", "com.mysql", "mysql-connector-j", "8.0.33", "https://repo1.maven.org/maven2/com/mysql/mysql-connector-j/8.0.33/mysql-connector-j-8.0.33.jar", "4b2d94b6d30ea5ca2fe76b0b1b3fe2c99e8e1e1ba0b2d41b8d88b63b3b3e4b3", false, "org.h2.Driver"));
        this.plugin.getLogger().info("Registered " + this.dependencyRegistry.size() + " known dependencies");
    }

    private DependencyResult downloadAndVerifyDependency(DependencyInfo info) {
        String dependencyName = info.getName();
        try {
            this.dependencyStates.put(dependencyName, DependencyState.DOWNLOADING);
            Path targetFile = this.libsDirectory.resolve(info.getFileName());
            if (Files.exists(targetFile, new LinkOption[0]) && this.verifyChecksum(targetFile, info.getSha256Checksum())) {
                ClassLoader classLoader = this.createClassLoader(targetFile);
                this.dependencyClassLoaders.put(dependencyName, classLoader);
                this.dependencyStates.put(dependencyName, DependencyState.LOADED);
                this.circuitBreakers.put(dependencyName, CircuitBreakerState.CLOSED);
                this.plugin.getLogger().info("Dependency " + dependencyName + " already available and verified");
                return DependencyResult.success(DependencyState.LOADED, "Already available");
            }
            for (int attempt = 1; attempt <= 3; ++attempt) {
                try {
                    this.plugin.getLogger().info("Downloading " + dependencyName + " (attempt " + attempt + "/3)");
                    this.downloadFile(info.getDownloadUrl(), targetFile);
                    if (this.verifyChecksum(targetFile, info.getSha256Checksum())) {
                        ClassLoader classLoader = this.createClassLoader(targetFile);
                        this.dependencyClassLoaders.put(dependencyName, classLoader);
                        this.dependencyStates.put(dependencyName, DependencyState.LOADED);
                        this.circuitBreakers.put(dependencyName, CircuitBreakerState.CLOSED);
                        this.plugin.getLogger().info("Successfully downloaded and verified " + dependencyName);
                        return DependencyResult.success(DependencyState.LOADED, "Downloaded and verified");
                    }
                    this.dependencyStates.put(dependencyName, DependencyState.CORRUPTED);
                    Files.deleteIfExists(targetFile);
                    if (attempt == 3) {
                        this.recordFailure(dependencyName);
                        return DependencyResult.failure(DependencyState.CORRUPTED, "Checksum verification failed after 3 attempts");
                    }
                    Thread.sleep(1000L * (long)attempt);
                    continue;
                }
                catch (Exception e) {
                    if (attempt == 3) {
                        this.recordFailure(dependencyName);
                        throw e;
                    }
                    Thread.sleep(1000L * (long)attempt);
                }
            }
            this.recordFailure(dependencyName);
            return DependencyResult.failure(DependencyState.FAILED, "Max retries exceeded");
        }
        catch (Exception e) {
            this.recordFailure(dependencyName);
            this.dependencyStates.put(dependencyName, DependencyState.FAILED);
            return DependencyResult.failure(DependencyState.FAILED, e.getMessage(), e);
        }
    }

    private CompletableFuture<DependencyResult> attemptFallback(DependencyInfo info) {
        String dependencyName = info.getName();
        if (info.getFallbackClass() != null) {
            try {
                Class.forName(info.getFallbackClass());
                this.dependencyStates.put(dependencyName, DependencyState.FALLBACK);
                this.plugin.getLogger().info("Using fallback for " + dependencyName + ": " + info.getFallbackClass());
                return CompletableFuture.completedFuture(DependencyResult.fallback("Using fallback: " + info.getFallbackClass()));
            }
            catch (ClassNotFoundException e) {
                this.plugin.getLogger().warning("Fallback class not available: " + info.getFallbackClass());
            }
        }
        if (info.isRequired()) {
            return CompletableFuture.completedFuture(DependencyResult.failure(DependencyState.FAILED, "Required dependency unavailable and no fallback"));
        }
        this.dependencyStates.put(dependencyName, DependencyState.FALLBACK);
        return CompletableFuture.completedFuture(DependencyResult.fallback("Optional dependency unavailable, continuing without it"));
    }

    private void downloadFile(String url, Path targetFile) throws IOException, InterruptedException {
        CompletableFuture<Void> downloadFuture = CompletableFuture.runAsync(() -> {
            try (InputStream in = new URL(url).openStream();
                 FileOutputStream out = new FileOutputStream(targetFile.toFile());){
                int bytesRead;
                byte[] buffer = new byte[8192];
                while ((bytesRead = in.read(buffer)) != -1) {
                    out.write(buffer, 0, bytesRead);
                }
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }, this.executorService);
        try {
            downloadFuture.get(DOWNLOAD_TIMEOUT.toSeconds(), TimeUnit.SECONDS);
        }
        catch (TimeoutException e) {
            downloadFuture.cancel(true);
            throw new IOException("Download timeout after " + DOWNLOAD_TIMEOUT.toSeconds() + " seconds", e);
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof RuntimeException && e.getCause().getCause() instanceof IOException) {
                throw (IOException)e.getCause().getCause();
            }
            throw new IOException("Download failed", e);
        }
    }

    private boolean verifyChecksum(Path file, String expectedSha256) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] fileBytes = Files.readAllBytes(file);
            byte[] hashBytes = digest.digest(fileBytes);
            StringBuilder hexString = new StringBuilder();
            for (byte b : hashBytes) {
                String hex = Integer.toHexString(0xFF & b);
                if (hex.length() == 1) {
                    hexString.append('0');
                }
                hexString.append(hex);
            }
            return hexString.toString().equals(expectedSha256.toLowerCase());
        }
        catch (Exception e) {
            this.plugin.getLogger().log(Level.WARNING, "Failed to verify checksum", e);
            return false;
        }
    }

    private ClassLoader createClassLoader(Path jarFile) throws IOException {
        URL jarUrl = jarFile.toUri().toURL();
        return new URLClassLoader(new URL[]{jarUrl}, ((Object)((Object)this.plugin)).getClass().getClassLoader());
    }

    private void recordFailure(String dependencyName) {
        this.lastFailureTime.put(dependencyName, Instant.now());
        this.circuitBreakers.put(dependencyName, CircuitBreakerState.OPEN);
        this.plugin.getLogger().warning("Circuit breaker opened for dependency: " + dependencyName);
    }

    private void runHealthCheck() {
        try {
            boolean allHealthy = true;
            for (Map.Entry<String, DependencyInfo> entry : this.dependencyRegistry.entrySet()) {
                String name = entry.getKey();
                DependencyInfo info = entry.getValue();
                if (!info.isRequired() || this.isDependencyAvailable(name)) continue;
                allHealthy = false;
                this.plugin.getLogger().warning("Health check failed for required dependency: " + name);
            }
            this.healthy = allHealthy;
            if (!this.healthy) {
                this.plugin.getLogger().warning("Dependency system is unhealthy - some required dependencies are unavailable");
            }
        }
        catch (Exception e) {
            this.plugin.getLogger().log(Level.WARNING, "Health check failed", e);
            this.healthy = false;
        }
    }

    private static enum CircuitBreakerState {
        CLOSED,
        OPEN,
        HALF_OPEN;

    }
}

