/*
 * Decompiled with CFR 0.152.
 */
package me.koyere.ecoxpert.core.failsafe;

import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import javax.inject.Inject;
import javax.inject.Singleton;
import me.koyere.ecoxpert.EcoXpertPlugin;
import me.koyere.ecoxpert.core.data.DataManager;
import me.koyere.ecoxpert.core.failsafe.EconomyFailsafeManager;
import me.koyere.ecoxpert.core.failsafe.FailsafeHealth;
import me.koyere.ecoxpert.core.failsafe.RepairResult;
import me.koyere.ecoxpert.core.failsafe.TransactionHistory;
import me.koyere.ecoxpert.core.failsafe.TransactionRecord;
import me.koyere.ecoxpert.core.failsafe.ValidationResult;

@Singleton
public class EconomyFailsafeManagerImpl
implements EconomyFailsafeManager {
    private final EcoXpertPlugin plugin;
    private final DataManager dataManager;
    private final ExecutorService executorService;
    private final ScheduledExecutorService scheduledService;
    private final AtomicBoolean readOnlyMode = new AtomicBoolean(false);
    private final AtomicReference<String> readOnlyReason = new AtomicReference();
    private final Map<String, CheckpointData> checkpoints = new ConcurrentHashMap<String, CheckpointData>();
    private final Queue<TransactionRecord> transactionQueue = new ConcurrentLinkedQueue<TransactionRecord>();
    private static final int MAX_BACKUPS = 10;
    private static final int MAX_TRANSACTION_QUEUE = 10000;
    private static final long BACKUP_INTERVAL_HOURS = 6L;
    private static final long HEALTH_CHECK_INTERVAL_MINUTES = 5L;
    private Path backupDirectory;
    private Path transactionLogFile;
    private volatile boolean initialized = false;
    private volatile FailsafeHealth lastHealthStatus;

    @Inject
    public EconomyFailsafeManagerImpl(EcoXpertPlugin plugin, DataManager dataManager) {
        this.plugin = plugin;
        this.dataManager = dataManager;
        this.executorService = Executors.newFixedThreadPool(2, r -> {
            Thread t = new Thread(r, "EcoXpert-Failsafe");
            t.setDaemon(true);
            return t;
        });
        this.scheduledService = Executors.newScheduledThreadPool(1, r -> {
            Thread t = new Thread(r, "EcoXpert-FailsafeScheduler");
            t.setDaemon(true);
            return t;
        });
    }

    @Override
    public void initialize() {
        if (this.initialized) {
            return;
        }
        try {
            this.plugin.getLogger().info("Initializing economy failsafe system...");
            this.backupDirectory = this.plugin.getDataFolder().toPath().resolve("backups");
            Files.createDirectories(this.backupDirectory, new FileAttribute[0]);
            this.transactionLogFile = this.plugin.getDataFolder().toPath().resolve("transaction.log");
            this.cleanOldBackups();
            this.scheduledService.scheduleAtFixedRate(this::performScheduledBackup, 6L, 6L, TimeUnit.HOURS);
            this.scheduledService.scheduleAtFixedRate(this::performScheduledHealthCheck, 5L, 5L, TimeUnit.MINUTES);
            this.scheduledService.scheduleAtFixedRate(this::flushTransactionLog, 30L, 30L, TimeUnit.SECONDS);
            this.performHealthCheck().join();
            this.initialized = true;
            this.plugin.getLogger().info("Economy failsafe system initialized successfully");
        }
        catch (Exception e) {
            this.plugin.getLogger().log(Level.SEVERE, "Failed to initialize failsafe system", e);
            throw new RuntimeException("Failsafe initialization failed", e);
        }
    }

    @Override
    public void shutdown() {
        if (!this.initialized) {
            return;
        }
        this.plugin.getLogger().info("Shutting down economy failsafe system...");
        this.flushTransactionLog();
        this.scheduledService.shutdown();
        this.executorService.shutdown();
        try {
            if (!this.executorService.awaitTermination(10L, TimeUnit.SECONDS)) {
                this.executorService.shutdownNow();
            }
            if (!this.scheduledService.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.scheduledService.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.executorService.shutdownNow();
            this.scheduledService.shutdownNow();
        }
        this.initialized = false;
        this.plugin.getLogger().info("Economy failsafe system shut down successfully");
    }

    @Override
    public CompletableFuture<Void> createBackup(String reason) {
        return CompletableFuture.runAsync(() -> {
            try {
                String timestamp = DateTimeFormatter.ofPattern("yyyy-MM-dd_HH-mm-ss").format(Instant.now().atZone(ZoneOffset.UTC));
                String backupName = "economy_" + timestamp + ".backup";
                Path backupFile = this.backupDirectory.resolve(backupName);
                this.plugin.getLogger().info("Creating economy backup: " + backupName + " (Reason: " + reason + ")");
                this.dataManager.exportDatabase(backupFile).join();
                Path metadataFile = this.backupDirectory.resolve(backupName + ".meta");
                Properties metadata = new Properties();
                metadata.setProperty("timestamp", Instant.now().toString());
                metadata.setProperty("reason", reason);
                metadata.setProperty("plugin_version", this.plugin.getDescription().getVersion());
                try (FileOutputStream out = new FileOutputStream(metadataFile.toFile());){
                    metadata.store(out, "EcoXpert Backup Metadata");
                }
                this.cleanOldBackups();
                this.plugin.getLogger().info("Economy backup created successfully: " + backupName);
            }
            catch (Exception e) {
                this.plugin.getLogger().log(Level.SEVERE, "Failed to create backup", e);
                throw new RuntimeException("Backup creation failed", e);
            }
        }, this.executorService);
    }

    @Override
    public CompletableFuture<Void> restoreFromBackup() {
        return CompletableFuture.runAsync(() -> {
            try {
                Optional<Path> latestBackup = this.findLatestBackup();
                if (latestBackup.isEmpty()) {
                    throw new RuntimeException("No backups available for restore");
                }
                this.restoreFromBackupFile(latestBackup.get());
            }
            catch (Exception e) {
                this.plugin.getLogger().log(Level.SEVERE, "Failed to restore from backup", e);
                throw new RuntimeException("Backup restore failed", e);
            }
        }, this.executorService);
    }

    @Override
    public CompletableFuture<Void> restoreFromBackup(String backupId) {
        return CompletableFuture.runAsync(() -> {
            try {
                Path backupFile = this.backupDirectory.resolve(backupId);
                if (!Files.exists(backupFile, new LinkOption[0])) {
                    throw new RuntimeException("Backup not found: " + backupId);
                }
                this.restoreFromBackupFile(backupFile);
            }
            catch (Exception e) {
                this.plugin.getLogger().log(Level.SEVERE, "Failed to restore from specific backup", e);
                throw new RuntimeException("Backup restore failed", e);
            }
        }, this.executorService);
    }

    @Override
    public CompletableFuture<ValidationResult> performConsistencyCheck() {
        return CompletableFuture.supplyAsync(() -> {
            try {
                ArrayList<String> errors = new ArrayList<String>();
                ArrayList<String> warnings = new ArrayList<String>();
                if (!this.dataManager.isHealthy()) {
                    errors.add("Database connectivity issues detected");
                }
                if (!Files.exists(this.transactionLogFile, new LinkOption[0])) {
                    warnings.add("Transaction log file not found");
                } else {
                    try {
                        Files.size(this.transactionLogFile);
                    }
                    catch (IOException e) {
                        errors.add("Transaction log file is corrupted or unreadable");
                    }
                }
                try {
                    List<Path> backups = this.findAllBackups();
                    if (backups.isEmpty()) {
                        warnings.add("No backups available");
                    }
                }
                catch (Exception e) {
                    errors.add("Backup directory issues: " + e.getMessage());
                }
                boolean valid = errors.isEmpty();
                ValidationResult.ValidationSeverity severity = !errors.isEmpty() ? ValidationResult.ValidationSeverity.CRITICAL : (!warnings.isEmpty() ? ValidationResult.ValidationSeverity.WARNING : ValidationResult.ValidationSeverity.HEALTHY);
                String summary = String.format("Consistency check completed: %d errors, %d warnings", errors.size(), warnings.size());
                return new ValidationResult(valid, errors, warnings, severity, summary);
            }
            catch (Exception e) {
                this.plugin.getLogger().log(Level.SEVERE, "Consistency check failed", e);
                return new ValidationResult(false, List.of("Consistency check failed: " + e.getMessage()), List.of(), ValidationResult.ValidationSeverity.CRITICAL, "Consistency check failed due to system error");
            }
        }, this.executorService);
    }

    @Override
    public void enterReadOnlyMode(String reason) {
        if (this.readOnlyMode.compareAndSet(false, true)) {
            this.readOnlyReason.set(reason);
            this.plugin.getLogger().warning("ECONOMY ENTERING READ-ONLY MODE: " + reason);
            this.createBackup("Emergency backup before read-only mode: " + reason);
        }
    }

    @Override
    public void exitReadOnlyMode() {
        if (this.readOnlyMode.compareAndSet(true, false)) {
            String reason = this.readOnlyReason.getAndSet(null);
            this.plugin.getLogger().info("Economy exiting read-only mode (was: " + reason + ")");
        }
    }

    @Override
    public boolean isReadOnlyMode() {
        return this.readOnlyMode.get();
    }

    @Override
    public String getReadOnlyReason() {
        return this.readOnlyReason.get();
    }

    @Override
    public CompletableFuture<Void> recordTransaction(TransactionRecord transaction) {
        return CompletableFuture.runAsync(() -> {
            this.transactionQueue.offer(transaction);
            while (this.transactionQueue.size() > 10000) {
                this.transactionQueue.poll();
            }
        }, this.executorService);
    }

    @Override
    public CompletableFuture<TransactionHistory> getTransactionHistory(UUID playerUuid, int limit) {
        return CompletableFuture.completedFuture(new TransactionHistory(playerUuid, List.of(), 0, false));
    }

    @Override
    public CompletableFuture<RepairResult> detectAndRepairCorruption() {
        return CompletableFuture.supplyAsync(() -> {
            ArrayList<String> issuesFound = new ArrayList<String>();
            ArrayList<String> actionsPerformed = new ArrayList<String>();
            boolean corruptionDetected = false;
            boolean repairSuccessful = true;
            try {
                ValidationResult validation = this.performConsistencyCheck().join();
                if (!validation.isValid()) {
                    corruptionDetected = true;
                    issuesFound.addAll(validation.getErrors());
                    if (validation.getSeverity() == ValidationResult.ValidationSeverity.CRITICAL) {
                        this.plugin.getLogger().warning("Critical corruption detected, entering read-only mode");
                        this.enterReadOnlyMode("Automatic corruption detection");
                        actionsPerformed.add("Entered read-only mode due to critical corruption");
                        try {
                            Optional<Path> backup = this.findLatestBackup();
                            if (backup.isPresent()) {
                                this.plugin.getLogger().info("Attempting automatic restore from latest backup");
                                this.restoreFromBackupFile(backup.get());
                                actionsPerformed.add("Restored from latest backup: " + String.valueOf(backup.get().getFileName()));
                                this.exitReadOnlyMode();
                                actionsPerformed.add("Exited read-only mode after successful restore");
                            } else {
                                repairSuccessful = false;
                                actionsPerformed.add("No backups available for automatic restore");
                            }
                        }
                        catch (Exception e) {
                            repairSuccessful = false;
                            actionsPerformed.add("Automatic restore failed: " + e.getMessage());
                        }
                    }
                }
                String summary = corruptionDetected ? (repairSuccessful ? "Corruption detected and repaired" : "Corruption detected but repair failed") : "No corruption detected";
                return new RepairResult(corruptionDetected, repairSuccessful, issuesFound, actionsPerformed, summary);
            }
            catch (Exception e) {
                this.plugin.getLogger().log(Level.SEVERE, "Corruption detection failed", e);
                return new RepairResult(false, false, List.of("Corruption detection failed: " + e.getMessage()), List.of(), "Detection process failed");
            }
        }, this.executorService);
    }

    @Override
    public CompletableFuture<String> createCheckpoint(String operationId) {
        return CompletableFuture.supplyAsync(() -> {
            String checkpointId = UUID.randomUUID().toString();
            CheckpointData checkpoint = new CheckpointData(operationId, Instant.now());
            this.checkpoints.put(checkpointId, checkpoint);
            if (this.checkpoints.size() > 100) {
                this.checkpoints.entrySet().removeIf(entry -> ((CheckpointData)entry.getValue()).timestamp.isBefore(Instant.now().minusSeconds(3600L)));
            }
            return checkpointId;
        }, this.executorService);
    }

    @Override
    public CompletableFuture<Void> rollbackToCheckpoint(String checkpointId) {
        return CompletableFuture.runAsync(() -> {
            CheckpointData checkpoint = this.checkpoints.get(checkpointId);
            if (checkpoint == null) {
                throw new RuntimeException("Checkpoint not found: " + checkpointId);
            }
            this.plugin.getLogger().info("Rolling back to checkpoint: " + checkpointId + " (Operation: " + checkpoint.operationId + ")");
        }, this.executorService);
    }

    @Override
    public CompletableFuture<Void> commitCheckpoint(String checkpointId) {
        return CompletableFuture.runAsync(() -> this.checkpoints.remove(checkpointId), this.executorService);
    }

    @Override
    public FailsafeHealth getHealthStatus() {
        return this.lastHealthStatus != null ? this.lastHealthStatus : new FailsafeHealth(false, Instant.EPOCH, Map.of(), "Not initialized");
    }

    @Override
    public CompletableFuture<FailsafeHealth> performHealthCheck() {
        return CompletableFuture.supplyAsync(() -> {
            FailsafeHealth health;
            HashMap<String, String> metrics = new HashMap<String, String>();
            boolean healthy = true;
            Object status = "Healthy";
            try {
                if (!this.initialized) {
                    healthy = false;
                    status = "Not initialized";
                } else {
                    if (this.isReadOnlyMode()) {
                        healthy = false;
                        status = "Read-only mode: " + this.getReadOnlyReason();
                    }
                    boolean dbHealthy = this.dataManager.isHealthy();
                    metrics.put("database_healthy", String.valueOf(dbHealthy));
                    if (!dbHealthy) {
                        healthy = false;
                        status = "Database unhealthy";
                    }
                    boolean backupDirExists = Files.exists(this.backupDirectory, new LinkOption[0]) && Files.isDirectory(this.backupDirectory, new LinkOption[0]);
                    metrics.put("backup_directory_exists", String.valueOf(backupDirExists));
                    if (!backupDirExists) {
                        healthy = false;
                        status = "Backup directory missing";
                    }
                    int queueSize = this.transactionQueue.size();
                    metrics.put("transaction_queue_size", String.valueOf(queueSize));
                    if ((double)queueSize > 8000.0) {
                        status = "Transaction queue near capacity";
                    }
                    int checkpointCount = this.checkpoints.size();
                    metrics.put("active_checkpoints", String.valueOf(checkpointCount));
                }
            }
            catch (Exception e) {
                healthy = false;
                status = "Health check failed: " + e.getMessage();
                this.plugin.getLogger().log(Level.WARNING, "Failsafe health check failed", e);
            }
            this.lastHealthStatus = health = new FailsafeHealth(healthy, Instant.now(), metrics, (String)status);
            return health;
        }, this.executorService);
    }

    private void restoreFromBackupFile(Path backupFile) throws Exception {
        this.plugin.getLogger().warning("RESTORING ECONOMY FROM BACKUP: " + String.valueOf(backupFile.getFileName()));
        this.enterReadOnlyMode("Restore operation in progress");
        try {
            this.dataManager.importDatabase(backupFile).join();
            this.plugin.getLogger().info("Economy restore completed successfully");
        }
        finally {
            this.exitReadOnlyMode();
        }
    }

    private Optional<Path> findLatestBackup() throws IOException {
        return Files.list(this.backupDirectory).filter(path -> path.getFileName().toString().endsWith(".backup")).max(Comparator.comparing(path -> {
            try {
                return Files.getLastModifiedTime(path, new LinkOption[0]);
            }
            catch (IOException e) {
                return FileTime.fromMillis(0L);
            }
        }));
    }

    private List<Path> findAllBackups() throws IOException {
        return Files.list(this.backupDirectory).filter(path -> path.getFileName().toString().endsWith(".backup")).sorted(Comparator.comparing(path -> {
            try {
                return Files.getLastModifiedTime(path, new LinkOption[0]);
            }
            catch (IOException e) {
                return FileTime.fromMillis(0L);
            }
        })).toList();
    }

    private void cleanOldBackups() {
        try {
            List<Path> backups = this.findAllBackups();
            if (backups.size() > 10) {
                List<Path> toDelete = backups.subList(0, backups.size() - 10);
                for (Path backup : toDelete) {
                    Files.deleteIfExists(backup);
                    Files.deleteIfExists(this.backupDirectory.resolve(String.valueOf(backup.getFileName()) + ".meta"));
                }
                this.plugin.getLogger().info("Cleaned " + toDelete.size() + " old backups");
            }
        }
        catch (Exception e) {
            this.plugin.getLogger().log(Level.WARNING, "Failed to clean old backups", e);
        }
    }

    private void performScheduledBackup() {
        try {
            this.createBackup("Scheduled automatic backup").join();
        }
        catch (Exception e) {
            this.plugin.getLogger().log(Level.WARNING, "Scheduled backup failed", e);
        }
    }

    private void performScheduledHealthCheck() {
        try {
            FailsafeHealth health = this.performHealthCheck().join();
            if (!health.isHealthy()) {
                this.plugin.getLogger().warning("Failsafe health check failed: " + health.getStatus());
            }
        }
        catch (Exception e) {
            this.plugin.getLogger().log(Level.WARNING, "Scheduled health check failed", e);
        }
    }

    private void flushTransactionLog() {
        if (this.transactionQueue.isEmpty()) {
            return;
        }
        try (PrintWriter writer = new PrintWriter(new FileWriter(this.transactionLogFile.toFile(), true));){
            TransactionRecord transaction;
            while ((transaction = this.transactionQueue.poll()) != null) {
                writer.println(String.format("{\"id\":\"%s\",\"timestamp\":\"%s\",\"type\":\"%s\",\"amount\":\"%s\",\"success\":%b}", transaction.getTransactionId(), transaction.getTimestamp().toString(), transaction.getType(), transaction.getAmount().toString(), transaction.isSuccessful()));
            }
        }
        catch (IOException e) {
            this.plugin.getLogger().log(Level.WARNING, "Failed to flush transaction log", e);
        }
    }

    private static class CheckpointData {
        final String operationId;
        final Instant timestamp;

        CheckpointData(String operationId, Instant timestamp) {
            this.operationId = operationId;
            this.timestamp = timestamp;
        }
    }
}

