/*
 * Decompiled with CFR 0.152.
 */
package me.koyere.ecoxpert.modules.inflation;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.function.Function;
import me.koyere.ecoxpert.modules.inflation.EconomicIntelligenceEngine;
import me.koyere.ecoxpert.modules.inflation.EconomicSnapshot;

public class EconomicMemory {
    private final ConcurrentLinkedQueue<EconomicSnapshot> historicalSnapshots = new ConcurrentLinkedQueue();
    private final ConcurrentLinkedQueue<CycleTransition> cycleHistory = new ConcurrentLinkedQueue();
    private final ConcurrentLinkedQueue<EconomicIntervention> interventionHistory = new ConcurrentLinkedQueue();
    private static final int MAX_SNAPSHOTS = 2880;
    private static final int MAX_CYCLES = 100;
    private static final int MAX_INTERVENTIONS = 200;
    private double cachedHealthTrend = 0.0;
    private double cachedInflationTrend = 0.0;
    private double cachedActivityTrend = 0.0;
    private LocalDateTime lastTrendCalculation = LocalDateTime.now().minusHours(1L);
    private List<EconomicPattern> identifiedPatterns = new ArrayList<EconomicPattern>();
    private EconomicForecast currentForecast = null;

    public void recordSnapshot(EconomicSnapshot snapshot) {
        this.historicalSnapshots.offer(snapshot);
        while (this.historicalSnapshots.size() > 2880) {
            this.historicalSnapshots.poll();
        }
        this.invalidateTrendCache();
        if (this.historicalSnapshots.size() % 12 == 0) {
            this.updatePatternRecognition();
        }
        if (this.historicalSnapshots.size() % 48 == 0) {
            this.updateEconomicForecast();
        }
    }

    public void recordCycleChange(EconomicIntelligenceEngine.EconomicCycle newCycle, EconomicSnapshot snapshot) {
        CycleTransition transition = new CycleTransition(LocalDateTime.now(), snapshot.getCycle(), newCycle, snapshot.getEconomicHealth(), snapshot.getInflationRate(), this.getCycleDuration(snapshot.getCycle()));
        this.cycleHistory.offer(transition);
        while (this.cycleHistory.size() > 100) {
            this.cycleHistory.poll();
        }
    }

    public void recordIntervention(EconomicIntervention intervention) {
        this.interventionHistory.offer(intervention);
        while (this.interventionHistory.size() > 200) {
            this.interventionHistory.poll();
        }
    }

    public void loadHistoricalData() {
    }

    public double getHealthTrend() {
        if (this.shouldRecalculateTrends()) {
            this.recalculateTrends();
        }
        return this.cachedHealthTrend;
    }

    public double getInflationTrend() {
        if (this.shouldRecalculateTrends()) {
            this.recalculateTrends();
        }
        return this.cachedInflationTrend;
    }

    public double getActivityTrend() {
        if (this.shouldRecalculateTrends()) {
            this.recalculateTrends();
        }
        return this.cachedActivityTrend;
    }

    public long getTimeSinceLastCycleChange() {
        if (this.cycleHistory.isEmpty()) {
            return 3600L;
        }
        CycleTransition lastTransition = null;
        if (!this.cycleHistory.isEmpty()) {
            CycleTransition[] transitions = this.cycleHistory.toArray(new CycleTransition[0]);
            lastTransition = transitions[transitions.length - 1];
        }
        if (lastTransition != null) {
            return Duration.between(lastTransition.timestamp, LocalDateTime.now()).toSeconds();
        }
        return 3600L;
    }

    public EconomicIntelligenceEngine.EconomicCycle predictNextCycle(EconomicIntelligenceEngine.EconomicCycle currentCycle, EconomicSnapshot currentSnapshot) {
        if (this.cycleHistory.isEmpty()) {
            return currentCycle;
        }
        List<CycleTransition> similarConditions = this.findSimilarConditions(currentSnapshot);
        if (similarConditions.isEmpty()) {
            return currentCycle;
        }
        return this.findMostCommonNextCycle(currentCycle, similarConditions);
    }

    public double getInterventionEffectiveness(EconomicSnapshot.InterventionType type) {
        ArrayList<EconomicIntervention> interventions = new ArrayList<EconomicIntervention>(this.interventionHistory);
        double totalEffectiveness = 0.0;
        int count = 0;
        for (EconomicIntervention intervention : interventions) {
            if (intervention.type != type || !(intervention.effectiveness >= 0.0)) continue;
            totalEffectiveness += intervention.effectiveness;
            ++count;
        }
        return count > 0 ? totalEffectiveness / (double)count : 0.5;
    }

    public List<EconomicAnomaly> detectAnomalies(EconomicSnapshot currentSnapshot) {
        ArrayList<EconomicAnomaly> anomalies = new ArrayList<EconomicAnomaly>();
        if (Math.abs(this.getHealthTrend()) > 0.5) {
            anomalies.add(new EconomicAnomaly(EconomicAnomaly.Type.RAPID_HEALTH_CHANGE, "Economic health changing rapidly: " + String.format("%.2f", this.getHealthTrend()), 0.7));
        }
        if (Math.abs(currentSnapshot.getInflationRate()) > 0.1) {
            anomalies.add(new EconomicAnomaly(EconomicAnomaly.Type.EXTREME_INFLATION, "Extreme inflation rate detected: " + String.format("%.2f%%", currentSnapshot.getInflationRate() * 100.0), 0.9));
        }
        if (currentSnapshot.getMarketVolatility() > 0.8) {
            anomalies.add(new EconomicAnomaly(EconomicAnomaly.Type.HIGH_VOLATILITY, "High market volatility detected: " + String.format("%.2f", currentSnapshot.getMarketVolatility()), 0.6));
        }
        if (currentSnapshot.getGiniCoefficient() > 0.8) {
            anomalies.add(new EconomicAnomaly(EconomicAnomaly.Type.WEALTH_INEQUALITY, "High wealth inequality detected: " + String.format("%.3f", currentSnapshot.getGiniCoefficient()), 0.5));
        }
        return anomalies;
    }

    public EconomicForecast getForecast() {
        if (this.currentForecast == null || this.currentForecast.isStale()) {
            this.updateEconomicForecast();
        }
        return this.currentForecast;
    }

    private boolean shouldRecalculateTrends() {
        return Duration.between(this.lastTrendCalculation, LocalDateTime.now()).toMinutes() > 15L;
    }

    private void invalidateTrendCache() {
        this.lastTrendCalculation = LocalDateTime.now().minusHours(1L);
    }

    private void recalculateTrends() {
        ArrayList<EconomicSnapshot> snapshots = new ArrayList<EconomicSnapshot>(this.historicalSnapshots);
        if (snapshots.size() < 12) {
            this.cachedHealthTrend = 0.0;
            this.cachedInflationTrend = 0.0;
            this.cachedActivityTrend = 0.0;
            this.lastTrendCalculation = LocalDateTime.now();
            return;
        }
        snapshots.sort((a, b) -> a.getTimestamp().compareTo(b.getTimestamp()));
        this.cachedHealthTrend = this.calculateTrend(snapshots, EconomicSnapshot::getEconomicHealth);
        this.cachedInflationTrend = this.calculateTrend(snapshots, EconomicSnapshot::getInflationRate);
        this.cachedActivityTrend = this.calculateTrend(snapshots, EconomicSnapshot::getMarketActivity);
        this.lastTrendCalculation = LocalDateTime.now();
    }

    private double calculateTrend(List<EconomicSnapshot> snapshots, Function<EconomicSnapshot, Double> valueExtractor) {
        int n = Math.min(24, snapshots.size());
        List<EconomicSnapshot> recent = snapshots.subList(snapshots.size() - n, snapshots.size());
        if (recent.size() < 2) {
            return 0.0;
        }
        double earliest = valueExtractor.apply(recent.get(0));
        double latest = valueExtractor.apply(recent.get(recent.size() - 1));
        return Math.max(-1.0, Math.min(1.0, (latest - earliest) / Math.max(0.1, Math.abs(earliest))));
    }

    private long getCycleDuration(EconomicIntelligenceEngine.EconomicCycle cycle) {
        return this.getTimeSinceLastCycleChange();
    }

    private void updatePatternRecognition() {
        this.identifiedPatterns.clear();
        ArrayList<EconomicSnapshot> snapshots = new ArrayList<EconomicSnapshot>(this.historicalSnapshots);
        if (snapshots.size() < 48) {
            return;
        }
        this.identifySeasonalPatterns(snapshots);
        this.identifyVolatilityPatterns(snapshots);
        this.identifyCyclePatterns();
    }

    private void identifySeasonalPatterns(List<EconomicSnapshot> snapshots) {
    }

    private void identifyVolatilityPatterns(List<EconomicSnapshot> snapshots) {
    }

    private void identifyCyclePatterns() {
    }

    private void updateEconomicForecast() {
        ArrayList<EconomicSnapshot> snapshots = new ArrayList<EconomicSnapshot>(this.historicalSnapshots);
        if (snapshots.size() < 12) {
            this.currentForecast = null;
            return;
        }
        this.currentForecast = new EconomicForecast(LocalDateTime.now(), this.predictHealthIn1Hour(), this.predictInflationIn1Hour(), this.predictActivityIn1Hour(), this.calculateForecastConfidence());
    }

    private double predictHealthIn1Hour() {
        return Math.max(0.0, Math.min(1.0, this.getCurrentValue(EconomicSnapshot::getEconomicHealth) + this.getHealthTrend() * 0.1));
    }

    private double predictInflationIn1Hour() {
        return this.getCurrentValue(EconomicSnapshot::getInflationRate) + this.getInflationTrend() * 0.01;
    }

    private double predictActivityIn1Hour() {
        return Math.max(0.0, Math.min(1.0, this.getCurrentValue(EconomicSnapshot::getMarketActivity) + this.getActivityTrend() * 0.05));
    }

    private double calculateForecastConfidence() {
        int dataPoints = this.historicalSnapshots.size();
        double dataQuality = Math.min(1.0, (double)dataPoints / 288.0);
        return dataQuality * 0.8;
    }

    private double getCurrentValue(Function<EconomicSnapshot, Double> valueExtractor) {
        EconomicSnapshot latest = null;
        if (!this.historicalSnapshots.isEmpty()) {
            EconomicSnapshot[] snapshots = this.historicalSnapshots.toArray(new EconomicSnapshot[0]);
            latest = snapshots[snapshots.length - 1];
        }
        return latest != null ? valueExtractor.apply(latest) : 0.0;
    }

    private List<CycleTransition> findSimilarConditions(EconomicSnapshot currentSnapshot) {
        ArrayList<CycleTransition> similar = new ArrayList<CycleTransition>();
        for (CycleTransition transition : this.cycleHistory) {
            if (!this.isSimilarCondition(transition, currentSnapshot)) continue;
            similar.add(transition);
        }
        return similar;
    }

    private boolean isSimilarCondition(CycleTransition transition, EconomicSnapshot current) {
        double healthDiff = Math.abs(transition.economicHealth - current.getEconomicHealth());
        double inflationDiff = Math.abs(transition.inflationRate - current.getInflationRate());
        return healthDiff < 0.2 && inflationDiff < 0.02;
    }

    private EconomicIntelligenceEngine.EconomicCycle findMostCommonNextCycle(EconomicIntelligenceEngine.EconomicCycle currentCycle, List<CycleTransition> transitions) {
        HashMap<EconomicIntelligenceEngine.EconomicCycle, Integer> cycleCounts = new HashMap<EconomicIntelligenceEngine.EconomicCycle, Integer>();
        for (CycleTransition transition : transitions) {
            if (transition.fromCycle != currentCycle) continue;
            cycleCounts.merge(transition.toCycle, 1, Integer::sum);
        }
        return cycleCounts.entrySet().stream().max(Map.Entry.comparingByValue()).map(Map.Entry::getKey).orElse(currentCycle);
    }

    public static class EconomicForecast {
        private final LocalDateTime created;
        private final double predictedHealth;
        private final double predictedInflation;
        private final double predictedActivity;
        private final double confidence;

        public EconomicForecast(LocalDateTime created, double predictedHealth, double predictedInflation, double predictedActivity, double confidence) {
            this.created = created;
            this.predictedHealth = predictedHealth;
            this.predictedInflation = predictedInflation;
            this.predictedActivity = predictedActivity;
            this.confidence = confidence;
        }

        public boolean isStale() {
            return Duration.between(this.created, LocalDateTime.now()).toHours() > 1L;
        }

        public LocalDateTime getCreated() {
            return this.created;
        }

        public double getPredictedHealth() {
            return this.predictedHealth;
        }

        public double getPredictedInflation() {
            return this.predictedInflation;
        }

        public double getPredictedActivity() {
            return this.predictedActivity;
        }

        public double getConfidence() {
            return this.confidence;
        }
    }

    public static class CycleTransition {
        public final LocalDateTime timestamp;
        public final EconomicIntelligenceEngine.EconomicCycle fromCycle;
        public final EconomicIntelligenceEngine.EconomicCycle toCycle;
        public final double economicHealth;
        public final double inflationRate;
        public final long duration;

        public CycleTransition(LocalDateTime timestamp, EconomicIntelligenceEngine.EconomicCycle fromCycle, EconomicIntelligenceEngine.EconomicCycle toCycle, double economicHealth, double inflationRate, long duration) {
            this.timestamp = timestamp;
            this.fromCycle = fromCycle;
            this.toCycle = toCycle;
            this.economicHealth = economicHealth;
            this.inflationRate = inflationRate;
            this.duration = duration;
        }
    }

    public static class EconomicIntervention {
        public final LocalDateTime timestamp;
        public final EconomicSnapshot.InterventionType type;
        public final double magnitude;
        public final double effectiveness;
        public final String details;

        public EconomicIntervention(LocalDateTime timestamp, EconomicSnapshot.InterventionType type, double magnitude, String details) {
            this.timestamp = timestamp;
            this.type = type;
            this.magnitude = magnitude;
            this.effectiveness = -1.0;
            this.details = details;
        }
    }

    public static class EconomicAnomaly {
        public final Type type;
        public final String description;
        public final double severity;
        public final LocalDateTime detected;

        public EconomicAnomaly(Type type, String description, double severity) {
            this.type = type;
            this.description = description;
            this.severity = severity;
            this.detected = LocalDateTime.now();
        }

        public static enum Type {
            RAPID_HEALTH_CHANGE,
            EXTREME_INFLATION,
            HIGH_VOLATILITY,
            WEALTH_INEQUALITY,
            UNUSUAL_ACTIVITY;

        }
    }

    public static class EconomicPattern {
        public final String name;
        public final String description;
        public final double confidence;
        public final LocalDateTime identified;

        public EconomicPattern(String name, String description, double confidence) {
            this.name = name;
            this.description = description;
            this.confidence = confidence;
            this.identified = LocalDateTime.now();
        }
    }
}

