/*
 * Decompiled with CFR 0.152.
 */
package me.koyere.ecoxpert.economy;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.logging.Level;
import javax.inject.Inject;
import javax.inject.Singleton;
import me.koyere.ecoxpert.EcoXpertPlugin;
import me.koyere.ecoxpert.core.config.ConfigManager;
import me.koyere.ecoxpert.core.data.DataManager;
import me.koyere.ecoxpert.core.data.QueryResult;
import me.koyere.ecoxpert.economy.EconomyManager;
import org.bukkit.configuration.file.FileConfiguration;

@Singleton
public class EconomyManagerImpl
implements EconomyManager {
    private final EcoXpertPlugin plugin;
    private final ConfigManager configManager;
    private final DataManager dataManager;
    private BigDecimal startingBalance;
    private BigDecimal maximumBalance;
    private String currencyNameSingular;
    private String currencyNamePlural;
    private String currencySymbol;
    private int decimalPlaces;
    private DecimalFormat moneyFormat;

    @Inject
    public EconomyManagerImpl(EcoXpertPlugin plugin, ConfigManager configManager, DataManager dataManager) {
        this.plugin = plugin;
        this.configManager = configManager;
        this.dataManager = dataManager;
    }

    private void debug(String message) {
        if (this.configManager != null && this.configManager.isDebugEnabled()) {
            this.plugin.getLogger().info("ECOXPERT DEBUG - " + message);
        }
    }

    @Override
    public void initialize() {
        this.plugin.getLogger().info("Initializing economy system...");
        this.loadConfiguration();
        this.setupMoneyFormat();
        this.plugin.getLogger().info("Economy system initialized successfully");
    }

    @Override
    public void shutdown() {
        this.plugin.getLogger().info("Economy system shutdown complete");
    }

    @Override
    public CompletableFuture<Boolean> hasAccount(UUID playerUuid) {
        this.debug("hasAccount called for: " + String.valueOf(playerUuid));
        return ((CompletableFuture)this.dataManager.executeQuery("SELECT player_uuid FROM ecoxpert_accounts WHERE player_uuid = ? LIMIT 1", playerUuid.toString()).thenApply(result -> {
            try (QueryResult queryResult = result;){
                boolean exists = result.next();
                this.debug("hasAccount result: " + exists + " for: " + String.valueOf(playerUuid));
                Boolean bl = exists;
                return bl;
            }
        })).exceptionally(throwable -> {
            this.plugin.getLogger().log(Level.SEVERE, "ECOXPERT ERROR - hasAccount failed for: " + String.valueOf(playerUuid), (Throwable)throwable);
            return false;
        });
    }

    @Override
    public CompletableFuture<Void> createAccount(UUID playerUuid, BigDecimal startingBalance) {
        this.debug("createAccount called for: " + String.valueOf(playerUuid) + " with balance: " + String.valueOf(startingBalance));
        return ((CompletableFuture)this.dataManager.executeUpdate("INSERT OR IGNORE INTO ecoxpert_accounts (player_uuid, balance) VALUES (?, ?)", playerUuid.toString(), startingBalance).thenCompose(rows -> {
            this.debug("createAccount affected " + rows + " rows for: " + String.valueOf(playerUuid));
            if (rows > 0) {
                return this.logTransaction(null, playerUuid, startingBalance, "ACCOUNT_CREATION", "Initial account creation");
            }
            return CompletableFuture.completedFuture(null);
        })).exceptionally(throwable -> {
            this.plugin.getLogger().log(Level.SEVERE, "ECOXPERT ERROR - createAccount failed for: " + String.valueOf(playerUuid), (Throwable)throwable);
            return null;
        });
    }

    @Override
    public CompletableFuture<BigDecimal> getBalance(UUID playerUuid) {
        this.debug("getBalance called for: " + String.valueOf(playerUuid));
        return ((CompletableFuture)this.ensureAccountExists(playerUuid).thenCompose(v -> {
            this.debug("ensureAccountExists completed, executing balance query");
            return this.dataManager.executeQuery("SELECT balance FROM ecoxpert_accounts WHERE player_uuid = ?", playerUuid.toString()).thenApply(result -> {
                try (QueryResult queryResult = result;){
                    if (result.next()) {
                        BigDecimal balance = result.getBigDecimal("balance");
                        this.debug("getBalance found balance: " + String.valueOf(balance) + " for: " + String.valueOf(playerUuid));
                        BigDecimal bigDecimal = balance;
                        return bigDecimal;
                    }
                    this.debug("getBalance found no rows for: " + String.valueOf(playerUuid));
                    BigDecimal bigDecimal = BigDecimal.ZERO;
                    return bigDecimal;
                }
            });
        })).exceptionally(throwable -> {
            this.plugin.getLogger().log(Level.SEVERE, "ECOXPERT ERROR - getBalance failed for: " + String.valueOf(playerUuid), (Throwable)throwable);
            return BigDecimal.ZERO;
        });
    }

    @Override
    public CompletableFuture<Void> setBalance(UUID playerUuid, BigDecimal balance, String reason) {
        this.validateAmount(balance);
        return ((CompletableFuture)this.ensureAccountExists(playerUuid).thenCompose(v -> this.dataManager.executeUpdate("UPDATE ecoxpert_accounts SET balance = ?, updated_at = CURRENT_TIMESTAMP WHERE player_uuid = ?", balance, playerUuid.toString()).thenCompose(rows -> {
            if (rows > 0) {
                return this.logTransaction(null, playerUuid, balance, "BALANCE_SET", reason);
            }
            return CompletableFuture.completedFuture(null);
        }))).thenApply(v -> null);
    }

    @Override
    public CompletableFuture<Void> addMoney(UUID playerUuid, BigDecimal amount, String reason) {
        this.validateAmount(amount);
        this.plugin.getLogger().info("ECOXPERT DEBUG - addMoney called for: " + String.valueOf(playerUuid) + " amount: " + String.valueOf(amount));
        return ((CompletableFuture)((CompletableFuture)this.ensureAccountExists(playerUuid).thenCompose(v -> {
            this.plugin.getLogger().info("ECOXPERT DEBUG - ensureAccountExists completed, executing addMoney update");
            return this.dataManager.executeUpdate("UPDATE ecoxpert_accounts SET balance = balance + ?, updated_at = CURRENT_TIMESTAMP WHERE player_uuid = ?", amount, playerUuid.toString()).thenCompose(rows -> {
                this.plugin.getLogger().info("ECOXPERT DEBUG - addMoney update affected " + rows + " rows");
                if (rows > 0) {
                    return this.logTransaction(null, playerUuid, amount, "DEPOSIT", reason);
                }
                return CompletableFuture.completedFuture(null);
            });
        })).thenApply(v -> {
            this.plugin.getLogger().info("ECOXPERT DEBUG - addMoney completed for: " + String.valueOf(playerUuid));
            return null;
        })).exceptionally(throwable -> {
            this.plugin.getLogger().log(Level.SEVERE, "ECOXPERT ERROR - addMoney failed for: " + String.valueOf(playerUuid), (Throwable)throwable);
            return null;
        });
    }

    @Override
    public CompletableFuture<Boolean> removeMoney(UUID playerUuid, BigDecimal amount, String reason) {
        this.validateAmount(amount);
        return this.ensureAccountExists(playerUuid).thenCompose(v -> this.hasSufficientFunds(playerUuid, amount).thenCompose(hasFunds -> {
            if (!hasFunds.booleanValue()) {
                return CompletableFuture.completedFuture(false);
            }
            return this.dataManager.executeUpdate("UPDATE ecoxpert_accounts SET balance = balance - ?, updated_at = CURRENT_TIMESTAMP WHERE player_uuid = ?", amount, playerUuid.toString()).thenCompose(rows -> {
                if (rows > 0) {
                    return this.logTransaction(playerUuid, null, amount, "WITHDRAWAL", reason).thenApply(result -> true);
                }
                return CompletableFuture.completedFuture(false);
            });
        }));
    }

    @Override
    public CompletableFuture<Boolean> transferMoney(UUID fromUuid, UUID toUuid, BigDecimal amount, String reason) {
        this.validateAmount(amount);
        if (fromUuid.equals(toUuid)) {
            throw new IllegalArgumentException("Cannot transfer money to the same account");
        }
        return ((CompletableFuture)this.ensureAccountExists(fromUuid).thenCompose(v -> this.ensureAccountExists(toUuid))).thenCompose(v -> this.dataManager.beginTransaction().thenCompose(transaction -> ((CompletableFuture)transaction.executeQuery("SELECT balance FROM ecoxpert_accounts WHERE player_uuid = ?", fromUuid.toString()).thenCompose(result -> {
            try {
                if (!result.next()) {
                    transaction.rollback();
                    CompletableFuture<Boolean> completableFuture = CompletableFuture.completedFuture(false);
                    return completableFuture;
                }
                BigDecimal currentBalance = result.getBigDecimal("balance");
                if (currentBalance.compareTo(amount) < 0) {
                    transaction.rollback();
                    CompletableFuture<Boolean> completableFuture = CompletableFuture.completedFuture(false);
                    return completableFuture;
                }
                CompletionStage completionStage = transaction.executeUpdate("UPDATE ecoxpert_accounts SET balance = balance - ?, updated_at = CURRENT_TIMESTAMP WHERE player_uuid = ?", amount, fromUuid.toString()).thenCompose(rows1 -> {
                    if (rows1 == 0) {
                        transaction.rollback();
                        return CompletableFuture.completedFuture(false);
                    }
                    return transaction.executeUpdate("UPDATE ecoxpert_accounts SET balance = balance + ?, updated_at = CURRENT_TIMESTAMP WHERE player_uuid = ?", amount, toUuid.toString()).thenCompose(rows2 -> {
                        if (rows2 == 0) {
                            transaction.rollback();
                            return CompletableFuture.completedFuture(false);
                        }
                        return transaction.executeUpdate("INSERT INTO ecoxpert_transactions (from_uuid, to_uuid, amount, type, description) VALUES (?, ?, ?, ?, ?)", fromUuid.toString(), toUuid.toString(), amount, "TRANSFER", reason).thenCompose(rows3 -> transaction.commit().thenApply(commitResult -> true));
                    });
                });
                return completionStage;
            }
            catch (Exception e) {
                transaction.rollback();
                throw new RuntimeException(e);
            }
            finally {
                result.close();
            }
        })).exceptionally(throwable -> {
            transaction.rollback();
            return false;
        })));
    }

    @Override
    public CompletableFuture<Boolean> hasSufficientFunds(UUID playerUuid, BigDecimal amount) {
        return this.getBalance(playerUuid).thenApply(balance -> balance.compareTo(amount) >= 0);
    }

    @Override
    public BigDecimal getStartingBalance() {
        return this.startingBalance;
    }

    @Override
    public BigDecimal getMaximumBalance() {
        return this.maximumBalance;
    }

    @Override
    public String formatMoney(BigDecimal amount) {
        return this.currencySymbol + this.moneyFormat.format(amount);
    }

    @Override
    public String getCurrencyNameSingular() {
        return this.currencyNameSingular;
    }

    @Override
    public String getCurrencyNamePlural() {
        return this.currencyNamePlural;
    }

    @Override
    public String getCurrencySymbol() {
        return this.currencySymbol;
    }

    @Override
    public CompletableFuture<Integer> applyWealthTax(BigDecimal rate, BigDecimal threshold, String reason) {
        if (rate == null || rate.signum() <= 0 || threshold == null || threshold.signum() < 0) {
            return CompletableFuture.completedFuture(0);
        }
        String sql = "UPDATE ecoxpert_accounts\nSET balance = balance - (balance * ?), updated_at = CURRENT_TIMESTAMP\nWHERE balance > ?\n";
        return this.dataManager.executeUpdate(sql, rate, threshold).thenApply(rows -> {
            this.plugin.getLogger().info("Applied wealth tax at rate " + String.valueOf(rate) + ", threshold " + String.valueOf(threshold) + ". Affected accounts: " + rows);
            return rows;
        });
    }

    private void loadConfiguration() {
        FileConfiguration config = this.configManager.getConfig();
        this.startingBalance = BigDecimal.valueOf(config.getDouble("economy.starting-balance", 1000.0));
        this.maximumBalance = config.getDouble("economy.maximum-balance", 0.0) > 0.0 ? BigDecimal.valueOf(config.getDouble("economy.maximum-balance")) : null;
        this.currencyNameSingular = config.getString("economy.currency.name-singular", "dollar");
        this.currencyNamePlural = config.getString("economy.currency.name-plural", "dollars");
        this.currencySymbol = config.getString("economy.currency.symbol", "$");
        this.decimalPlaces = config.getInt("economy.currency.decimal-places", 2);
    }

    private void setupMoneyFormat() {
        StringBuilder pattern = new StringBuilder("#,##0");
        if (this.decimalPlaces > 0) {
            pattern.append(".");
            pattern.append("0".repeat(this.decimalPlaces));
        }
        this.moneyFormat = new DecimalFormat(pattern.toString());
        this.moneyFormat.setRoundingMode(RoundingMode.HALF_UP);
    }

    private CompletableFuture<Void> ensureAccountExists(UUID playerUuid) {
        this.plugin.getLogger().info("ECOXPERT DEBUG - ensureAccountExists called for: " + String.valueOf(playerUuid));
        return this.hasAccount(playerUuid).thenCompose(exists -> {
            this.plugin.getLogger().info("ECOXPERT DEBUG - hasAccount returned: " + exists + " for: " + String.valueOf(playerUuid));
            if (!exists.booleanValue()) {
                this.plugin.getLogger().info("ECOXPERT DEBUG - Creating account with starting balance: " + String.valueOf(this.startingBalance));
                return this.createAccount(playerUuid, this.startingBalance);
            }
            this.plugin.getLogger().info("ECOXPERT DEBUG - Account exists, no need to create");
            return CompletableFuture.completedFuture(null);
        });
    }

    private CompletableFuture<Void> logTransaction(UUID fromUuid, UUID toUuid, BigDecimal amount, String type, String description) {
        return this.dataManager.executeUpdate("INSERT INTO ecoxpert_transactions (from_uuid, to_uuid, amount, type, description) VALUES (?, ?, ?, ?, ?)", fromUuid != null ? fromUuid.toString() : null, toUuid != null ? toUuid.toString() : null, amount, type, description).thenApply(v -> null);
    }

    private void validateAmount(BigDecimal amount) {
        if (amount == null) {
            throw new IllegalArgumentException("Amount cannot be null");
        }
        if (amount.compareTo(BigDecimal.ZERO) < 0) {
            throw new IllegalArgumentException("Amount cannot be negative");
        }
        if (this.maximumBalance != null && amount.compareTo(this.maximumBalance) > 0) {
            throw new IllegalArgumentException("Amount exceeds maximum balance limit");
        }
        if (amount.scale() > this.decimalPlaces) {
            throw new IllegalArgumentException("Amount has too many decimal places");
        }
    }
}

