/*
 * Decompiled with CFR 0.152.
 */
package me.koyere.ecoxpert.modules.bank;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import me.koyere.ecoxpert.modules.bank.BankAccount;
import me.koyere.ecoxpert.modules.bank.BankAccountTier;

public class InterestCalculator {
    private static final int CALCULATION_SCALE = 10;
    private static final RoundingMode ROUNDING_MODE = RoundingMode.HALF_UP;
    private static final BigDecimal DAYS_PER_YEAR = new BigDecimal("365");
    private static final BigDecimal HOURS_PER_DAY = new BigDecimal("24");
    private static final BigDecimal MINUTES_PER_HOUR = new BigDecimal("60");
    private static final BigDecimal MINIMUM_INTEREST_BALANCE = new BigDecimal("100");
    private static final BigDecimal MAXIMUM_DAILY_INTEREST = new BigDecimal("10000");

    public static BigDecimal calculateDailyInterest(BankAccount account) {
        if (account == null || account.isFrozen()) {
            return BigDecimal.ZERO;
        }
        BigDecimal balance = account.getBalance();
        if (balance.compareTo(MINIMUM_INTEREST_BALANCE) < 0) {
            return BigDecimal.ZERO;
        }
        BankAccountTier tier = account.getTier();
        BigDecimal dailyRate = tier.getDailyInterestRate();
        BigDecimal onePlusRate = BigDecimal.ONE.add(dailyRate);
        BigDecimal interestAmount = balance.multiply(onePlusRate.subtract(BigDecimal.ONE));
        if (interestAmount.compareTo(MAXIMUM_DAILY_INTEREST) > 0) {
            interestAmount = MAXIMUM_DAILY_INTEREST;
        }
        return interestAmount.setScale(2, ROUNDING_MODE);
    }

    public static BigDecimal calculateInterestForPeriod(BigDecimal principal, BankAccountTier tier, int days) {
        BigDecimal maxInterest;
        if (principal == null || principal.compareTo(MINIMUM_INTEREST_BALANCE) < 0 || days <= 0) {
            return BigDecimal.ZERO;
        }
        BigDecimal dailyRate = tier.getDailyInterestRate();
        BigDecimal onePlusRate = BigDecimal.ONE.add(dailyRate);
        BigDecimal compound = onePlusRate.pow(days);
        BigDecimal futureValue = principal.multiply(compound);
        BigDecimal interestEarned = futureValue.subtract(principal);
        if (interestEarned.compareTo(maxInterest = MAXIMUM_DAILY_INTEREST.multiply(new BigDecimal(days))) > 0) {
            interestEarned = maxInterest;
        }
        return interestEarned.setScale(2, ROUNDING_MODE);
    }

    public static BigDecimal calculateProjectedInterest(BankAccount account, int days) {
        if (account == null || account.isFrozen() || days <= 0) {
            return BigDecimal.ZERO;
        }
        return InterestCalculator.calculateInterestForPeriod(account.getBalance(), account.getTier(), days);
    }

    public static BigDecimal calculateAccruedInterest(BankAccount account) {
        LocalDateTime now;
        if (account == null || account.isFrozen()) {
            return BigDecimal.ZERO;
        }
        LocalDateTime lastCalculation = account.getLastInterestCalculation();
        long daysSinceLastCalculation = ChronoUnit.DAYS.between(lastCalculation, now = LocalDateTime.now());
        if (daysSinceLastCalculation <= 0L) {
            return BigDecimal.ZERO;
        }
        if (daysSinceLastCalculation > 30L) {
            daysSinceLastCalculation = 30L;
        }
        return InterestCalculator.calculateInterestForPeriod(account.getBalance(), account.getTier(), (int)daysSinceLastCalculation);
    }

    public static BigDecimal calculateAPY(BankAccountTier tier) {
        BigDecimal dailyRate = tier.getDailyInterestRate();
        BigDecimal onePlusRate = BigDecimal.ONE.add(dailyRate);
        BigDecimal compoundAnnual = onePlusRate.pow(365);
        BigDecimal apy = compoundAnnual.subtract(BigDecimal.ONE);
        return apy.multiply(new BigDecimal("100")).setScale(3, ROUNDING_MODE);
    }

    public static int calculateDaysToReachBalance(BigDecimal currentBalance, BigDecimal targetBalance, BankAccountTier tier) {
        if (currentBalance.compareTo(targetBalance) >= 0) {
            return 0;
        }
        if (currentBalance.compareTo(MINIMUM_INTEREST_BALANCE) < 0) {
            return -1;
        }
        BigDecimal dailyRate = tier.getDailyInterestRate();
        double current = currentBalance.doubleValue();
        double target = targetBalance.doubleValue();
        double rate = dailyRate.doubleValue();
        if (rate <= 0.0) {
            return -1;
        }
        double days = Math.log(target / current) / Math.log(1.0 + rate);
        return Math.min((int)Math.ceil(days), 36500);
    }

    public static boolean validateInterestCalculation(BigDecimal principal, BankAccountTier tier, int days) {
        if (principal == null || principal.compareTo(BigDecimal.ZERO) < 0) {
            return false;
        }
        if (tier == null) {
            return false;
        }
        if (days < 0 || days > 36500) {
            return false;
        }
        BigDecimal dailyRate = tier.getDailyInterestRate();
        return dailyRate.compareTo(BigDecimal.ZERO) >= 0 && dailyRate.compareTo(new BigDecimal("0.1")) <= 0;
    }

    public static BigDecimal getMinimumInterestBalance() {
        return MINIMUM_INTEREST_BALANCE;
    }

    public static BigDecimal getMaximumDailyInterest() {
        return MAXIMUM_DAILY_INTEREST;
    }

    public static String formatInterestRate(BigDecimal rate) {
        BigDecimal percentage = rate.multiply(new BigDecimal("100"));
        return String.valueOf(percentage.setScale(3, ROUNDING_MODE)) + "%";
    }

    public static String formatAPY(BankAccountTier tier) {
        BigDecimal apy = InterestCalculator.calculateAPY(tier);
        return String.valueOf(apy.setScale(2, ROUNDING_MODE)) + "%";
    }
}

