/*
 * Decompiled with CFR 0.152.
 */
package me.koyere.ecoxpert.modules.events;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadLocalRandom;
import me.koyere.ecoxpert.EcoXpertPlugin;
import me.koyere.ecoxpert.core.config.ConfigManager;
import me.koyere.ecoxpert.core.data.DataManager;
import me.koyere.ecoxpert.core.education.EducationNotifier;
import me.koyere.ecoxpert.core.translation.TranslationManager;
import me.koyere.ecoxpert.economy.EconomyManager;
import me.koyere.ecoxpert.modules.events.EconomicEvent;
import me.koyere.ecoxpert.modules.events.EconomicEventType;
import me.koyere.ecoxpert.modules.inflation.EconomicIntelligenceEngine;
import me.koyere.ecoxpert.modules.inflation.InflationManager;
import me.koyere.ecoxpert.modules.market.MarketManager;
import org.bukkit.Bukkit;
import org.bukkit.Material;
import org.bukkit.configuration.file.FileConfiguration;
import org.bukkit.entity.Player;
import org.bukkit.plugin.Plugin;

public class EconomicEventEngine {
    private final EcoXpertPlugin plugin;
    private final EconomyManager economyManager;
    private final MarketManager marketManager;
    private final InflationManager inflationManager;
    private final Map<String, EconomicEvent> activeEvents = new ConcurrentHashMap<String, EconomicEvent>();
    private final List<EconomicEvent> eventHistory = new ArrayList<EconomicEvent>();
    private boolean eventEngineActive = false;
    private final Map<EconomicEventType, LocalDateTime> lastEventEndTime = new ConcurrentHashMap<EconomicEventType, LocalDateTime>();
    private LocalDateTime lastEventTime = LocalDateTime.now().minusHours(2L);
    private int consecutiveQuietHours = 0;
    private final Map<EconomicEventType, EventTemplate> eventTemplates = new HashMap<EconomicEventType, EventTemplate>();

    public EconomicEventEngine(EcoXpertPlugin plugin, EconomyManager economyManager, MarketManager marketManager, InflationManager inflationManager) {
        this.plugin = plugin;
        this.economyManager = economyManager;
        this.marketManager = marketManager;
        this.inflationManager = inflationManager;
        this.initializeEventTemplates();
    }

    public CompletableFuture<Void> initialize() {
        return CompletableFuture.runAsync(() -> {
            this.plugin.getLogger().info("\ud83c\udfaa Initializing Dynamic Economic Events Engine...");
            this.startEventIntelligenceEngine();
            this.startEventScheduler();
            this.startAntiStagnationSystem();
            this.eventEngineActive = true;
            this.plugin.getLogger().info("\u2705 Dynamic Economic Events Engine operational");
            this.plugin.getLogger().info("\ud83e\udde0 Event Intelligence: ACTIVE | Anti-Stagnation: ENABLED");
        });
    }

    public void shutdown() {
        this.plugin.getLogger().info("\ud83d\udd0c Shutting down Dynamic Economic Events Engine...");
        this.activeEvents.values().forEach(this::endEvent);
        this.activeEvents.clear();
        this.eventEngineActive = false;
        this.plugin.getLogger().info("Dynamic Economic Events Engine shutdown complete");
    }

    private void startEventIntelligenceEngine() {
        Bukkit.getScheduler().runTaskTimerAsynchronously((Plugin)this.plugin, () -> {
            try {
                this.analyzeEconomicConditionsForEvents();
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Event intelligence analysis failed: " + e.getMessage());
            }
        }, 18000L, 18000L);
    }

    private void analyzeEconomicConditionsForEvents() {
        if (!this.eventEngineActive || !this.inflationManager.isActive()) {
            return;
        }
        this.plugin.getLogger().info("\ud83e\udde0 Analyzing economic conditions for dynamic events...");
        EconomicIntelligenceEngine.EconomicCycle currentCycle = this.inflationManager.getCurrentCycle();
        double economicHealth = this.inflationManager.getEconomicHealth();
        double inflationRate = this.inflationManager.getInflationRate();
        double eventProbability = this.calculateEventProbability(currentCycle, economicHealth, inflationRate);
        this.plugin.getLogger().info("\ud83d\udcca Economic Analysis | Cycle: " + String.valueOf((Object)currentCycle) + " | Health: " + String.format("%.1f%%", economicHealth * 100.0) + " | Event Probability: " + String.format("%.1f%%", eventProbability * 100.0));
        if (this.shouldTriggerEvent(eventProbability)) {
            EconomicEventType optimalEventType = this.selectOptimalEventType(currentCycle, economicHealth, inflationRate);
            this.triggerIntelligentEvent(optimalEventType);
        }
        this.updateStagnationTracking();
    }

    private double calculateEventProbability(EconomicIntelligenceEngine.EconomicCycle cycle, double health, double inflation) {
        long hoursSinceLastEvent;
        double baseProbability = 0.3;
        baseProbability += (switch (cycle) {
            default -> throw new IncompatibleClassChangeError();
            case EconomicIntelligenceEngine.EconomicCycle.DEPRESSION -> 0.4;
            case EconomicIntelligenceEngine.EconomicCycle.RECESSION -> 0.2;
            case EconomicIntelligenceEngine.EconomicCycle.STABLE -> -0.1;
            case EconomicIntelligenceEngine.EconomicCycle.GROWTH -> 0.1;
            case EconomicIntelligenceEngine.EconomicCycle.BOOM -> 0.3;
            case EconomicIntelligenceEngine.EconomicCycle.BUBBLE -> 0.5;
        });
        if (health < 0.3) {
            baseProbability += 0.3;
        }
        if (health > 0.9) {
            baseProbability += 0.2;
        }
        if (Math.abs(inflation) > 0.05) {
            baseProbability += 0.2;
        }
        if ((hoursSinceLastEvent = Duration.between(this.lastEventTime, LocalDateTime.now()).toHours()) < 2L) {
            baseProbability *= 0.1;
        }
        return Math.max(0.0, Math.min(1.0, baseProbability));
    }

    private EconomicEventType selectOptimalEventType(EconomicIntelligenceEngine.EconomicCycle cycle, double health, double inflation) {
        EnumMap<EconomicEventType, Double> weights = new EnumMap<EconomicEventType, Double>(EconomicEventType.class);
        for (EconomicEventType t : EconomicEventType.values()) {
            double w = this.getEventWeight(t, cycle, health, inflation);
            if (!(w > 0.0)) continue;
            weights.put(t, w);
        }
        if (weights.isEmpty()) {
            return EconomicEventType.MARKET_DISCOVERY;
        }
        return this.pickWeighted(weights);
    }

    private double getEventWeight(EconomicEventType type, EconomicIntelligenceEngine.EconomicCycle cycle, double health, double inflation) {
        double base = 1.0;
        try {
            ConfigManager cfg = this.plugin.getServiceRegistry().getInstance(ConfigManager.class);
            FileConfiguration ev = cfg.getModuleConfig("events");
            String key = (type.name().toLowerCase() + ".weight").replace(' ', '_');
            base = Math.max(0.0, ev.getDouble(key, 1.0));
        }
        catch (Exception cfg) {
            // empty catch block
        }
        double modifier = 1.0;
        switch (cycle) {
            case DEPRESSION: {
                if (type == EconomicEventType.GOVERNMENT_STIMULUS) {
                    modifier *= 2.0;
                }
                if (type == EconomicEventType.TRADE_BOOM) {
                    modifier *= 1.5;
                }
                if (type != EconomicEventType.MARKET_CORRECTION) break;
                modifier *= 0.7;
                break;
            }
            case RECESSION: {
                if (type == EconomicEventType.TRADE_BOOM) {
                    modifier *= 1.6;
                }
                if (type != EconomicEventType.GOVERNMENT_STIMULUS) break;
                modifier *= 1.4;
                break;
            }
            case STABLE: {
                if (type != EconomicEventType.MARKET_DISCOVERY && type != EconomicEventType.SEASONAL_DEMAND) break;
                modifier *= 1.3;
                break;
            }
            case GROWTH: {
                if (type != EconomicEventType.INVESTMENT_OPPORTUNITY && type != EconomicEventType.TECHNOLOGICAL_BREAKTHROUGH) break;
                modifier *= 1.5;
                break;
            }
            case BOOM: {
                if (type == EconomicEventType.LUXURY_DEMAND) {
                    modifier *= 1.7;
                }
                if (type != EconomicEventType.MARKET_CORRECTION) break;
                modifier *= 1.2;
                break;
            }
            case BUBBLE: {
                if (type != EconomicEventType.MARKET_CORRECTION && type != EconomicEventType.BLACK_SWAN_EVENT) break;
                modifier *= 1.6;
            }
        }
        if (health < 0.4 && (type == EconomicEventType.GOVERNMENT_STIMULUS || type == EconomicEventType.TRADE_BOOM)) {
            modifier *= 1.5;
        }
        if (health > 0.85 && (type == EconomicEventType.LUXURY_DEMAND || type == EconomicEventType.INVESTMENT_OPPORTUNITY)) {
            modifier *= 1.3;
        }
        if (Math.abs(inflation) > 0.08 && type == EconomicEventType.MARKET_CORRECTION) {
            modifier *= 1.4;
        }
        return base * modifier;
    }

    private EconomicEventType pickWeighted(Map<EconomicEventType, Double> weights) {
        double total = 0.0;
        for (double w : weights.values()) {
            total += w;
        }
        if (total <= 0.0) {
            return EconomicEventType.MARKET_DISCOVERY;
        }
        double r = ThreadLocalRandom.current().nextDouble() * total;
        double acc = 0.0;
        for (Map.Entry<EconomicEventType, Double> e : weights.entrySet()) {
            if (!(r <= (acc += e.getValue().doubleValue()))) continue;
            return e.getKey();
        }
        return EconomicEventType.MARKET_DISCOVERY;
    }

    private void triggerIntelligentEvent(EconomicEventType eventType) {
        if (this.isTypeActive(eventType)) {
            this.plugin.getLogger().info("Event " + String.valueOf((Object)eventType) + " already active, skipping");
            return;
        }
        EventTemplate template = this.eventTemplates.get((Object)eventType);
        if (template == null) {
            this.plugin.getLogger().warning("No template found for event type: " + String.valueOf((Object)eventType));
            return;
        }
        EconomicEvent event = this.createIntelligentEvent(template, eventType);
        this.startEvent(event);
        this.lastEventTime = LocalDateTime.now();
        this.consecutiveQuietHours = 0;
    }

    public boolean triggerEvent(EconomicEventType eventType) {
        try {
            if (this.isTypeActive(eventType)) {
                this.plugin.getLogger().info("Event " + String.valueOf((Object)eventType) + " already active, skipping");
                return false;
            }
            if (!this.activeEvents.isEmpty()) {
                this.plugin.getLogger().info("Another event is active; skipping new trigger");
                return false;
            }
            int cooldownHours = this.getEventCooldownHours(eventType);
            LocalDateTime lastEnd = this.lastEventEndTime.get((Object)eventType);
            if (lastEnd != null && Duration.between(lastEnd, LocalDateTime.now()).toHours() < (long)cooldownHours) {
                this.plugin.getLogger().info("Cooldown active for event " + String.valueOf((Object)eventType) + "; skipping trigger");
                return false;
            }
            EventTemplate template = this.eventTemplates.get((Object)eventType);
            if (template == null) {
                this.plugin.getLogger().warning("No template found for event type: " + String.valueOf((Object)eventType));
                return false;
            }
            EconomicEvent event = this.createIntelligentEvent(template, eventType);
            this.startEvent(event);
            this.lastEventTime = LocalDateTime.now();
            this.consecutiveQuietHours = 0;
            return true;
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Failed to trigger event " + String.valueOf((Object)eventType) + ": " + e.getMessage());
            return false;
        }
    }

    private boolean isTypeActive(EconomicEventType type) {
        for (EconomicEvent ev : this.activeEvents.values()) {
            if (ev.getType() != type || ev.getStatus() != EconomicEvent.EventStatus.ACTIVE) continue;
            return true;
        }
        return false;
    }

    public boolean isEngineActive() {
        return this.eventEngineActive;
    }

    public int getActiveEventsCount() {
        int c = 0;
        for (EconomicEvent ev : this.activeEvents.values()) {
            if (ev.getStatus() != EconomicEvent.EventStatus.ACTIVE) continue;
            ++c;
        }
        return c;
    }

    public int getEventCooldownHours(EconomicEventType type) {
        try {
            ConfigManager cfg = this.plugin.getServiceRegistry().getInstance(ConfigManager.class);
            FileConfiguration ev = cfg.getModuleConfig("events");
            int global = ev.getInt("cooldown_hours", 6);
            String key = (type.name().toLowerCase() + ".cooldown_hours").replace(' ', '_');
            return ev.getInt(key, global);
        }
        catch (Exception e) {
            return 6;
        }
    }

    public double getConfiguredWeight(EconomicEventType type) {
        try {
            ConfigManager cfg = this.plugin.getServiceRegistry().getInstance(ConfigManager.class);
            FileConfiguration ev = cfg.getModuleConfig("events");
            String key = (type.name().toLowerCase() + ".weight").replace(' ', '_');
            return Math.max(0.0, ev.getDouble(key, 1.0));
        }
        catch (Exception e) {
            return 1.0;
        }
    }

    public long getRemainingCooldownHours(EconomicEventType type) {
        try {
            LocalDateTime lastEnd = this.lastEventEndTime.get((Object)type);
            if (lastEnd == null) {
                return 0L;
            }
            int cooldown = this.getEventCooldownHours(type);
            long elapsed = Duration.between(lastEnd, LocalDateTime.now()).toHours();
            long remaining = (long)cooldown - elapsed;
            return Math.max(0L, remaining);
        }
        catch (Exception e) {
            return 0L;
        }
    }

    private void persistEvent(EconomicEvent event, String status) {
        try {
            DataManager dm = this.plugin.getServiceRegistry().getInstance(DataManager.class);
            String paramsJson = this.toJson(event.getParameters());
            dm.executeUpdate("INSERT INTO ecoxpert_economic_events (event_id, type, status, parameters, start_time) VALUES (?, ?, ?, ?, CURRENT_TIMESTAMP)", event.getId(), event.getType().name(), status, paramsJson).join();
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private void persistEventEnd(EconomicEvent event) {
        try {
            DataManager dm = this.plugin.getServiceRegistry().getInstance(DataManager.class);
            long minutes = Duration.between(event.getStartTime(), LocalDateTime.now()).toMinutes();
            this.addMetric(event, "metrics.duration_minutes", minutes);
            String paramsJson = this.toJson(event.getParameters());
            dm.executeUpdate("UPDATE ecoxpert_economic_events SET status = ?, end_time = CURRENT_TIMESTAMP, parameters = ? WHERE event_id = ?", "COMPLETED", paramsJson, event.getId()).join();
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private String toJson(Map<String, Object> map) {
        if (map == null || map.isEmpty()) {
            return "{}";
        }
        StringBuilder sb = new StringBuilder();
        sb.append('{');
        boolean first = true;
        for (Map.Entry<String, Object> e : map.entrySet()) {
            if (!first) {
                sb.append(',');
            }
            first = false;
            sb.append('\"').append(this.escape(e.getKey())).append('\"').append(':');
            sb.append(this.toJsonValue(e.getValue()));
        }
        sb.append('}');
        return sb.toString();
    }

    private String toJsonValue(Object v) {
        if (v == null) {
            return "null";
        }
        if (v instanceof Number) {
            return v.toString();
        }
        if (v instanceof Boolean) {
            return v.toString();
        }
        if (v instanceof CharSequence) {
            return "\"" + this.escape(v.toString()) + "\"";
        }
        if (v instanceof Material) {
            return "\"" + ((Material)v).name() + "\"";
        }
        if (v instanceof List) {
            StringBuilder sb = new StringBuilder();
            sb.append('[');
            boolean first = true;
            for (Object o : (List)v) {
                if (!first) {
                    sb.append(',');
                }
                first = false;
                sb.append(this.toJsonValue(o));
            }
            sb.append(']');
            return sb.toString();
        }
        return "\"" + this.escape(v.toString()) + "\"";
    }

    private String escape(String s) {
        return s.replace("\\", "\\\\").replace("\"", "\\\"");
    }

    private EconomicEvent createIntelligentEvent(EventTemplate template, EconomicEventType eventType) {
        String eventId = this.generateEventId(eventType);
        int duration = this.calculateIntelligentDuration(template.baseDuration);
        double intensity = this.calculateIntelligentIntensity(template.baseIntensity);
        List<Material> affectedItems = this.selectIntelligentItems(template.potentialItems);
        Map<String, Object> parameters = this.createSmartEventParameters(eventType, intensity, affectedItems);
        return new EconomicEvent(eventId, eventType, template.name, template.description, LocalDateTime.now(), duration, intensity, affectedItems, parameters, EconomicEvent.EventStatus.ACTIVE);
    }

    private void startEvent(EconomicEvent event) {
        this.plugin.getLogger().info("\ud83c\udfaa Starting Dynamic Economic Event: " + event.getName() + " | Duration: " + event.getDuration() + "min | Intensity: " + String.format("%.1f%%", event.getIntensity() * 100.0));
        this.activeEvents.put(event.getId(), event);
        this.eventHistory.add(event);
        this.broadcastEventStart(event);
        this.applyEventEffects(event);
        try {
            TranslationManager tm = this.plugin.getServiceRegistry().getInstance(TranslationManager.class);
            EducationNotifier.broadcastEvent(this.plugin, tm, event);
        }
        catch (Exception exception) {
            // empty catch block
        }
        this.persistEvent(event, "ACTIVE");
        this.scheduleEventEnd(event);
        if (this.inflationManager != null) {
            this.plugin.getLogger().info("\ud83d\udd17 Event integrated with Economic Intelligence System");
        }
    }

    private void applyEventEffects(EconomicEvent event) {
        switch (event.getType()) {
            case GOVERNMENT_STIMULUS: {
                this.applyGovernmentStimulus(event);
                break;
            }
            case TRADE_BOOM: {
                this.applyTradeBoom(event);
                break;
            }
            case MARKET_DISCOVERY: {
                this.applyMarketDiscovery(event);
                break;
            }
            case INVESTMENT_OPPORTUNITY: {
                this.applyInvestmentOpportunity(event);
                break;
            }
            case LUXURY_DEMAND: {
                this.applyLuxuryDemand(event);
                break;
            }
            case MARKET_CORRECTION: {
                this.applyMarketCorrection(event);
                break;
            }
            case RESOURCE_SHORTAGE: {
                this.applyResourceShortage(event);
                break;
            }
            case TECHNOLOGICAL_BREAKTHROUGH: {
                this.applyTechnologicalBreakthrough(event);
                break;
            }
            case SEASONAL_DEMAND: {
                this.applySeasonalDemand(event);
                break;
            }
            case BLACK_SWAN_EVENT: {
                this.applyBlackSwanEvent(event);
            }
        }
    }

    private void applyGovernmentStimulus(EconomicEvent event) {
        this.plugin.getLogger().info("\ud83d\udc89 GOVERNMENT STIMULUS: Distributing emergency economic aid");
        double totalStimulus = (Double)event.getParameters().get("total_stimulus");
        double perPlayerAmount = totalStimulus / (double)Math.max(1, Bukkit.getOnlinePlayers().size());
        TranslationManager tm = this.plugin.getServiceRegistry().getInstance(TranslationManager.class);
        Bukkit.getOnlinePlayers().forEach(player -> {
            this.economyManager.addMoney(player.getUniqueId(), BigDecimal.valueOf(perPlayerAmount), "Government Economic Stimulus - " + event.getName());
            String amountStr = this.economyManager.formatMoney(BigDecimal.valueOf(perPlayerAmount));
            player.sendMessage(tm.getMessage("prefix", new Object[0]) + tm.getPlayerMessage((Player)player, "events.stimulus.received", amountStr));
        });
        String totalStr = this.economyManager.formatMoney(BigDecimal.valueOf(totalStimulus));
        Bukkit.broadcastMessage((String)(tm.getMessage("prefix", new Object[0]) + tm.getMessage("events.stimulus.injected", totalStr, new Object[0])));
        this.addMetric(event, "metrics.type", "stimulus");
        this.addMetric(event, "metrics.total_stimulus", totalStimulus);
        this.addMetric(event, "metrics.per_player_amount", perPlayerAmount);
        this.addMetric(event, "metrics.recipients", Bukkit.getOnlinePlayers().size());
    }

    private void applyTradeBoom(EconomicEvent event) {
        this.plugin.getLogger().info("\ud83d\udcc8 TRADE BOOM: Boosting market activity and rewards");
        double multiplier = (Double)event.getParameters().get("trade_multiplier");
        TranslationManager tm = this.plugin.getServiceRegistry().getInstance(TranslationManager.class);
        String pct = String.format("%.0f", (multiplier - 1.0) * 100.0);
        Bukkit.broadcastMessage((String)(tm.getMessage("prefix", new Object[0]) + tm.getMessage("events.trade_boom.announce", pct, new Object[0])));
        this.addMetric(event, "metrics.type", "global_bonus");
        this.addMetric(event, "metrics.trade_bonus_pct", (multiplier - 1.0) * 100.0);
        if (this.marketManager != null) {
            this.plugin.getLogger().info("\ud83d\udd17 Trade boom integrated with market system");
            this.marketManager.setGlobalPriceFactors(0.99, 1.01);
            Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, () -> this.marketManager.setGlobalPriceFactors(1.0, 1.0), 12000L);
        }
    }

    private void applyMarketDiscovery(EconomicEvent event) {
        this.plugin.getLogger().info("\ud83d\udd0d MARKET DISCOVERY: New valuable resources discovered");
        List newItems = (List)event.getParameters().get("discovered_items");
        double priceBoost = (Double)event.getParameters().get("discovery_bonus");
        StringBuilder itemList = new StringBuilder();
        for (int i = 0; i < newItems.size(); ++i) {
            Material item = (Material)newItems.get(i);
            if (i > 0) {
                itemList.append(", ");
            }
            itemList.append(item.name().toLowerCase().replace('_', ' '));
        }
        TranslationManager tm = this.plugin.getServiceRegistry().getInstance(TranslationManager.class);
        String pct = String.format("%.0f", (priceBoost - 1.0) * 100.0);
        Bukkit.broadcastMessage((String)(tm.getMessage("prefix", new Object[0]) + tm.getMessage("events.market_discovery.announce", itemList.toString(), pct)));
        this.addMetric(event, "metrics.type", "per_item");
        this.addMetric(event, "metrics.items", newItems.size());
        this.addMetric(event, "metrics.sell_boost_pct", (priceBoost - 1.0) * 100.0);
        if (this.marketManager != null && !newItems.isEmpty()) {
            HashMap<Material, double[]> factors = new HashMap<Material, double[]>();
            double sellBoost = Math.min(1.2, 1.0 + (priceBoost - 1.0) * 0.5);
            for (Material m : newItems) {
                factors.put(m, new double[]{1.0, sellBoost});
            }
            this.marketManager.applyTemporaryItemFactors(factors, 10);
        }
    }

    private void initializeEventTemplates() {
        this.eventTemplates.put(EconomicEventType.GOVERNMENT_STIMULUS, new EventTemplate("Government Economic Stimulus", "Emergency government aid to boost economic activity during downturns", 60, 0.7, Arrays.asList(new Material[0])));
        this.eventTemplates.put(EconomicEventType.TRADE_BOOM, new EventTemplate("Trade Boom Period", "Increased demand for trading creates bonus opportunities", 45, 0.5, Arrays.asList(Material.DIAMOND, Material.EMERALD, Material.NETHERITE_INGOT)));
        this.eventTemplates.put(EconomicEventType.MARKET_DISCOVERY, new EventTemplate("Valuable Resource Discovery", "New deposits discovered, increasing market value of specific resources", 90, 0.6, Arrays.asList(Material.GOLD_INGOT, Material.IRON_INGOT, Material.COPPER_INGOT, Material.REDSTONE, Material.LAPIS_LAZULI)));
        this.eventTemplates.put(EconomicEventType.TECHNOLOGICAL_BREAKTHROUGH, new EventTemplate("Technological Breakthrough", "Innovation increases efficiency and creates new market opportunities", 120, 0.8, Arrays.asList(Material.REDSTONE, Material.REPEATER, Material.COMPARATOR, Material.PISTON, Material.OBSERVER)));
        this.eventTemplates.put(EconomicEventType.LUXURY_DEMAND, new EventTemplate("Luxury Goods Demand", "High demand for luxury items during economic prosperity", 30, 0.4, Arrays.asList(Material.DIAMOND, Material.EMERALD, Material.GOLD_BLOCK, Material.NETHERITE_BLOCK)));
        this.eventTemplates.put(EconomicEventType.INVESTMENT_OPPORTUNITY, new EventTemplate("Investment Opportunity", "Special investment opportunities with increased returns", 75, 0.6, Arrays.asList(Material.EMERALD, Material.DIAMOND, Material.GOLD_INGOT)));
        this.eventTemplates.put(EconomicEventType.MARKET_CORRECTION, new EventTemplate("Market Correction", "Natural market adjustment to prevent economic bubbles", 40, 0.7, Arrays.asList(new Material[0])));
        this.eventTemplates.put(EconomicEventType.RESOURCE_SHORTAGE, new EventTemplate("Resource Shortage", "Temporary shortage increases value of specific resources", 60, 0.8, Arrays.asList(Material.IRON_INGOT, Material.COAL, Material.WHEAT, Material.OAK_LOG, Material.COBBLESTONE)));
        this.eventTemplates.put(EconomicEventType.SEASONAL_DEMAND, new EventTemplate("Seasonal Market Demand", "Seasonal changes affect market preferences", 180, 0.3, Arrays.asList(Material.PUMPKIN, Material.WHEAT, Material.CARROT, Material.POTATO, Material.BEETROOT)));
        this.eventTemplates.put(EconomicEventType.BLACK_SWAN_EVENT, new EventTemplate("Unexpected Economic Event", "Rare, unpredictable event with significant economic impact", 20, 1.0, Arrays.asList(new Material[0])));
    }

    private void startAntiStagnationSystem() {
        Bukkit.getScheduler().runTaskTimerAsynchronously((Plugin)this.plugin, () -> {
            try {
                this.checkForEconomicStagnation();
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Anti-stagnation system failed: " + e.getMessage());
            }
        }, 72000L, 72000L);
    }

    private void checkForEconomicStagnation() {
        ++this.consecutiveQuietHours;
        this.plugin.getLogger().info("\ud83d\udd0d Anti-Stagnation Check | Quiet Hours: " + this.consecutiveQuietHours);
        if (this.consecutiveQuietHours >= 4) {
            this.plugin.getLogger().warning("\u26a0\ufe0f Economic stagnation detected! Triggering emergency stimulation");
            EconomicEventType antiStagnationEvent = this.selectAntiStagnationEvent();
            this.triggerIntelligentEvent(antiStagnationEvent);
            this.consecutiveQuietHours = 0;
        }
        if (Bukkit.getOnlinePlayers().size() > 0 && this.consecutiveQuietHours >= 2 && ThreadLocalRandom.current().nextDouble() < 0.4) {
            this.triggerIntelligentEvent(EconomicEventType.MARKET_DISCOVERY);
        }
    }

    private EconomicEventType selectAntiStagnationEvent() {
        if (this.inflationManager != null && this.inflationManager.isActive()) {
            double economicHealth = this.inflationManager.getEconomicHealth();
            if (economicHealth < 0.5) {
                return EconomicEventType.GOVERNMENT_STIMULUS;
            }
            return EconomicEventType.TRADE_BOOM;
        }
        return EconomicEventType.MARKET_DISCOVERY;
    }

    private void startEventScheduler() {
        Bukkit.getScheduler().runTaskTimerAsynchronously((Plugin)this.plugin, () -> {
            try {
                this.manageActiveEvents();
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Event scheduler failed: " + e.getMessage());
            }
        }, 6000L, 6000L);
    }

    private void manageActiveEvents() {
        ArrayList<EconomicEvent> expiredEvents = new ArrayList<EconomicEvent>();
        for (EconomicEvent event : this.activeEvents.values()) {
            if (!event.hasExpired()) continue;
            expiredEvents.add(event);
        }
        expiredEvents.forEach(this::endEvent);
    }

    private void endEvent(EconomicEvent event) {
        this.plugin.getLogger().info("\ud83c\udfc1 Ending Economic Event: " + event.getName());
        this.activeEvents.remove(event.getId());
        event.setStatus(EconomicEvent.EventStatus.COMPLETED);
        this.lastEventEndTime.put(event.getType(), LocalDateTime.now());
        this.broadcastEventEnd(event);
        this.applyEventEndEffects(event);
        this.persistEventEnd(event);
    }

    public boolean endEventById(String eventId) {
        EconomicEvent ev = this.activeEvents.get(eventId);
        if (ev == null) {
            return false;
        }
        this.endEvent(ev);
        return true;
    }

    private boolean shouldTriggerEvent(double probability) {
        return ThreadLocalRandom.current().nextDouble() < probability;
    }

    private String generateEventId(EconomicEventType eventType) {
        return eventType.name() + "_" + System.currentTimeMillis();
    }

    private int calculateIntelligentDuration(int baseDuration) {
        double multiplier = 1.0;
        if (this.inflationManager != null && this.inflationManager.isActive()) {
            double economicHealth = this.inflationManager.getEconomicHealth();
            if (economicHealth < 0.4) {
                multiplier = 0.7;
            } else if (economicHealth > 0.8) {
                multiplier = 1.3;
            }
        }
        return (int)((double)baseDuration * multiplier);
    }

    private double calculateIntelligentIntensity(double baseIntensity) {
        double intensity = baseIntensity;
        int playerCount = Bukkit.getOnlinePlayers().size();
        if (playerCount > 20) {
            intensity *= 1.2;
        } else if (playerCount < 5) {
            intensity *= 0.8;
        }
        return Math.max(0.1, Math.min(1.0, intensity));
    }

    private List<Material> selectIntelligentItems(List<Material> potentialItems) {
        if (potentialItems.isEmpty()) {
            return new ArrayList<Material>();
        }
        int itemCount = ThreadLocalRandom.current().nextInt(1, Math.min(4, potentialItems.size() + 1));
        ArrayList<Material> selected = new ArrayList<Material>(potentialItems);
        Collections.shuffle(selected);
        return selected.subList(0, itemCount);
    }

    private Map<String, Object> createSmartEventParameters(EconomicEventType eventType, double intensity, List<Material> items) {
        HashMap<String, Object> params = new HashMap<String, Object>();
        switch (eventType) {
            case GOVERNMENT_STIMULUS: {
                double totalStimulus = intensity * (double)Bukkit.getOnlinePlayers().size() * 1000.0;
                params.put("total_stimulus", totalStimulus);
                break;
            }
            case TRADE_BOOM: {
                double multiplier = 1.0 + intensity * 0.5;
                params.put("trade_multiplier", multiplier);
                break;
            }
            case MARKET_DISCOVERY: {
                params.put("discovered_items", items);
                params.put("discovery_bonus", 1.0 + intensity);
            }
        }
        return params;
    }

    private void updateStagnationTracking() {
        this.consecutiveQuietHours = this.activeEvents.isEmpty() ? ++this.consecutiveQuietHours : 0;
    }

    private void scheduleEventEnd(EconomicEvent event) {
        Bukkit.getScheduler().runTaskLaterAsynchronously((Plugin)this.plugin, () -> {
            if (this.activeEvents.containsKey(event.getId())) {
                this.endEvent(event);
            }
        }, (long)(event.getDuration() * 60) * 20L);
    }

    private void broadcastEventStart(EconomicEvent event) {
        TranslationManager tm = this.plugin.getServiceRegistry().getInstance(TranslationManager.class);
        Bukkit.broadcastMessage((String)tm.getMessage("events.banner", new Object[0]));
        Bukkit.broadcastMessage((String)tm.getMessage("events.started", event.getName(), new Object[0]));
        Bukkit.broadcastMessage((String)tm.getMessage("events.info.description", event.getDescription(), new Object[0]));
        Bukkit.broadcastMessage((String)tm.getMessage("events.info.duration", event.getDuration()));
    }

    private void broadcastEventEnd(EconomicEvent event) {
        TranslationManager tm = this.plugin.getServiceRegistry().getInstance(TranslationManager.class);
        Bukkit.broadcastMessage((String)tm.getMessage("events.banner", new Object[0]));
        Bukkit.broadcastMessage((String)tm.getMessage("events.ended", event.getName(), new Object[0]));
    }

    private void applyEventEndEffects(EconomicEvent event) {
    }

    private void applyInvestmentOpportunity(EconomicEvent event) {
        if (this.marketManager == null) {
            return;
        }
        try {
            ConfigManager cfg = this.plugin.getServiceRegistry().getInstance(ConfigManager.class);
            FileConfiguration marketCfg = cfg.getModuleConfig("market");
            FileConfiguration evcfg = cfg.getModuleConfig("events");
            int minutes = evcfg.getInt("investment_opportunity.duration_minutes", 10);
            double buyDelta = evcfg.getDouble("investment_opportunity.buy_delta", -0.02);
            double sellDelta = evcfg.getDouble("investment_opportunity.sell_delta", 0.05);
            HashSet<Material> targets = new HashSet<Material>();
            if (event.getAffectedItems() != null && !event.getAffectedItems().isEmpty()) {
                targets.addAll(event.getAffectedItems());
            } else {
                for (String s : new String[]{"EMERALD", "DIAMOND", "GOLD_INGOT"}) {
                    try {
                        targets.add(Material.valueOf((String)s));
                    }
                    catch (IllegalArgumentException illegalArgumentException) {
                        // empty catch block
                    }
                }
            }
            if (targets.isEmpty()) {
                return;
            }
            HashMap<Material, double[]> factors = new HashMap<Material, double[]>();
            for (Material m : targets) {
                factors.put(m, new double[]{1.0 + buyDelta, 1.0 + sellDelta});
            }
            this.marketManager.applyTemporaryItemFactors(factors, minutes);
            TranslationManager tm = this.plugin.getServiceRegistry().getInstance(TranslationManager.class);
            String up = String.format("%.0f", Math.max(0.0, sellDelta) * 100.0);
            Bukkit.broadcastMessage((String)(tm.getMessage("prefix", new Object[0]) + tm.getMessage("events.investment.announce", up, new Object[0])));
            this.addMetric(event, "metrics.type", "per_item");
            this.addMetric(event, "metrics.items", targets.size());
            this.addMetric(event, "metrics.buy_delta", buyDelta);
            this.addMetric(event, "metrics.sell_delta", sellDelta);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private void applyLuxuryDemand(EconomicEvent event) {
        if (this.marketManager == null) {
            return;
        }
        HashSet<Material> targets = new HashSet<Material>();
        try {
            ConfigManager cfg = this.plugin.getServiceRegistry().getInstance(ConfigManager.class);
            FileConfiguration marketCfg = cfg.getModuleConfig("market");
            FileConfiguration evcfg = cfg.getModuleConfig("events");
            String category = evcfg.getString("luxury_demand.category", "luxury");
            List mats = marketCfg.getStringList("categories." + category + ".materials");
            for (String s : mats) {
                try {
                    targets.add(Material.valueOf((String)s));
                }
                catch (IllegalArgumentException illegalArgumentException) {}
            }
            if (event.getAffectedItems() != null) {
                targets.addAll(event.getAffectedItems());
            }
            if (!targets.isEmpty()) {
                int minutes = evcfg.getInt("luxury_demand.duration_minutes", 10);
                double buyDelta = evcfg.getDouble("luxury_demand.category_buy_delta", 0.04);
                double sellDelta = evcfg.getDouble("luxury_demand.category_sell_delta", 0.08);
                HashMap<Material, double[]> factors = new HashMap<Material, double[]>();
                for (Material m : targets) {
                    factors.put(m, new double[]{1.0 + buyDelta, 1.0 + sellDelta});
                }
                this.marketManager.applyTemporaryItemFactors(factors, minutes);
                TranslationManager tm = this.plugin.getServiceRegistry().getInstance(TranslationManager.class);
                String pct = String.format("%.0f", sellDelta * 100.0);
                Bukkit.broadcastMessage((String)(tm.getMessage("prefix", new Object[0]) + tm.getMessage("events.luxury_demand.announce", pct, new Object[0])));
                this.addMetric(event, "metrics.type", "category");
                this.addMetric(event, "metrics.items", targets.size());
                this.addMetric(event, "metrics.category", category);
                this.addMetric(event, "metrics.buy_delta", buyDelta);
                this.addMetric(event, "metrics.sell_delta", sellDelta);
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private void applyMarketCorrection(EconomicEvent event) {
        try {
            ConfigManager cfg = this.plugin.getServiceRegistry().getInstance(ConfigManager.class);
            FileConfiguration evcfg = cfg.getModuleConfig("events");
            int minutes = evcfg.getInt("market_correction.duration_minutes", 10);
            double buyDelta = evcfg.getDouble("market_correction.global_buy_factor_delta", 0.02);
            double sellDelta = evcfg.getDouble("market_correction.global_sell_factor_delta", -0.02);
            double[] prev = this.marketManager.getGlobalPriceFactors();
            double newBuy = Math.max(0.5, prev[0] + buyDelta);
            double newSell = Math.max(0.5, prev[1] + sellDelta);
            this.marketManager.setGlobalPriceFactors(newBuy, newSell);
            TranslationManager tm = this.plugin.getServiceRegistry().getInstance(TranslationManager.class);
            String buyPct = String.format("%.0f", buyDelta * 100.0);
            String sellPct = String.format("%.0f", Math.abs(sellDelta) * 100.0);
            Bukkit.broadcastMessage((String)(tm.getMessage("prefix", new Object[0]) + tm.getMessage("events.market_correction.announce", buyPct, sellPct)));
            Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, () -> this.marketManager.setGlobalPriceFactors(prev[0], prev[1]), (long)minutes * 60L * 20L);
            this.addMetric(event, "metrics.type", "global");
            this.addMetric(event, "metrics.buy_delta", buyDelta);
            this.addMetric(event, "metrics.sell_delta", sellDelta);
            this.addMetric(event, "metrics.duration_minutes", minutes);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private void applyResourceShortage(EconomicEvent event) {
        if (this.marketManager == null || event.getAffectedItems() == null || event.getAffectedItems().isEmpty()) {
            return;
        }
        try {
            ConfigManager cfg = this.plugin.getServiceRegistry().getInstance(ConfigManager.class);
            FileConfiguration evcfg = cfg.getModuleConfig("events");
            int minutes = evcfg.getInt("resource_shortage.duration_minutes", 10);
            double delta = evcfg.getDouble("resource_shortage.buy_sell_delta", 0.08);
            HashMap<Material, double[]> factors = new HashMap<Material, double[]>();
            for (Material m : event.getAffectedItems()) {
                factors.put(m, new double[]{1.0 + delta, 1.0 + delta});
            }
            this.marketManager.applyTemporaryItemFactors(factors, minutes);
            TranslationManager tm = this.plugin.getServiceRegistry().getInstance(TranslationManager.class);
            String pct = String.format("%.0f", delta * 100.0);
            Bukkit.broadcastMessage((String)(tm.getMessage("prefix", new Object[0]) + tm.getMessage("events.resource_shortage.announce", pct, new Object[0])));
            this.addMetric(event, "metrics.type", "per_item");
            this.addMetric(event, "metrics.items", event.getAffectedItems().size());
            this.addMetric(event, "metrics.buy_delta", delta);
            this.addMetric(event, "metrics.sell_delta", delta);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private void applyTechnologicalBreakthrough(EconomicEvent event) {
        if (this.marketManager == null) {
            return;
        }
        try {
            ConfigManager cfg = this.plugin.getServiceRegistry().getInstance(ConfigManager.class);
            FileConfiguration marketCfg = cfg.getModuleConfig("market");
            FileConfiguration evcfg = cfg.getModuleConfig("events");
            String category = evcfg.getString("technological_breakthrough.category", "redstone");
            List mats = marketCfg.getStringList("categories." + category + ".materials");
            HashSet<Material> targets = new HashSet<Material>();
            for (String s : mats) {
                try {
                    targets.add(Material.valueOf((String)s));
                }
                catch (IllegalArgumentException illegalArgumentException) {}
            }
            if (event.getAffectedItems() != null) {
                targets.addAll(event.getAffectedItems());
            }
            if (targets.isEmpty()) {
                return;
            }
            int minutes = evcfg.getInt("technological_breakthrough.duration_minutes", 10);
            double buyDelta = evcfg.getDouble("technological_breakthrough.buy_delta", -0.05);
            double sellDelta = evcfg.getDouble("technological_breakthrough.sell_delta", -0.02);
            HashMap<Material, double[]> factors = new HashMap<Material, double[]>();
            for (Material m : targets) {
                factors.put(m, new double[]{1.0 + buyDelta, 1.0 + sellDelta});
            }
            this.marketManager.applyTemporaryItemFactors(factors, minutes);
            TranslationManager tm = this.plugin.getServiceRegistry().getInstance(TranslationManager.class);
            String buyPct = String.format("%.0f", Math.abs(buyDelta) * 100.0);
            String sellPct = String.format("%.0f", Math.abs(sellDelta) * 100.0);
            Bukkit.broadcastMessage((String)(tm.getMessage("prefix", new Object[0]) + tm.getMessage("events.tech_breakthrough.announce", buyPct, sellPct)));
            this.addMetric(event, "metrics.type", "category");
            this.addMetric(event, "metrics.items", targets.size());
            this.addMetric(event, "metrics.category", category);
            this.addMetric(event, "metrics.buy_delta", buyDelta);
            this.addMetric(event, "metrics.sell_delta", sellDelta);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private void applySeasonalDemand(EconomicEvent event) {
        if (this.marketManager == null) {
            return;
        }
        try {
            ConfigManager cfg = this.plugin.getServiceRegistry().getInstance(ConfigManager.class);
            FileConfiguration marketCfg = cfg.getModuleConfig("market");
            FileConfiguration evcfg = cfg.getModuleConfig("events");
            String category = evcfg.getString("seasonal_demand.category", "food");
            int minutes = evcfg.getInt("seasonal_demand.duration_minutes", 10);
            double buyDelta = evcfg.getDouble("seasonal_demand.buy_delta", 0.0);
            double sellDelta = evcfg.getDouble("seasonal_demand.sell_delta", 0.06);
            List mats = marketCfg.getStringList("categories." + category + ".materials");
            HashSet<Material> targets = new HashSet<Material>();
            for (Object s : mats) {
                try {
                    targets.add(Material.valueOf((String)s));
                }
                catch (IllegalArgumentException illegalArgumentException) {}
            }
            if (event.getAffectedItems() != null) {
                targets.addAll(event.getAffectedItems());
            }
            if (targets.isEmpty()) {
                return;
            }
            HashMap<Material, double[]> factors = new HashMap<Material, double[]>();
            for (Material m : targets) {
                factors.put(m, new double[]{1.0 + buyDelta, 1.0 + sellDelta});
            }
            this.marketManager.applyTemporaryItemFactors(factors, minutes);
            TranslationManager tm = this.plugin.getServiceRegistry().getInstance(TranslationManager.class);
            String pct = String.format("%.0f", Math.max(buyDelta, sellDelta) * 100.0);
            Bukkit.broadcastMessage((String)(tm.getMessage("prefix", new Object[0]) + tm.getMessage("events.seasonal_demand.announce", pct, new Object[0])));
            this.addMetric(event, "metrics.type", "category");
            this.addMetric(event, "metrics.items", targets.size());
            this.addMetric(event, "metrics.category", category);
            this.addMetric(event, "metrics.buy_delta", buyDelta);
            this.addMetric(event, "metrics.sell_delta", sellDelta);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private void applyBlackSwanEvent(EconomicEvent event) {
        try {
            ConfigManager cfg = this.plugin.getServiceRegistry().getInstance(ConfigManager.class);
            FileConfiguration evcfg = cfg.getModuleConfig("events");
            int minutes = evcfg.getInt("black_swan_event.duration_minutes", 20);
            double buyDelta = evcfg.getDouble("black_swan_event.global_buy_factor_delta", -0.1);
            double sellDelta = evcfg.getDouble("black_swan_event.global_sell_factor_delta", -0.1);
            double[] prev = this.marketManager.getGlobalPriceFactors();
            double newBuy = Math.max(0.3, prev[0] + buyDelta);
            double newSell = Math.max(0.3, prev[1] + sellDelta);
            this.marketManager.setGlobalPriceFactors(newBuy, newSell);
            TranslationManager tm = this.plugin.getServiceRegistry().getInstance(TranslationManager.class);
            String b = String.format("%.0f", Math.abs(buyDelta) * 100.0);
            String s = String.format("%.0f", Math.abs(sellDelta) * 100.0);
            Bukkit.broadcastMessage((String)(tm.getMessage("prefix", new Object[0]) + tm.getMessage("events.black_swan.announce", b, s)));
            Bukkit.getScheduler().runTaskLater((Plugin)this.plugin, () -> this.marketManager.setGlobalPriceFactors(prev[0], prev[1]), (long)minutes * 60L * 20L);
            this.addMetric(event, "metrics.type", "global");
            this.addMetric(event, "metrics.buy_delta", buyDelta);
            this.addMetric(event, "metrics.sell_delta", sellDelta);
            this.addMetric(event, "metrics.duration_minutes", minutes);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private void addMetric(EconomicEvent event, String key, Object value) {
        try {
            event.getParameters().put(key, value);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    public Map<String, EconomicEvent> getActiveEvents() {
        return new HashMap<String, EconomicEvent>(this.activeEvents);
    }

    public List<EconomicEvent> getEventHistory() {
        return new ArrayList<EconomicEvent>(this.eventHistory);
    }

    public boolean isEventEngineActive() {
        return this.eventEngineActive;
    }

    public void pauseEngine() {
        this.eventEngineActive = false;
    }

    public void resumeEngine() {
        this.eventEngineActive = true;
    }

    public int getConsecutiveQuietHours() {
        return this.consecutiveQuietHours;
    }

    public long getHoursSinceLastEvent() {
        return Duration.between(this.lastEventTime, LocalDateTime.now()).toHours();
    }

    private static class EventTemplate {
        final String name;
        final String description;
        final int baseDuration;
        final double baseIntensity;
        final List<Material> potentialItems;

        EventTemplate(String name, String description, int baseDuration, double baseIntensity, List<Material> potentialItems) {
            this.name = name;
            this.description = description;
            this.baseDuration = baseDuration;
            this.baseIntensity = baseIntensity;
            this.potentialItems = potentialItems;
        }
    }
}

