/*
 * Decompiled with CFR 0.152.
 */
package org.google.animated_frames;

import com.mojang.blaze3d.platform.NativeImage;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.metadata.IIOMetadata;
import javax.imageio.stream.ImageInputStream;
import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.texture.AbstractTexture;
import net.minecraft.client.renderer.texture.DynamicTexture;
import net.minecraft.client.renderer.texture.TextureManager;
import net.minecraft.resources.ResourceLocation;
import net.minecraftforge.api.distmarker.Dist;
import net.minecraftforge.api.distmarker.OnlyIn;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.google.animated_frames.AFConfig;
import org.google.animated_frames.Utils;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

@OnlyIn(value=Dist.CLIENT)
public class MediaHandler
implements AutoCloseable {
    private static final Logger LOGGER = LogManager.getLogger((String)"animated_frames");
    private static final ExecutorService EXECUTOR = Executors.newCachedThreadPool();
    private final List<ResourceLocation> frameTextures = new ArrayList<ResourceLocation>();
    private final List<Integer> frameDelays = new ArrayList<Integer>();
    private final AtomicInteger mediaWidth = new AtomicInteger(0);
    private final AtomicInteger mediaHeight = new AtomicInteger(0);
    private final AtomicBoolean isLoaded = new AtomicBoolean(false);
    private final TextureManager textureManager;
    private final String modId;
    private final String mediaName;
    private volatile boolean loadingFailed = false;
    private final AtomicLong estimatedMemory = new AtomicLong(0L);

    public MediaHandler(String modId, String mediaName) {
        this.modId = modId;
        this.mediaName = mediaName;
        this.textureManager = Minecraft.m_91087_().m_91097_();
        File configDir = new File("config/AnimatedFrames");
        File mediaFile = new File(configDir, (String)(mediaName.endsWith(".gif") || mediaName.endsWith(".png") ? mediaName : mediaName + ".gif"));
        try {
            byte[] mediaData = Files.readAllBytes(mediaFile.toPath());
            int maxSize = (Integer)AFConfig.MAX_FILE_SIZE_MB.get() * 1024 * 1024;
            if (!Utils.checkMemoryForGif((long)mediaData.length * 10L)) {
                LOGGER.error("Insufficient memory to load media {} (estimated size: {} bytes)", (Object)mediaName, (Object)((long)mediaData.length * 10L));
                Utils.sendErrorMessage(null, "Insufficient memory to load media " + mediaName, true);
                this.loadingFailed = true;
                return;
            }
            if (mediaData.length > maxSize) {
                LOGGER.error("Media {} exceeds max size of {}MB", (Object)mediaName, AFConfig.MAX_FILE_SIZE_MB.get());
                Utils.sendErrorMessage(null, "The media file " + mediaName + " exceeds the maximum size of " + String.valueOf(AFConfig.MAX_FILE_SIZE_MB.get()) + "MB", true);
                this.loadingFailed = true;
                return;
            }
            this.validateMediaHeader(mediaData, mediaName);
            LOGGER.info("Starting to load media {}, size: {} bytes", (Object)mediaName, (Object)mediaData.length);
            this.loadMediaAsync(mediaData);
            if (mediaData.length > 0x200000) {
                System.gc();
            }
        }
        catch (IOException e) {
            LOGGER.error("IOException loading media {}: {}", (Object)mediaName, (Object)e.getMessage(), (Object)e);
            Utils.sendErrorMessage(null, "Error loading media: " + e.getMessage(), true);
            this.loadingFailed = true;
        }
    }

    public MediaHandler(String modId, String mediaName, byte[] mediaData) {
        this.modId = modId;
        this.mediaName = mediaName;
        this.textureManager = Minecraft.m_91087_().m_91097_();
        int maxSize = (Integer)AFConfig.MAX_FILE_SIZE_MB.get() * 1024 * 1024;
        if (!Utils.checkMemoryForGif((long)mediaData.length * 10L)) {
            LOGGER.error("Insufficient memory to load media {} (estimated size: {} bytes)", (Object)mediaName, (Object)((long)mediaData.length * 10L));
            Utils.sendErrorMessage(null, "Insufficient memory to load media " + mediaName, true);
            this.loadingFailed = true;
            return;
        }
        if (mediaData.length > maxSize) {
            LOGGER.error("Media {} exceeds max size of {}MB", (Object)mediaName, AFConfig.MAX_FILE_SIZE_MB.get());
            Utils.sendErrorMessage(null, "The media file " + mediaName + " exceeds the maximum size of " + String.valueOf(AFConfig.MAX_FILE_SIZE_MB.get()) + "MB", true);
            this.loadingFailed = true;
            return;
        }
        this.validateMediaHeader(mediaData, mediaName);
        LOGGER.info("Starting to load media from bytes {}, size: {} bytes", (Object)mediaName, (Object)mediaData.length);
        this.loadMediaAsync(mediaData);
        if (mediaData.length > 0x200000) {
            System.gc();
        }
    }

    private void validateMediaHeader(byte[] data, String mediaName) {
        String extension = mediaName.substring(mediaName.lastIndexOf(46) + 1).toLowerCase();
        if (extension.equals("png")) {
            if (data.length < 8 || !new String(data, 1, 3).equals("PNG")) {
                LOGGER.error("Invalid PNG header for {}", (Object)mediaName);
                Utils.sendErrorMessage(null, "The file " + mediaName + " is not a valid PNG", true);
                this.loadingFailed = true;
            }
        } else if (extension.equals("gif")) {
            if (data.length < 6 || !new String(data, 0, 6).startsWith("GIF89a") && !new String(data, 0, 6).startsWith("GIF87a")) {
                LOGGER.error("Invalid GIF header for {}", (Object)mediaName);
                Utils.sendErrorMessage(null, "The file " + mediaName + " is not a valid GIF", true);
                this.loadingFailed = true;
            }
        } else {
            LOGGER.error("Unsupported media format for {}", (Object)mediaName);
            Utils.sendErrorMessage(null, "Unsupported media format: " + extension, true);
            this.loadingFailed = true;
        }
    }

    private void loadMediaAsync(byte[] mediaData) {
        EXECUTOR.submit(() -> {
            try (ImageInputStream stream = ImageIO.createImageInputStream(new ByteArrayInputStream(mediaData));){
                ImageReader reader = ImageIO.getImageReaders(stream).next();
                reader.setInput(stream);
                int numFrames = Math.min(reader.getNumImages(true), (Integer)AFConfig.MAX_FRAMES.get());
                for (int frameIndex = 0; frameIndex < numFrames; ++frameIndex) {
                    BufferedImage frame = reader.read(frameIndex);
                    if (frame == null) continue;
                    this.mediaWidth.compareAndSet(0, frame.getWidth());
                    this.mediaHeight.compareAndSet(0, frame.getHeight());
                    BufferedImage scaledFrame = this.scaleImage(frame);
                    if (scaledFrame == null) continue;
                    long frameMemory = (long)scaledFrame.getWidth() * (long)scaledFrame.getHeight() * 4L;
                    if (!Utils.checkMemoryForGif(frameMemory)) {
                        LOGGER.error("Insufficient memory for frame {} of {} (size: {} bytes)", (Object)frameIndex, (Object)this.mediaName, (Object)frameMemory);
                        Utils.sendErrorMessage(null, "Insufficient memory for frame " + frameIndex + " of " + this.mediaName, true);
                        this.loadingFailed = true;
                        return;
                    }
                    this.estimatedMemory.addAndGet(frameMemory);
                    int delay = this.getFrameDelay(reader, frameIndex);
                    this.frameDelays.add(delay);
                    ResourceLocation texture = this.registerTexture(scaledFrame, frameIndex);
                    if (texture == null) continue;
                    this.frameTextures.add(texture);
                }
                if (this.frameTextures.isEmpty()) {
                    this.loadingFailed = true;
                    LOGGER.error("No frames loaded for media {}", (Object)this.mediaName);
                    Utils.sendErrorMessage(null, "No frames could be loaded for " + this.mediaName, true);
                } else {
                    this.isLoaded.set(true);
                    LOGGER.info("Loaded {} frames for media {}", (Object)this.frameTextures.size(), (Object)this.mediaName);
                }
            }
            catch (Exception e) {
                LOGGER.error("Error loading media {}: {}", (Object)this.mediaName, (Object)e.getMessage(), (Object)e);
                Utils.sendErrorMessage(null, "Error loading media: " + e.getMessage(), true);
                this.loadingFailed = true;
            }
        });
    }

    private BufferedImage scaleImage(BufferedImage original) {
        if (original == null) {
            return null;
        }
        int newWidth = (int)((double)original.getWidth() * (Double)AFConfig.SCALE_FACTOR.get());
        int newHeight = (int)((double)original.getHeight() * (Double)AFConfig.SCALE_FACTOR.get());
        try {
            BufferedImage scaled = new BufferedImage(newWidth, newHeight, 2);
            Graphics2D g2d = scaled.createGraphics();
            g2d.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
            g2d.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
            g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
            g2d.drawImage(original, 0, 0, newWidth, newHeight, null);
            g2d.dispose();
            original.flush();
            return scaled;
        }
        catch (Exception e) {
            LOGGER.error("Error scaling media for {}: {}", (Object)this.mediaName, (Object)e.getMessage(), (Object)e);
            return null;
        }
    }

    private int getFrameDelay(ImageReader reader, int frameIndex) {
        try {
            IIOMetadata metadata = reader.getImageMetadata(frameIndex);
            Node root = metadata.getAsTree("javax_imageio_gif_image_1.0");
            NodeList children = root.getChildNodes();
            for (int i = 0; i < children.getLength(); ++i) {
                String delayTime;
                Node node = children.item(i);
                if (!"GraphicControlExtension".equals(node.getNodeName()) || (delayTime = ((Element)node).getAttribute("delayTime")) == null || delayTime.isEmpty()) continue;
                int delay = Integer.parseInt(delayTime) * 10;
                LOGGER.debug("Frame {} delay for media {}: {}ms", (Object)frameIndex, (Object)this.mediaName, (Object)delay);
                return Math.max(delay, 50);
            }
            LOGGER.debug("No delay found for frame {} of media {}, using default 100ms", (Object)frameIndex, (Object)this.mediaName);
            return 100;
        }
        catch (Exception e) {
            LOGGER.error("Error getting delay for frame {} of media {}: {}", (Object)frameIndex, (Object)this.mediaName, (Object)e.getMessage(), (Object)e);
            return 100;
        }
    }

    private ResourceLocation registerTexture(BufferedImage image, int frameIndex) {
        if (image == null) {
            LOGGER.error("Cannot register null image for frame {} of {}", (Object)frameIndex, (Object)this.mediaName);
            Utils.sendErrorMessage(null, "Cannot register null image for frame " + frameIndex + " of " + this.mediaName, true);
            return null;
        }
        try {
            NativeImage nativeImage = this.convertToNativeImageOptimized(image);
            if (nativeImage == null) {
                LOGGER.error("Failed to convert frame {} to NativeImage for {}", (Object)frameIndex, (Object)this.mediaName);
                Utils.sendErrorMessage(null, "Failed to convert frame " + frameIndex + " to NativeImage for " + this.mediaName, true);
                return null;
            }
            if (!Utils.checkMemoryForGif((long)(nativeImage.m_84982_() * nativeImage.m_85084_()) * 4L)) {
                LOGGER.error("Insufficient memory to register texture for frame {} of {}", (Object)frameIndex, (Object)this.mediaName);
                Utils.sendErrorMessage(null, "Insufficient memory to register texture for frame " + frameIndex + " of " + this.mediaName, true);
                nativeImage.close();
                return null;
            }
            DynamicTexture texture = new DynamicTexture(nativeImage);
            ResourceLocation resourceLocation = new ResourceLocation(this.modId, "media_" + this.mediaName + "_" + frameIndex);
            this.textureManager.m_118495_(resourceLocation, (AbstractTexture)texture);
            LOGGER.debug("Successfully registered texture for frame {} of {}", (Object)frameIndex, (Object)this.mediaName);
            return resourceLocation;
        }
        catch (Exception e) {
            LOGGER.error("Error registering texture for frame {} of {}: {}", (Object)frameIndex, (Object)this.mediaName, (Object)e.getMessage(), (Object)e);
            Utils.sendErrorMessage(null, "Error registering texture for frame " + frameIndex + " for " + this.mediaName + ": " + e.getMessage(), true);
            return null;
        }
    }

    private NativeImage convertToNativeImageOptimized(BufferedImage image) {
        if (image == null) {
            LOGGER.error("Cannot convert null image for {}", (Object)this.mediaName);
            Utils.sendErrorMessage(null, "Cannot convert null image for " + this.mediaName, true);
            return null;
        }
        try {
            NativeImage nativeImage = new NativeImage(image.getWidth(), image.getHeight(), true);
            for (int y = 0; y < image.getHeight(); ++y) {
                for (int x = 0; x < image.getWidth(); ++x) {
                    int argb = image.getRGB(x, y);
                    int alpha = argb >> 24 & 0xFF;
                    int red = argb >> 16 & 0xFF;
                    int green = argb >> 8 & 0xFF;
                    int blue = argb & 0xFF;
                    int abgr = alpha << 24 | blue << 16 | green << 8 | red;
                    nativeImage.m_84988_(x, y, abgr);
                }
            }
            image.flush();
            return nativeImage;
        }
        catch (Exception e) {
            LOGGER.error("Error converting BufferedImage to NativeImage for {}: {}", (Object)this.mediaName, (Object)e.getMessage(), (Object)e);
            Utils.sendErrorMessage(null, "Error converting media for " + this.mediaName + ": " + e.getMessage(), true);
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<ResourceLocation> getFrameTextures() {
        List<ResourceLocation> list = this.frameTextures;
        synchronized (list) {
            return new ArrayList<ResourceLocation>(this.frameTextures);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<Integer> getFrameDelays() {
        List<Integer> list = this.frameDelays;
        synchronized (list) {
            return new ArrayList<Integer>(this.frameDelays);
        }
    }

    public int getMediaWidth() {
        return this.mediaWidth.get();
    }

    public int getMediaHeight() {
        return this.mediaHeight.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean isLoaded() {
        List<ResourceLocation> list = this.frameTextures;
        synchronized (list) {
            return this.isLoaded.get() && !this.loadingFailed && !this.frameTextures.isEmpty() && this.frameTextures.size() == this.frameDelays.size() && this.frameTextures.stream().allMatch(t -> t != null);
        }
    }

    public boolean hasFailed() {
        return this.loadingFailed;
    }

    public String getMediaName() {
        return this.mediaName;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        List<ResourceLocation> list = this.frameTextures;
        synchronized (list) {
            for (ResourceLocation texture : this.frameTextures) {
                this.textureManager.m_118513_(texture);
            }
            this.frameTextures.clear();
            this.frameDelays.clear();
            this.isLoaded.set(false);
            LOGGER.info("Closed media handler for {}", (Object)this.mediaName);
        }
    }
}

