/*
 * Decompiled with CFR 0.152.
 */
package com.nsr.ai.plugin.util;

import com.nsr.ai.plugin.NSRAIPlugin;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Base64;
import java.util.UUID;
import java.util.logging.Level;
import javax.crypto.Cipher;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.SecretKeySpec;

public class SecurityUtil {
    private static final String ALGORITHM = "AES";
    private static final String CIPHER_TRANSFORMATION = "AES/GCM/NoPadding";
    private static final int GCM_IV_LENGTH = 12;
    private static final int GCM_TAG_LENGTH = 16;
    private static final int KEY_LENGTH_BITS = 256;
    private static final String MASTER_KEY_FILE = "master.key";
    private static final String SALT_FILE = "server.salt";
    private static final String MASTER_SECRET_CONF_FILE = "master_secret.conf";
    private static final String SECURITY_FOLDER_NAME = "security";
    private final NSRAIPlugin plugin;
    private SecretKey serverMasterKey;
    private byte[] serverSalt;
    private byte[] masterSecretBytes;

    public SecurityUtil(NSRAIPlugin plugin) {
        this.plugin = plugin;
        this.loadOrCreateServerKeysAndSalt();
    }

    private void loadOrCreateServerKeysAndSalt() {
        File securityFolder = new File(this.plugin.getDataFolder(), SECURITY_FOLDER_NAME);
        if (!securityFolder.exists()) {
            securityFolder.mkdirs();
        }
        File masterKeyFile = new File(securityFolder, MASTER_KEY_FILE);
        File saltFile = new File(securityFolder, SALT_FILE);
        File masterSecretConfFile = new File(securityFolder, MASTER_SECRET_CONF_FILE);
        try {
            if (masterSecretConfFile.exists()) {
                this.masterSecretBytes = Files.readAllBytes(masterSecretConfFile.toPath());
                this.plugin.getLogger().info("[NSR-AI Security] Loaded master secret.");
            } else {
                this.masterSecretBytes = this.generateRandomBytes(32);
                Files.write(masterSecretConfFile.toPath(), this.masterSecretBytes, new OpenOption[0]);
                this.plugin.getLogger().info("[NSR-AI Security] Generated and saved new master secret.");
            }
            if (saltFile.exists()) {
                this.serverSalt = Files.readAllBytes(saltFile.toPath());
                this.plugin.getLogger().info("[NSR-AI Security] Loaded server salt.");
            } else {
                this.serverSalt = this.generateRandomBytes(16);
                Files.write(saltFile.toPath(), this.serverSalt, new OpenOption[0]);
                this.plugin.getLogger().info("[NSR-AI Security] Generated and saved new server salt.");
            }
            if (masterKeyFile.exists()) {
                byte[] encodedKey = Files.readAllBytes(masterKeyFile.toPath());
                this.serverMasterKey = new SecretKeySpec(encodedKey, ALGORITHM);
                this.plugin.getLogger().info("[NSR-AI Security] Loaded server master key.");
            } else {
                this.serverMasterKey = this.generateMasterKey(this.masterSecretBytes, this.serverSalt);
                Files.write(masterKeyFile.toPath(), this.serverMasterKey.getEncoded(), new OpenOption[0]);
                this.plugin.getLogger().info("[NSR-AI Security] Generated and saved new server master key.");
            }
        }
        catch (IOException | InvalidKeyException | NoSuchAlgorithmException e) {
            this.plugin.getLogger().log(Level.SEVERE, "[NSR-AI Security] Failed to load or create server master key/salt/secret. Plugin may not function correctly.", e);
        }
    }

    private byte[] hkdfExtract(byte[] salt, byte[] ikm) throws NoSuchAlgorithmException, InvalidKeyException {
        Mac hmacSha256 = Mac.getInstance("HmacSHA256");
        if (salt == null || salt.length == 0) {
            salt = new byte[hmacSha256.getMacLength()];
        }
        hmacSha256.init(new SecretKeySpec(salt, "HmacSHA256"));
        return hmacSha256.doFinal(ikm);
    }

    private byte[] hkdfExpand(byte[] prk, byte[] info, int outputLength) throws NoSuchAlgorithmException, InvalidKeyException {
        Mac hmacSha256 = Mac.getInstance("HmacSHA256");
        hmacSha256.init(new SecretKeySpec(prk, "HmacSHA256"));
        byte[] okm = new byte[outputLength];
        byte[] T = new byte[]{};
        int bytesRemaining = outputLength;
        byte counter = 1;
        while (bytesRemaining > 0) {
            hmacSha256.update(T);
            if (info != null && info.length > 0) {
                hmacSha256.update(info);
            }
            hmacSha256.update(counter);
            T = hmacSha256.doFinal();
            int bytesToCopy = Math.min(bytesRemaining, T.length);
            System.arraycopy(T, 0, okm, outputLength - bytesRemaining, bytesToCopy);
            bytesRemaining -= bytesToCopy;
            counter = (byte)(counter + 1);
        }
        return okm;
    }

    public SecretKey generateMasterKey(byte[] secret, byte[] salt) throws NoSuchAlgorithmException, InvalidKeyException {
        byte[] prk = this.hkdfExtract(salt, secret);
        byte[] derivedKeyBytes = this.hkdfExpand(prk, null, 32);
        return new SecretKeySpec(derivedKeyBytes, ALGORITHM);
    }

    public SecretKey deriveUserKey(UUID playerUuid, String projectId) throws NoSuchAlgorithmException, InvalidKeyException {
        String userKeyMaterial = playerUuid.toString() + ":" + projectId;
        byte[] userKeyBytes = userKeyMaterial.getBytes(StandardCharsets.UTF_8);
        byte[] prk = this.hkdfExtract(this.serverSalt, userKeyBytes);
        byte[] derivedKeyBytes = this.hkdfExpand(prk, null, 32);
        return new SecretKeySpec(derivedKeyBytes, ALGORITHM);
    }

    public String encrypt(String plainText, SecretKey secretKey) throws Exception {
        byte[] plainTextBytes = plainText.getBytes(StandardCharsets.UTF_8);
        byte[] iv = this.generateRandomBytes(12);
        Cipher cipher = Cipher.getInstance(CIPHER_TRANSFORMATION);
        cipher.init(1, (Key)secretKey, new GCMParameterSpec(128, iv));
        byte[] cipherText = cipher.doFinal(plainTextBytes);
        byte[] encryptedData = new byte[iv.length + cipherText.length];
        System.arraycopy(iv, 0, encryptedData, 0, iv.length);
        System.arraycopy(cipherText, 0, encryptedData, iv.length, cipherText.length);
        return Base64.getEncoder().encodeToString(encryptedData);
    }

    public String decrypt(String encryptedText, SecretKey secretKey) throws Exception {
        byte[] decodedData = Base64.getDecoder().decode(encryptedText);
        if (decodedData.length < 28) {
            throw new IllegalArgumentException("Encrypted data is too short to contain IV and GCM tag.");
        }
        byte[] iv = Arrays.copyOfRange(decodedData, 0, 12);
        byte[] cipherText = Arrays.copyOfRange(decodedData, 12, decodedData.length);
        Cipher cipher = Cipher.getInstance(CIPHER_TRANSFORMATION);
        cipher.init(2, (Key)secretKey, new GCMParameterSpec(128, iv));
        byte[] plainTextBytes = cipher.doFinal(cipherText);
        return new String(plainTextBytes, StandardCharsets.UTF_8);
    }

    public String hashSha256(String data) throws NoSuchAlgorithmException {
        MessageDigest digest = MessageDigest.getInstance("SHA-256");
        byte[] hash = digest.digest(data.getBytes(StandardCharsets.UTF_8));
        return Base64.getEncoder().encodeToString(hash);
    }

    public byte[] generateRandomBytes(int length) {
        byte[] bytes = new byte[length];
        new SecureRandom().nextBytes(bytes);
        return bytes;
    }

    public SecretKey getServerMasterKey() {
        return this.serverMasterKey;
    }
}

