/*
 * Decompiled with CFR 0.152.
 */
package org.stepan.audio_disc.download;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.concurrent.CompletableFuture;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import net.fabricmc.loader.api.FabricLoader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.stepan.audio_disc.util.FilePermissionUtil;

public class FFmpegManager {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"AudioDisc");
    private static final String WINDOWS_URL = "https://www.gyan.dev/ffmpeg/builds/ffmpeg-release-essentials.zip";
    private static final String LINUX_AMD64_URL = "https://johnvansickle.com/ffmpeg/releases/ffmpeg-release-amd64-static.tar.xz";
    private static final String LINUX_ARM64_URL = "https://johnvansickle.com/ffmpeg/releases/ffmpeg-release-arm64-static.tar.xz";
    private static final Path FFMPEG_DIR = FabricLoader.getInstance().getGameDir().resolve("audiodisc").resolve("ffmpeg");
    private static Path ffmpegExecutable = null;

    public static boolean isAvailable() {
        if (FFmpegManager.isSystemFFmpegAvailable()) {
            return true;
        }
        return FFmpegManager.isModFFmpegAvailable();
    }

    public static String getExecutablePath() {
        if (FFmpegManager.isSystemFFmpegAvailable()) {
            return "ffmpeg";
        }
        if (ffmpegExecutable != null && Files.exists(ffmpegExecutable, new LinkOption[0])) {
            return ffmpegExecutable.toString();
        }
        return null;
    }

    public static CompletableFuture<Boolean> downloadAndInstall() {
        return CompletableFuture.supplyAsync(() -> {
            try {
                String downloadUrl;
                LOGGER.info("Starting automatic FFmpeg download...");
                if (!Files.exists(FFMPEG_DIR, new LinkOption[0])) {
                    Files.createDirectories(FFMPEG_DIR, new FileAttribute[0]);
                }
                if ((downloadUrl = FFmpegManager.getDownloadUrlForPlatform()) == null) {
                    FFmpegManager.logUnsupportedPlatform();
                    return false;
                }
                boolean success = FFmpegManager.isWindows() ? FFmpegManager.downloadAndInstallWindows(downloadUrl) : FFmpegManager.downloadAndInstallLinux(downloadUrl);
                if (!success) {
                    return false;
                }
                Path executable = FFmpegManager.findFFmpegExecutable();
                if (executable == null) {
                    LOGGER.error("FFmpeg executable not found after extraction");
                    FFmpegManager.logExtractionFailure();
                    return false;
                }
                ffmpegExecutable = executable;
                LOGGER.info("Successfully downloaded and installed FFmpeg to: {}", (Object)executable);
                if (FFmpegManager.testFFmpeg(executable.toString())) {
                    LOGGER.info("FFmpeg installation verified successfully");
                    return true;
                }
                FFmpegManager.logVerificationFailure(executable);
                return false;
            }
            catch (Exception e) {
                LOGGER.error("Failed to download and install FFmpeg", (Throwable)e);
                return false;
            }
        });
    }

    private static boolean isSystemFFmpegAvailable() {
        return FFmpegManager.testFFmpeg("ffmpeg");
    }

    private static boolean isModFFmpegAvailable() {
        Path executable;
        if (ffmpegExecutable == null && (executable = FFmpegManager.findFFmpegExecutable()) != null && Files.exists(executable, new LinkOption[0])) {
            ffmpegExecutable = executable;
            return FFmpegManager.testFFmpeg(executable.toString());
        }
        return ffmpegExecutable != null && Files.exists(ffmpegExecutable, new LinkOption[0]) && FFmpegManager.testFFmpeg(ffmpegExecutable.toString());
    }

    private static boolean downloadAndInstallWindows(String downloadUrl) {
        try {
            LOGGER.info("Downloading FFmpeg from: {}", (Object)downloadUrl);
            Path tempZip = FFMPEG_DIR.resolve("ffmpeg-temp.zip");
            if (!FFmpegManager.downloadFile(downloadUrl, tempZip)) {
                FFmpegManager.logDownloadFailure();
                return false;
            }
            LOGGER.info("Extracting FFmpeg...");
            if (!FFmpegManager.extractFFmpeg(tempZip)) {
                FFmpegManager.logExtractionFailure();
                return false;
            }
            try {
                Files.deleteIfExists(tempZip);
            }
            catch (Exception e) {
                LOGGER.warn("Failed to delete temp file: {}", (Object)e.getMessage());
            }
            return true;
        }
        catch (Exception e) {
            LOGGER.error("Failed to install FFmpeg on Windows", (Throwable)e);
            return false;
        }
    }

    private static boolean downloadAndInstallLinux(String downloadUrl) {
        try {
            LOGGER.info("Downloading FFmpeg from: {}", (Object)downloadUrl);
            Path tempTar = FFMPEG_DIR.resolve("ffmpeg-temp.tar.xz");
            if (!FFmpegManager.downloadFile(downloadUrl, tempTar)) {
                FFmpegManager.logDownloadFailure();
                return false;
            }
            LOGGER.info("Extracting FFmpeg...");
            if (!FFmpegManager.extractFFmpegLinux(tempTar)) {
                FFmpegManager.logExtractionFailure();
                return false;
            }
            Path executable = FFMPEG_DIR.resolve("ffmpeg");
            if (Files.exists(executable, new LinkOption[0]) && !FilePermissionUtil.setExecutablePermissions(executable)) {
                FFmpegManager.logPermissionFailure(executable);
            }
            try {
                Files.deleteIfExists(tempTar);
            }
            catch (Exception e) {
                LOGGER.warn("Failed to delete temp file: {}", (Object)e.getMessage());
            }
            return true;
        }
        catch (Exception e) {
            LOGGER.error("Failed to install FFmpeg on Linux", (Throwable)e);
            return false;
        }
    }

    private static boolean extractFFmpegLinux(Path tarFile) {
        try {
            LOGGER.info("Extracting FFmpeg from tar.xz archive...");
            ProcessBuilder pb = new ProcessBuilder("tar", "-xJf", tarFile.toString(), "-C", FFMPEG_DIR.toString(), "--strip-components=1", "--wildcards", "*/ffmpeg");
            pb.redirectErrorStream(true);
            Process process = pb.start();
            StringBuilder output = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    output.append(line).append("\n");
                    LOGGER.debug("tar output: {}", (Object)line);
                }
            }
            int exitCode = process.waitFor();
            if (exitCode == 0) {
                LOGGER.info("FFmpeg extracted successfully");
                return true;
            }
            LOGGER.error("tar extraction failed with exit code {}: {}", (Object)exitCode, (Object)output);
            return false;
        }
        catch (IOException e) {
            LOGGER.error("Failed to execute tar command (tar may not be installed): {}", (Object)e.getMessage());
            return false;
        }
        catch (Exception e) {
            LOGGER.error("Failed to extract FFmpeg on Linux", (Throwable)e);
            return false;
        }
    }

    private static String getLinuxArchitecture() {
        String arch = System.getProperty("os.arch").toLowerCase();
        if (arch.contains("amd64") || arch.contains("x86_64")) {
            return "amd64";
        }
        if (arch.contains("aarch64") || arch.contains("arm64")) {
            return "arm64";
        }
        return null;
    }

    private static boolean testFFmpeg(String executable) {
        try {
            ProcessBuilder pb = new ProcessBuilder(executable, "-version");
            pb.redirectErrorStream(true);
            Process process = pb.start();
            StringBuilder output = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    output.append(line).append("\n");
                }
            }
            int exitCode = process.waitFor();
            if (exitCode == 0) {
                LOGGER.debug("FFmpeg version check successful: {}", (Object)output.toString().split("\n")[0]);
                return true;
            }
            LOGGER.error("FFmpeg version check failed with exit code {}: {}", (Object)exitCode, (Object)output);
            return false;
        }
        catch (IOException e) {
            LOGGER.error("Failed to execute FFmpeg (file may not be executable or not found): {}", (Object)e.getMessage());
            return false;
        }
        catch (Exception e) {
            LOGGER.error("FFmpeg verification error: {}", (Object)e.getMessage());
            return false;
        }
    }

    private static boolean downloadFile(String urlString, Path destination) {
        try {
            URL url = new URL(urlString);
            HttpURLConnection connection = (HttpURLConnection)url.openConnection();
            connection.setRequestMethod("GET");
            connection.setConnectTimeout(30000);
            connection.setReadTimeout(300000);
            connection.setRequestProperty("User-Agent", "AudioDisc-Mod/1.0");
            int responseCode = connection.getResponseCode();
            if (responseCode != 200) {
                LOGGER.error("Failed to download FFmpeg: HTTP {}", (Object)responseCode);
                return false;
            }
            long fileSize = connection.getContentLengthLong();
            LOGGER.info("Downloading FFmpeg ({} MB)...", (Object)(fileSize > 0L ? String.format("%.1f", (double)fileSize / 1024.0 / 1024.0) : "unknown size"));
            try (InputStream in = connection.getInputStream();){
                Files.copy(in, destination, StandardCopyOption.REPLACE_EXISTING);
            }
            LOGGER.info("Download completed: {}", (Object)destination);
            return true;
        }
        catch (Exception e) {
            LOGGER.error("Failed to download FFmpeg", (Throwable)e);
            return false;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static boolean extractFFmpeg(Path zipFile) {
        try {
            try (ZipInputStream zis = new ZipInputStream(Files.newInputStream(zipFile, new OpenOption[0]));){
                ZipEntry entry;
                while ((entry = zis.getNextEntry()) != null) {
                    String entryName = entry.getName();
                    if (entryName.endsWith("ffmpeg.exe") && !entry.isDirectory()) {
                        Path outputPath = FFMPEG_DIR.resolve("ffmpeg.exe");
                        LOGGER.info("Extracting FFmpeg executable: {} -> {}", (Object)entryName, (Object)outputPath);
                        try (OutputStream out = Files.newOutputStream(outputPath, new OpenOption[0]);){
                            int length;
                            byte[] buffer = new byte[8192];
                            while ((length = zis.read(buffer)) != -1) {
                                out.write(buffer, 0, length);
                            }
                        }
                        LOGGER.info("FFmpeg extracted successfully");
                        boolean bl = true;
                        return bl;
                    }
                    zis.closeEntry();
                }
            }
            LOGGER.error("ffmpeg.exe not found in downloaded archive");
            return false;
        }
        catch (Exception e) {
            LOGGER.error("Failed to extract FFmpeg", (Throwable)e);
            return false;
        }
    }

    private static Path findFFmpegExecutable() {
        try {
            if (FFmpegManager.isWindows()) {
                Path executable = FFMPEG_DIR.resolve("ffmpeg.exe");
                if (Files.exists(executable, new LinkOption[0])) {
                    return executable;
                }
            } else {
                Path executable = FFMPEG_DIR.resolve("ffmpeg");
                if (Files.exists(executable, new LinkOption[0])) {
                    return executable;
                }
            }
        }
        catch (Exception e) {
            LOGGER.warn("Error finding FFmpeg executable: {}", (Object)e.getMessage());
        }
        return null;
    }

    private static String getDownloadUrlForPlatform() {
        String os = System.getProperty("os.name").toLowerCase();
        if (os.contains("win")) {
            return WINDOWS_URL;
        }
        if (os.contains("nix") || os.contains("nux") || os.contains("aix")) {
            String arch = FFmpegManager.getLinuxArchitecture();
            if ("amd64".equals(arch)) {
                LOGGER.info("Detected Linux x86_64 architecture");
                return LINUX_AMD64_URL;
            }
            if ("arm64".equals(arch)) {
                LOGGER.info("Detected Linux ARM64 architecture");
                return LINUX_ARM64_URL;
            }
            LOGGER.warn("Unsupported Linux architecture: {}", (Object)System.getProperty("os.arch"));
            return null;
        }
        return null;
    }

    private static void logDownloadFailure() {
        LOGGER.error("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
        LOGGER.error("Failed to download FFmpeg automatically");
        LOGGER.error("M4A/WebM conversion will not be available");
        LOGGER.error("");
        LOGGER.error("Manual installation options:");
        LOGGER.error("1. Install FFmpeg system-wide:");
        LOGGER.error("   - Linux: sudo apt install ffmpeg  OR  sudo yum install ffmpeg");
        LOGGER.error("   - macOS: brew install ffmpeg");
        LOGGER.error("   - Windows: Download from https://ffmpeg.org/download.html");
        LOGGER.error("2. Disable auto-install in config: autoInstallFFmpeg = false");
        LOGGER.error("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
    }

    private static void logExtractionFailure() {
        LOGGER.error("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
        LOGGER.error("Failed to extract FFmpeg");
        LOGGER.error("M4A/WebM conversion will not be available");
        LOGGER.error("");
        LOGGER.error("Possible causes:");
        LOGGER.error("1. On Linux: tar command may not be installed");
        LOGGER.error("   Install with: sudo apt install tar");
        LOGGER.error("2. Downloaded file may be corrupted");
        LOGGER.error("3. Insufficient disk space");
        LOGGER.error("");
        LOGGER.error("Alternative: Install FFmpeg system-wide:");
        LOGGER.error("   - Linux: sudo apt install ffmpeg");
        LOGGER.error("   - macOS: brew install ffmpeg");
        LOGGER.error("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
    }

    private static void logPermissionFailure(Path executable) {
        LOGGER.warn("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
        LOGGER.warn("Failed to set executable permissions for FFmpeg");
        LOGGER.warn("File location: {}", (Object)executable);
        LOGGER.warn("");
        LOGGER.warn("Manual fix:");
        LOGGER.warn("Run this command in your terminal:");
        LOGGER.warn("  chmod +x {}", (Object)executable);
        LOGGER.warn("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
    }

    private static void logVerificationFailure(Path executable) {
        LOGGER.error("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
        LOGGER.error("FFmpeg installation verification failed");
        LOGGER.error("M4A/WebM conversion will not be available");
        LOGGER.error("");
        LOGGER.error("Troubleshooting steps:");
        LOGGER.error("1. Check if file has executable permissions:");
        LOGGER.error("   ls -l {}", (Object)executable);
        LOGGER.error("");
        LOGGER.error("2. If permissions are missing, run:");
        LOGGER.error("   chmod +x {}", (Object)executable);
        LOGGER.error("");
        LOGGER.error("3. Test manually:");
        LOGGER.error("   {} -version", (Object)executable);
        LOGGER.error("");
        LOGGER.error("4. Alternative: Install FFmpeg system-wide:");
        LOGGER.error("   - Linux: sudo apt install ffmpeg");
        LOGGER.error("   - macOS: brew install ffmpeg");
        LOGGER.error("");
        LOGGER.error("5. Or disable auto-install in config: autoInstallFFmpeg = false");
        LOGGER.error("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
    }

    private static void logUnsupportedPlatform() {
        String os = System.getProperty("os.name");
        String arch = System.getProperty("os.arch");
        LOGGER.error("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
        LOGGER.error("Unsupported platform for automatic FFmpeg download");
        LOGGER.error("OS: {}, Architecture: {}", (Object)os, (Object)arch);
        LOGGER.error("");
        LOGGER.error("Please install FFmpeg manually:");
        LOGGER.error("1. Visit: https://ffmpeg.org/download.html");
        LOGGER.error("2. Download FFmpeg for your platform");
        LOGGER.error("3. Install it system-wide or place in PATH");
        LOGGER.error("");
        LOGGER.error("Or disable auto-install in config: autoInstallFFmpeg = false");
        LOGGER.error("\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
    }

    private static boolean isWindows() {
        return System.getProperty("os.name").toLowerCase().contains("win");
    }

    public static long getInstallationSize() {
        if (ffmpegExecutable != null && Files.exists(ffmpegExecutable, new LinkOption[0])) {
            try {
                return Files.size(ffmpegExecutable);
            }
            catch (IOException e) {
                return 0L;
            }
        }
        return 0L;
    }

    public static boolean uninstall() {
        try {
            if (ffmpegExecutable != null && Files.exists(ffmpegExecutable, new LinkOption[0])) {
                Files.delete(ffmpegExecutable);
                ffmpegExecutable = null;
                LOGGER.info("Uninstalled mod-installed FFmpeg");
                return true;
            }
            return false;
        }
        catch (Exception e) {
            LOGGER.error("Failed to uninstall FFmpeg", (Throwable)e);
            return false;
        }
    }
}

