/*
 * Decompiled with CFR 0.152.
 */
package org.stepan.audio_disc.processing;

import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.stepan.audio_disc.model.AudioMetadata;
import org.stepan.audio_disc.model.ValidationResult;

public class AudioProcessor {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"AudioDisc");
    private static final Set<String> SUPPORTED_FORMATS = Set.of("mp3", "wav", "ogg", "webm", "m4a");
    private static final long MAX_FILE_SIZE = 0x3200000L;
    private static final byte[] MP3_MAGIC = new byte[]{-1, -5};
    private static final byte[] MP3_MAGIC_ALT = new byte[]{-1, -13};
    private static final byte[] MP3_ID3 = new byte[]{73, 68, 51};
    private static final byte[] OGG_MAGIC = new byte[]{79, 103, 103, 83};
    private static final byte[] WAV_RIFF = new byte[]{82, 73, 70, 70};
    private static final byte[] WAV_WAVE = new byte[]{87, 65, 86, 69};
    private static final byte[] WEBM_MAGIC = new byte[]{26, 69, -33, -93};
    private static final byte[] M4A_FTYP = new byte[]{102, 116, 121, 112};

    public ValidationResult validateFormat(byte[] audioData) {
        if (audioData == null || audioData.length == 0) {
            return ValidationResult.failure("Audio data is empty");
        }
        if ((long)audioData.length > 0x3200000L) {
            return ValidationResult.failure("File size exceeds maximum allowed size of 50MB");
        }
        String detectedFormat = this.detectFormat(audioData);
        if (detectedFormat == null) {
            return ValidationResult.failure("Unsupported or unrecognized audio format");
        }
        if (!SUPPORTED_FORMATS.contains(detectedFormat)) {
            return ValidationResult.failure("Format '" + detectedFormat + "' is not supported");
        }
        LOGGER.debug("Audio format validated: {}", (Object)detectedFormat);
        return ValidationResult.success();
    }

    private String detectFormat(byte[] audioData) {
        if (audioData.length < 12) {
            return null;
        }
        if (this.startsWith(audioData, MP3_ID3)) {
            return "mp3";
        }
        if (this.startsWith(audioData, MP3_MAGIC) || this.startsWith(audioData, MP3_MAGIC_ALT)) {
            return "mp3";
        }
        if (this.startsWith(audioData, OGG_MAGIC)) {
            return "ogg";
        }
        if (this.startsWith(audioData, WAV_RIFF) && this.hasWaveFormat(audioData)) {
            return "wav";
        }
        if (this.startsWith(audioData, WEBM_MAGIC)) {
            return "webm";
        }
        if (this.hasM4AFormat(audioData)) {
            return "m4a";
        }
        return null;
    }

    private boolean startsWith(byte[] data, byte[] pattern) {
        if (data.length < pattern.length) {
            return false;
        }
        for (int i = 0; i < pattern.length; ++i) {
            if (data[i] == pattern[i]) continue;
            return false;
        }
        return true;
    }

    private boolean hasWaveFormat(byte[] data) {
        if (data.length < 12) {
            return false;
        }
        for (int i = 0; i < WAV_WAVE.length; ++i) {
            if (data[8 + i] == WAV_WAVE[i]) continue;
            return false;
        }
        return true;
    }

    private boolean hasM4AFormat(byte[] data) {
        if (data.length < 12) {
            return false;
        }
        for (int i = 0; i < M4A_FTYP.length; ++i) {
            if (data[4 + i] == M4A_FTYP[i]) continue;
            return false;
        }
        if (data.length >= 16) {
            String brand = new String(data, 8, 4);
            return brand.equals("M4A ") || brand.equals("mp41") || brand.equals("mp42") || brand.equals("isom");
        }
        return true;
    }

    public AudioMetadata extractMetadata(byte[] audioData) {
        String format = this.detectFormat(audioData);
        if (format == null) {
            format = "unknown";
        }
        long duration = this.estimateDuration(audioData, format);
        int bitrate = this.estimateBitrate(audioData, format);
        int sampleRate = this.estimateSampleRate(audioData, format);
        String title = this.extractTitle(audioData, format);
        LOGGER.debug("Extracted metadata - Format: {}, Duration: {}ms, Bitrate: {}kbps", new Object[]{format, duration, bitrate});
        return new AudioMetadata(format, duration, bitrate, sampleRate, title);
    }

    private long estimateDuration(byte[] audioData, String format) {
        long fileSize = audioData.length;
        switch (format) {
            case "mp3": {
                return fileSize * 8L / 128L;
            }
            case "ogg": {
                return fileSize * 8L / 128L;
            }
            case "m4a": {
                return fileSize * 8L / 128L;
            }
            case "webm": {
                return fileSize * 8L / 128L;
            }
            case "wav": {
                return this.estimateWavDuration(audioData);
            }
        }
        return 0L;
    }

    private long estimateWavDuration(byte[] audioData) {
        if (audioData.length < 44) {
            return 0L;
        }
        try {
            int sampleRate = this.readLittleEndianInt(audioData, 24);
            int byteRate = this.readLittleEndianInt(audioData, 28);
            if (byteRate > 0) {
                long dataSize = audioData.length - 44;
                return dataSize * 1000L / (long)byteRate;
            }
        }
        catch (Exception e) {
            LOGGER.warn("Failed to extract WAV duration: {}", (Object)e.getMessage());
        }
        return 0L;
    }

    private int readLittleEndianInt(byte[] data, int offset) {
        return data[offset] & 0xFF | (data[offset + 1] & 0xFF) << 8 | (data[offset + 2] & 0xFF) << 16 | (data[offset + 3] & 0xFF) << 24;
    }

    private int estimateBitrate(byte[] audioData, String format) {
        switch (format) {
            case "mp3": 
            case "ogg": 
            case "m4a": 
            case "webm": {
                return 128;
            }
            case "wav": {
                return 1411;
            }
        }
        return 0;
    }

    private int estimateSampleRate(byte[] audioData, String format) {
        if ("wav".equals(format) && audioData.length >= 28) {
            try {
                return this.readLittleEndianInt(audioData, 24);
            }
            catch (Exception e) {
                LOGGER.warn("Failed to extract WAV sample rate: {}", (Object)e.getMessage());
            }
        }
        return 44100;
    }

    private String extractTitle(byte[] audioData, String format) {
        if ("mp3".equals(format) && this.startsWith(audioData, MP3_ID3)) {
            return "Custom Audio";
        }
        return "Custom Audio";
    }

    public byte[] processAudio(byte[] audioData) {
        LOGGER.debug("Processing audio data: {} bytes", (Object)audioData.length);
        return audioData;
    }

    public static Set<String> getSupportedFormats() {
        return SUPPORTED_FORMATS;
    }

    public static long getMaxFileSize() {
        return 0x3200000L;
    }
}

