/*
 * Decompiled with CFR 0.152.
 */
package org.stepan.audio_disc.download;

import java.net.InetAddress;
import java.net.URI;
import java.net.UnknownHostException;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.stepan.audio_disc.download.DownloadTask;
import org.stepan.audio_disc.model.DownloadStatus;

public class AudioDownloadManager {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"AudioDisc");
    private final ExecutorService downloadExecutor;
    private final Map<UUID, DownloadTask> activeDownloads;
    private final long maxFileSize;
    private final int timeoutSeconds;
    private final int maxConcurrentDownloads;
    private final Semaphore downloadSemaphore;

    public AudioDownloadManager(long maxFileSize, int timeoutSeconds, int maxConcurrentDownloads) {
        this.downloadExecutor = Executors.newFixedThreadPool(maxConcurrentDownloads);
        this.activeDownloads = new ConcurrentHashMap<UUID, DownloadTask>();
        this.maxFileSize = maxFileSize;
        this.timeoutSeconds = timeoutSeconds;
        this.maxConcurrentDownloads = maxConcurrentDownloads;
        this.downloadSemaphore = new Semaphore(maxConcurrentDownloads);
    }

    public CompletableFuture<byte[]> downloadAudio(String url, UUID playerId, Consumer<Integer> progressCallback) {
        try {
            this.validateUrl(url);
        }
        catch (IllegalArgumentException e) {
            return CompletableFuture.failedFuture(e);
        }
        if (!this.downloadSemaphore.tryAcquire()) {
            return CompletableFuture.failedFuture(new IllegalStateException("Server download limit reached. Please try again later."));
        }
        UUID taskId = UUID.randomUUID();
        DownloadTask task = new DownloadTask(taskId, url, playerId, progressCallback, this.maxFileSize, this.timeoutSeconds);
        this.activeDownloads.put(taskId, task);
        LOGGER.info("Starting download for player {} from URL: {}", (Object)playerId, (Object)url);
        return CompletableFuture.supplyAsync(() -> {
            try {
                byte[] data = task.execute();
                LOGGER.info("Download completed for player {}: {} bytes", (Object)playerId, (Object)data.length);
                byte[] byArray = data;
                return byArray;
            }
            catch (Exception e) {
                LOGGER.error("Download failed for player {}: {}", (Object)playerId, (Object)e.getMessage());
                throw new CompletionException(e);
            }
            finally {
                this.activeDownloads.remove(taskId);
                this.downloadSemaphore.release();
            }
        }, this.downloadExecutor);
    }

    public void reportProgress(UUID downloadId, int percentage) {
        DownloadTask task = this.activeDownloads.get(downloadId);
        if (task != null) {
            LOGGER.debug("Download {} progress: {}%", (Object)downloadId, (Object)percentage);
        }
    }

    public void cancelDownload(UUID downloadId) {
        DownloadTask task = this.activeDownloads.get(downloadId);
        if (task != null) {
            task.cancel();
            this.activeDownloads.remove(downloadId);
            LOGGER.info("Download {} cancelled", (Object)downloadId);
        }
    }

    public DownloadStatus getDownloadStatus(UUID downloadId) {
        DownloadTask task = this.activeDownloads.get(downloadId);
        return task != null ? task.getStatus() : null;
    }

    private void validateUrl(String url) throws IllegalArgumentException {
        if (url == null || url.trim().isEmpty()) {
            throw new IllegalArgumentException("URL cannot be empty");
        }
        try {
            URI uri = new URI(url);
            String scheme = uri.getScheme();
            if (scheme == null || !scheme.equalsIgnoreCase("http") && !scheme.equalsIgnoreCase("https")) {
                throw new IllegalArgumentException("Only HTTP and HTTPS protocols are allowed");
            }
            String host = uri.getHost();
            if (host == null || host.isEmpty()) {
                throw new IllegalArgumentException("Invalid URL: no host specified");
            }
            try {
                InetAddress address = InetAddress.getByName(host);
                if (this.isBlockedAddress(address)) {
                    throw new IllegalArgumentException("Access to local/private IP addresses is not allowed");
                }
            }
            catch (UnknownHostException e) {
                throw new IllegalArgumentException("Cannot resolve host: " + host);
            }
        }
        catch (Exception e) {
            if (e instanceof IllegalArgumentException) {
                throw (IllegalArgumentException)e;
            }
            throw new IllegalArgumentException("Invalid URL format: " + e.getMessage());
        }
    }

    private boolean isBlockedAddress(InetAddress address) {
        byte[] bytes = address.getAddress();
        if (bytes[0] == 127) {
            return true;
        }
        if (bytes[0] == 10) {
            return true;
        }
        if (bytes[0] == -84 && (bytes[1] & 0xF0) == 16) {
            return true;
        }
        if (bytes[0] == -64 && bytes[1] == -88) {
            return true;
        }
        return bytes[0] == -87 && bytes[1] == -2;
    }

    public int getActiveDownloadCount() {
        return this.activeDownloads.size();
    }

    public void shutdown() {
        LOGGER.info("Shutting down AudioDownloadManager");
        this.activeDownloads.values().forEach(DownloadTask::cancel);
        this.activeDownloads.clear();
        this.downloadExecutor.shutdown();
        try {
            if (!this.downloadExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.downloadExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.downloadExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }
}

