/*
 * Decompiled with CFR 0.152.
 */
package org.stepan.audio_disc.download;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.concurrent.CompletableFuture;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import net.fabricmc.loader.api.FabricLoader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FFmpegManager {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"FFmpegManager");
    private static final String WINDOWS_URL = "https://www.gyan.dev/ffmpeg/builds/ffmpeg-release-essentials.zip";
    private static final String LINUX_URL = "https://johnvansickle.com/ffmpeg/releases/ffmpeg-release-amd64-static.tar.xz";
    private static final Path FFMPEG_DIR = FabricLoader.getInstance().getGameDir().resolve("audiodisc").resolve("ffmpeg");
    private static Path ffmpegExecutable = null;

    public static boolean isAvailable() {
        if (FFmpegManager.isSystemFFmpegAvailable()) {
            return true;
        }
        return FFmpegManager.isModFFmpegAvailable();
    }

    public static String getExecutablePath() {
        if (FFmpegManager.isSystemFFmpegAvailable()) {
            return "ffmpeg";
        }
        if (ffmpegExecutable != null && Files.exists(ffmpegExecutable, new LinkOption[0])) {
            return ffmpegExecutable.toString();
        }
        return null;
    }

    public static CompletableFuture<Boolean> downloadAndInstall() {
        return CompletableFuture.supplyAsync(() -> {
            try {
                String downloadUrl;
                LOGGER.info("Starting automatic FFmpeg download...");
                if (!Files.exists(FFMPEG_DIR, new LinkOption[0])) {
                    Files.createDirectories(FFMPEG_DIR, new FileAttribute[0]);
                }
                if ((downloadUrl = FFmpegManager.getDownloadUrlForPlatform()) == null) {
                    LOGGER.error("Unsupported platform for automatic FFmpeg download");
                    return false;
                }
                if (!FFmpegManager.isWindows()) {
                    LOGGER.info("Automatic FFmpeg installation only supported on Windows");
                    LOGGER.info("Please install FFmpeg manually on Linux/macOS");
                    return false;
                }
                LOGGER.info("Downloading FFmpeg from: {}", (Object)downloadUrl);
                Path tempZip = FFMPEG_DIR.resolve("ffmpeg-temp.zip");
                if (!FFmpegManager.downloadFile(downloadUrl, tempZip)) {
                    return false;
                }
                LOGGER.info("Extracting FFmpeg...");
                if (!FFmpegManager.extractFFmpeg(tempZip)) {
                    return false;
                }
                try {
                    Files.deleteIfExists(tempZip);
                }
                catch (Exception e) {
                    LOGGER.warn("Failed to delete temp file: {}", (Object)e.getMessage());
                }
                Path executable = FFmpegManager.findFFmpegExecutable();
                if (executable == null) {
                    LOGGER.error("FFmpeg executable not found after extraction");
                    return false;
                }
                ffmpegExecutable = executable;
                LOGGER.info("Successfully downloaded and installed FFmpeg to: {}", (Object)executable);
                if (FFmpegManager.testFFmpeg(executable.toString())) {
                    LOGGER.info("FFmpeg installation verified successfully");
                    return true;
                }
                LOGGER.error("FFmpeg installation verification failed");
                return false;
            }
            catch (Exception e) {
                LOGGER.error("Failed to download and install FFmpeg", (Throwable)e);
                return false;
            }
        });
    }

    private static boolean isSystemFFmpegAvailable() {
        return FFmpegManager.testFFmpeg("ffmpeg");
    }

    private static boolean isModFFmpegAvailable() {
        Path executable;
        if (ffmpegExecutable == null && (executable = FFmpegManager.findFFmpegExecutable()) != null && Files.exists(executable, new LinkOption[0])) {
            ffmpegExecutable = executable;
            return FFmpegManager.testFFmpeg(executable.toString());
        }
        return ffmpegExecutable != null && Files.exists(ffmpegExecutable, new LinkOption[0]) && FFmpegManager.testFFmpeg(ffmpegExecutable.toString());
    }

    private static boolean testFFmpeg(String executable) {
        try {
            ProcessBuilder pb = new ProcessBuilder(executable, "-version");
            Process process = pb.start();
            int exitCode = process.waitFor();
            return exitCode == 0;
        }
        catch (Exception e) {
            return false;
        }
    }

    private static boolean downloadFile(String urlString, Path destination) {
        try {
            URL url = new URL(urlString);
            HttpURLConnection connection = (HttpURLConnection)url.openConnection();
            connection.setRequestMethod("GET");
            connection.setConnectTimeout(30000);
            connection.setReadTimeout(300000);
            connection.setRequestProperty("User-Agent", "AudioDisc-Mod/1.0");
            int responseCode = connection.getResponseCode();
            if (responseCode != 200) {
                LOGGER.error("Failed to download FFmpeg: HTTP {}", (Object)responseCode);
                return false;
            }
            long fileSize = connection.getContentLengthLong();
            LOGGER.info("Downloading FFmpeg ({} MB)...", (Object)(fileSize > 0L ? String.format("%.1f", (double)fileSize / 1024.0 / 1024.0) : "unknown size"));
            try (InputStream in = connection.getInputStream();){
                Files.copy(in, destination, StandardCopyOption.REPLACE_EXISTING);
            }
            LOGGER.info("Download completed: {}", (Object)destination);
            return true;
        }
        catch (Exception e) {
            LOGGER.error("Failed to download FFmpeg", (Throwable)e);
            return false;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static boolean extractFFmpeg(Path zipFile) {
        try {
            try (ZipInputStream zis = new ZipInputStream(Files.newInputStream(zipFile, new OpenOption[0]));){
                ZipEntry entry;
                while ((entry = zis.getNextEntry()) != null) {
                    String entryName = entry.getName();
                    if (entryName.endsWith("ffmpeg.exe") && !entry.isDirectory()) {
                        Path outputPath = FFMPEG_DIR.resolve("ffmpeg.exe");
                        LOGGER.info("Extracting FFmpeg executable: {} -> {}", (Object)entryName, (Object)outputPath);
                        try (OutputStream out = Files.newOutputStream(outputPath, new OpenOption[0]);){
                            int length;
                            byte[] buffer = new byte[8192];
                            while ((length = zis.read(buffer)) != -1) {
                                out.write(buffer, 0, length);
                            }
                        }
                        LOGGER.info("FFmpeg extracted successfully");
                        boolean bl = true;
                        return bl;
                    }
                    zis.closeEntry();
                }
            }
            LOGGER.error("ffmpeg.exe not found in downloaded archive");
            return false;
        }
        catch (Exception e) {
            LOGGER.error("Failed to extract FFmpeg", (Throwable)e);
            return false;
        }
    }

    private static Path findFFmpegExecutable() {
        try {
            if (FFmpegManager.isWindows()) {
                Path executable = FFMPEG_DIR.resolve("ffmpeg.exe");
                if (Files.exists(executable, new LinkOption[0])) {
                    return executable;
                }
            } else {
                Path executable = FFMPEG_DIR.resolve("ffmpeg");
                if (Files.exists(executable, new LinkOption[0])) {
                    return executable;
                }
            }
        }
        catch (Exception e) {
            LOGGER.warn("Error finding FFmpeg executable: {}", (Object)e.getMessage());
        }
        return null;
    }

    private static String getDownloadUrlForPlatform() {
        String os = System.getProperty("os.name").toLowerCase();
        if (os.contains("win")) {
            return WINDOWS_URL;
        }
        if (os.contains("nix") || os.contains("nux") || os.contains("aix")) {
            return LINUX_URL;
        }
        return null;
    }

    private static boolean isWindows() {
        return System.getProperty("os.name").toLowerCase().contains("win");
    }

    public static long getInstallationSize() {
        if (ffmpegExecutable != null && Files.exists(ffmpegExecutable, new LinkOption[0])) {
            try {
                return Files.size(ffmpegExecutable);
            }
            catch (IOException e) {
                return 0L;
            }
        }
        return 0L;
    }

    public static boolean uninstall() {
        try {
            if (ffmpegExecutable != null && Files.exists(ffmpegExecutable, new LinkOption[0])) {
                Files.delete(ffmpegExecutable);
                ffmpegExecutable = null;
                LOGGER.info("Uninstalled mod-installed FFmpeg");
                return true;
            }
            return false;
        }
        catch (Exception e) {
            LOGGER.error("Failed to uninstall FFmpeg", (Throwable)e);
            return false;
        }
    }
}

