/*
 * Decompiled with CFR 0.152.
 */
package org.stepan.audio_disc.storage;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import net.minecraft.class_1799;
import org.stepan.audio_disc.Audio_disc;
import org.stepan.audio_disc.model.AudioData;
import org.stepan.audio_disc.model.AudioMetadata;
import org.stepan.audio_disc.util.NbtUtils;

public class AudioStorageManager {
    private static final String AUDIO_DIR = "audio";
    private static final String METADATA_FILE = "metadata.json";
    private static final int CACHE_SIZE = 50;
    private final Path storageDirectory;
    private final Path audioDirectory;
    private final Path metadataFile;
    private final Gson gson;
    private final Map<String, AudioData> audioCache;
    private final LinkedHashMap<String, Long> accessOrder;
    private final Map<String, MetadataEntry> metadataIndex;

    public AudioStorageManager(Path storageDirectory) {
        this.storageDirectory = storageDirectory;
        this.audioDirectory = storageDirectory.resolve(AUDIO_DIR);
        this.metadataFile = storageDirectory.resolve(METADATA_FILE);
        this.gson = new GsonBuilder().setPrettyPrinting().create();
        this.audioCache = new ConcurrentHashMap<String, AudioData>();
        this.accessOrder = new LinkedHashMap<String, Long>(50, 0.75f, true){

            @Override
            protected boolean removeEldestEntry(Map.Entry<String, Long> eldest) {
                if (this.size() > 50) {
                    AudioStorageManager.this.audioCache.remove(eldest.getKey());
                    return true;
                }
                return false;
            }
        };
        this.metadataIndex = new ConcurrentHashMap<String, MetadataEntry>();
        this.initializeStorage();
        this.loadMetadataIndex();
    }

    private void initializeStorage() {
        try {
            Files.createDirectories(this.audioDirectory, new FileAttribute[0]);
            Audio_disc.LOGGER.info("Audio storage directory initialized at: {}", (Object)this.audioDirectory);
        }
        catch (IOException e) {
            Audio_disc.LOGGER.error("Failed to create audio storage directory", (Throwable)e);
        }
    }

    private void loadMetadataIndex() {
        if (!Files.exists(this.metadataFile, new LinkOption[0])) {
            Audio_disc.LOGGER.info("No existing metadata file found, starting fresh");
            return;
        }
        try {
            String json = Files.readString(this.metadataFile);
            JsonObject root = JsonParser.parseString((String)json).getAsJsonObject();
            root.entrySet().forEach(entry -> {
                String audioId = (String)entry.getKey();
                JsonObject entryObj = ((JsonElement)entry.getValue()).getAsJsonObject();
                MetadataEntry metadata = new MetadataEntry(entryObj.get("uploadedBy").getAsString(), entryObj.get("uploadTime").getAsLong(), entryObj.get("format").getAsString(), entryObj.get("duration").getAsLong(), entryObj.get("bitrate").getAsInt(), entryObj.get("sampleRate").getAsInt(), entryObj.get("title").getAsString());
                this.metadataIndex.put(audioId, metadata);
            });
            Audio_disc.LOGGER.info("Loaded {} audio entries from metadata index", (Object)this.metadataIndex.size());
        }
        catch (Exception e) {
            Audio_disc.LOGGER.error("Failed to load metadata index", (Throwable)e);
        }
    }

    private void saveMetadataIndex() {
        try {
            JsonObject root = new JsonObject();
            this.metadataIndex.forEach((audioId, entry) -> {
                JsonObject entryObj = new JsonObject();
                entryObj.addProperty("uploadedBy", entry.uploadedBy);
                entryObj.addProperty("uploadTime", (Number)entry.uploadTime);
                entryObj.addProperty("format", entry.format);
                entryObj.addProperty("duration", (Number)entry.duration);
                entryObj.addProperty("bitrate", (Number)entry.bitrate);
                entryObj.addProperty("sampleRate", (Number)entry.sampleRate);
                entryObj.addProperty("title", entry.title);
                root.add(audioId, (JsonElement)entryObj);
            });
            Files.writeString(this.metadataFile, (CharSequence)this.gson.toJson((JsonElement)root), new OpenOption[0]);
        }
        catch (IOException e) {
            Audio_disc.LOGGER.error("Failed to save metadata index", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String storeAudio(byte[] audioData, AudioMetadata metadata, String uploadedBy) throws IOException {
        String audioId = UUID.randomUUID().toString();
        long uploadTime = System.currentTimeMillis();
        String extension = metadata.format().toLowerCase();
        Path audioFile = this.audioDirectory.resolve(audioId + "." + extension);
        Files.write(audioFile, audioData, new OpenOption[0]);
        AudioData data = new AudioData(audioId, audioData, metadata, uploadedBy, uploadTime);
        LinkedHashMap<String, Long> linkedHashMap = this.accessOrder;
        synchronized (linkedHashMap) {
            this.audioCache.put(audioId, data);
            this.accessOrder.put(audioId, System.currentTimeMillis());
        }
        this.metadataIndex.put(audioId, new MetadataEntry(uploadedBy, uploadTime, metadata.format(), metadata.duration(), metadata.bitrate(), metadata.sampleRate(), metadata.title()));
        this.saveMetadataIndex();
        Audio_disc.LOGGER.info("Stored audio file: {} ({})", (Object)audioId, (Object)metadata.title());
        return audioId;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Optional<AudioData> getAudio(String audioId) {
        if (audioId == null || audioId.isBlank()) {
            return Optional.empty();
        }
        AudioData cached = this.audioCache.get(audioId);
        if (cached != null) {
            LinkedHashMap<String, Long> linkedHashMap = this.accessOrder;
            synchronized (linkedHashMap) {
                this.accessOrder.put(audioId, System.currentTimeMillis());
            }
            return Optional.of(cached);
        }
        MetadataEntry metadataEntry = this.metadataIndex.get(audioId);
        if (metadataEntry == null) {
            return Optional.empty();
        }
        String extension = metadataEntry.format.toLowerCase();
        Path audioFile = this.audioDirectory.resolve(audioId + "." + extension);
        if (!Files.exists(audioFile, new LinkOption[0])) {
            Audio_disc.LOGGER.warn("Audio file not found: {}", (Object)audioId);
            return Optional.empty();
        }
        try {
            byte[] audioData = Files.readAllBytes(audioFile);
            AudioMetadata metadata = new AudioMetadata(metadataEntry.format, metadataEntry.duration, metadataEntry.bitrate, metadataEntry.sampleRate, metadataEntry.title);
            AudioData data = new AudioData(audioId, audioData, metadata, metadataEntry.uploadedBy, metadataEntry.uploadTime);
            LinkedHashMap<String, Long> linkedHashMap = this.accessOrder;
            synchronized (linkedHashMap) {
                this.audioCache.put(audioId, data);
                this.accessOrder.put(audioId, System.currentTimeMillis());
            }
            return Optional.of(data);
        }
        catch (IOException e) {
            Audio_disc.LOGGER.error("Failed to load audio file: {}", (Object)audioId, (Object)e);
            return Optional.empty();
        }
    }

    public void attachToDisc(class_1799 disc, String audioId) {
        MetadataEntry metadataEntry = this.metadataIndex.get(audioId);
        if (metadataEntry == null) {
            throw new IllegalArgumentException("Audio ID not found: " + audioId);
        }
        AudioMetadata metadata = new AudioMetadata(metadataEntry.format, metadataEntry.duration, metadataEntry.bitrate, metadataEntry.sampleRate, metadataEntry.title);
        NbtUtils.attachAudioToItem(disc, audioId, metadataEntry.uploadedBy, metadataEntry.uploadTime, metadata);
    }

    public void attachToDisc(class_1799 disc, String audioId, String uploadedBy) {
        this.attachToDisc(disc, audioId);
    }

    public void clearDiscAudio(class_1799 disc) {
        NbtUtils.clearAudioData(disc);
    }

    public Optional<String> getDiscAudioId(class_1799 disc) {
        return NbtUtils.getAudioId(disc);
    }

    public void setCustomTitle(class_1799 disc, String customTitle) {
        NbtUtils.setCustomTitle(disc, customTitle);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int cleanup(Set<String> referencedAudioIds) {
        int removedCount = 0;
        try {
            HashSet<String> allAudioIds = new HashSet<String>(this.metadataIndex.keySet());
            HashSet<String> orphanedIds = new HashSet<String>(allAudioIds);
            orphanedIds.removeAll(referencedAudioIds);
            for (String audioId : orphanedIds) {
                String extension;
                Path audioFile;
                MetadataEntry entry = this.metadataIndex.get(audioId);
                if (entry != null && Files.exists(audioFile = this.audioDirectory.resolve(audioId + "." + (extension = entry.format.toLowerCase())), new LinkOption[0])) {
                    Files.delete(audioFile);
                    ++removedCount;
                }
                this.metadataIndex.remove(audioId);
                this.audioCache.remove(audioId);
                LinkedHashMap<String, Long> linkedHashMap = this.accessOrder;
                synchronized (linkedHashMap) {
                    this.accessOrder.remove(audioId);
                }
            }
            if (removedCount > 0) {
                this.saveMetadataIndex();
                Audio_disc.LOGGER.info("Cleaned up {} unused audio files", (Object)removedCount);
            }
        }
        catch (IOException e) {
            Audio_disc.LOGGER.error("Error during cleanup", (Throwable)e);
        }
        return removedCount;
    }

    public int getAudioCount() {
        return this.metadataIndex.size();
    }

    public Set<String> getAllAudioIds() {
        return new HashSet<String>(this.metadataIndex.keySet());
    }

    private static class MetadataEntry {
        final String uploadedBy;
        final long uploadTime;
        final String format;
        final long duration;
        final int bitrate;
        final int sampleRate;
        final String title;

        MetadataEntry(String uploadedBy, long uploadTime, String format, long duration, int bitrate, int sampleRate, String title) {
            this.uploadedBy = uploadedBy;
            this.uploadTime = uploadTime;
            this.format = format;
            this.duration = duration;
            this.bitrate = bitrate;
            this.sampleRate = sampleRate;
            this.title = title;
        }
    }
}

