/*
 * Decompiled with CFR 0.152.
 */
package org.stepan.audio_disc.util;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.stepan.audio_disc.config.AudioDiscConfig;

public class RateLimiter {
    private final Map<UUID, PlayerRateInfo> playerRates = new ConcurrentHashMap<UUID, PlayerRateInfo>();
    private final int maxUploadsPerMinute;
    private final int maxUploadsPerHour;
    private final long cooldownMillis;
    private final boolean enabled;

    public RateLimiter(AudioDiscConfig config) {
        this.maxUploadsPerMinute = config.getMaxUploadsPerMinute();
        this.maxUploadsPerHour = config.getMaxUploadsPerHour();
        this.cooldownMillis = (long)config.getRateLimitCooldown() * 1000L;
        this.enabled = config.isEnableRateLimit();
    }

    public RateLimitResult checkLimit(UUID playerId) {
        if (!this.enabled) {
            return RateLimitResult.allowed();
        }
        long now = System.currentTimeMillis();
        PlayerRateInfo info = this.playerRates.computeIfAbsent(playerId, k -> new PlayerRateInfo());
        if (now - info.lastUploadTime < this.cooldownMillis) {
            long remainingCooldown = (this.cooldownMillis - (now - info.lastUploadTime)) / 1000L;
            return RateLimitResult.denied("\u041f\u043e\u0434\u043e\u0436\u0434\u0438\u0442\u0435 " + remainingCooldown + " \u0441\u0435\u043a\u0443\u043d\u0434 \u043f\u0435\u0440\u0435\u0434 \u0441\u043b\u0435\u0434\u0443\u044e\u0449\u0435\u0439 \u0437\u0430\u0433\u0440\u0443\u0437\u043a\u043e\u0439");
        }
        info.uploadTimes.removeIf(time -> now - time > 60000L);
        info.hourlyUploads.removeIf(time -> now - time > 3600000L);
        if (info.uploadTimes.size() >= this.maxUploadsPerMinute) {
            return RateLimitResult.denied("\u0414\u043e\u0441\u0442\u0438\u0433\u043d\u0443\u0442 \u043b\u0438\u043c\u0438\u0442 \u0437\u0430\u0433\u0440\u0443\u0437\u043e\u043a. \u041c\u0430\u043a\u0441\u0438\u043c\u0443\u043c " + this.maxUploadsPerMinute + " \u0437\u0430\u0433\u0440\u0443\u0437\u043e\u043a \u0432 \u043c\u0438\u043d\u0443\u0442\u0443");
        }
        if (info.hourlyUploads.size() >= this.maxUploadsPerHour) {
            return RateLimitResult.denied("\u0414\u043e\u0441\u0442\u0438\u0433\u043d\u0443\u0442 \u0447\u0430\u0441\u043e\u0432\u043e\u0439 \u043b\u0438\u043c\u0438\u0442 \u0437\u0430\u0433\u0440\u0443\u0437\u043e\u043a. \u041c\u0430\u043a\u0441\u0438\u043c\u0443\u043c " + this.maxUploadsPerHour + " \u0437\u0430\u0433\u0440\u0443\u0437\u043e\u043a \u0432 \u0447\u0430\u0441");
        }
        return RateLimitResult.allowed();
    }

    public void recordUpload(UUID playerId) {
        if (!this.enabled) {
            return;
        }
        long now = System.currentTimeMillis();
        PlayerRateInfo info = this.playerRates.computeIfAbsent(playerId, k -> new PlayerRateInfo());
        info.uploadTimes.add(now);
        info.hourlyUploads.add(now);
        info.lastUploadTime = now;
    }

    public long getRemainingCooldown(UUID playerId) {
        PlayerRateInfo info = this.playerRates.get(playerId);
        if (info == null) {
            return 0L;
        }
        long now = System.currentTimeMillis();
        long elapsed = now - info.lastUploadTime;
        if (elapsed >= this.cooldownMillis) {
            return 0L;
        }
        return (this.cooldownMillis - elapsed) / 1000L;
    }

    public void clearPlayer(UUID playerId) {
        this.playerRates.remove(playerId);
    }

    public void clearAll() {
        this.playerRates.clear();
    }

    public static class RateLimitResult {
        private final boolean allowed;
        private final String message;

        private RateLimitResult(boolean allowed, String message) {
            this.allowed = allowed;
            this.message = message;
        }

        public static RateLimitResult allowed() {
            return new RateLimitResult(true, null);
        }

        public static RateLimitResult denied(String message) {
            return new RateLimitResult(false, message);
        }

        public boolean isAllowed() {
            return this.allowed;
        }

        public String getMessage() {
            return this.message;
        }
    }

    private static class PlayerRateInfo {
        final List<Long> uploadTimes = new ArrayList<Long>();
        final List<Long> hourlyUploads = new ArrayList<Long>();
        long lastUploadTime = 0L;

        private PlayerRateInfo() {
        }
    }
}

