#version 150

uniform float AmbientLightFactor;
uniform float SkyFactor;
uniform float BlockFactor;
uniform int UseBrightLightmap;
uniform vec3 SkyLightColor;
uniform float NightVisionFactor;
uniform float DarknessScale;
uniform float DarkenWorldFactor;
uniform float BrightnessFactor;

in vec2 texCoord;

out vec4 fragColor;

float get_brightness(float level) {
    float curved_level = level / (3.0 - 2.0 * level);
    return mix(curved_level, 1.0, AmbientLightFactor);
}

vec3 notGamma(vec3 x) {
    vec3 nx = 1.0 - x;
    return 1.0 - nx * nx * nx * nx;
}

float apply_shadow_curve(float brightness, float factor) {
    return pow(brightness, factor);
}

void main() {
    float block_brightness = get_brightness(floor(texCoord.x * 16) / 15) * BlockFactor;
    float sky_brightness = get_brightness(floor(texCoord.y * 16) / 15) * SkyFactor;

    vec3 base = vec3(
    block_brightness,
    block_brightness * ((block_brightness * 0.6 + 0.4) * 0.6 + 0.4),
    block_brightness * (block_brightness * block_brightness * 0.6 + 0.4)
    );

    float luminance = dot(base, vec3(0.299, 0.587, 0.114));
    vec3 color = mix(vec3(luminance), base, 2.7);

    if (UseBrightLightmap != 0) {
        color = mix(color, vec3(0.99, 1.12, 1.0), 0.25);
        color = clamp(color, 0.0, 1.0);
    } else {
        color += SkyLightColor * sky_brightness;

        vec3 darkened = color * vec3(0.7, 0.6, 0.6);
        color = mix(color, darkened, DarkenWorldFactor);
    }

    if (NightVisionFactor > 0.0) {
        float max_comp = max(color.r, max(color.g, color.b));
        if (max_comp < 1.0) {
            vec3 bright = color / max_comp;
            color = mix(color, bright, NightVisionFactor);
        }
    }

    if (UseBrightLightmap == 0) {
        color = clamp(color - vec3(DarknessScale), 0.0, 1.0);
    }

    float shadowFactor = 2;
    color = vec3(
        apply_shadow_curve(color.r, shadowFactor),
        apply_shadow_curve(color.g, shadowFactor),
        apply_shadow_curve(color.b, shadowFactor)
    );

    float shadowStrength = 1.0 - dot(color, vec3(0.333));
    vec3 blueShadow = vec3(0.05, 0.05, 0.125);
    color = mix(color, blueShadow, clamp(shadowStrength, 0.0, 1.0));

    vec3 notGamma = notGamma(color);
    color = mix(color, notGamma, BrightnessFactor);
    color = clamp(color, 0.0, 1.0);

    fragColor = vec4(color, 1.0);
}
