/*
 * Decompiled with CFR 0.152.
 */
package gg.modl.minecraft.core.sync;

import gg.modl.minecraft.api.AbstractPlayer;
import gg.modl.minecraft.api.SimplePunishment;
import gg.modl.minecraft.api.http.ModlHttpClient;
import gg.modl.minecraft.api.http.PanelUnavailableException;
import gg.modl.minecraft.api.http.request.NotificationAcknowledgeRequest;
import gg.modl.minecraft.api.http.request.PunishmentAcknowledgeRequest;
import gg.modl.minecraft.api.http.request.SyncRequest;
import gg.modl.minecraft.api.http.response.SyncResponse;
import gg.modl.minecraft.core.Platform;
import gg.modl.minecraft.core.impl.cache.Cache;
import gg.modl.minecraft.core.locale.LocaleManager;
import gg.modl.minecraft.core.util.PunishmentMessages;
import java.net.HttpURLConnection;
import java.net.URL;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import lombok.Generated;
import org.jetbrains.annotations.NotNull;

public class SyncService {
    @NotNull
    private final Platform platform;
    @NotNull
    private final ModlHttpClient httpClient;
    @NotNull
    private final Cache cache;
    @NotNull
    private final Logger logger;
    @NotNull
    private final LocaleManager localeManager;
    @NotNull
    private final String apiUrl;
    @NotNull
    private final String apiKey;
    private final int pollingRateSeconds;
    private String lastSyncTimestamp;
    private ScheduledExecutorService syncExecutor;
    private boolean isRunning = false;

    public void start() {
        if (this.isRunning) {
            this.logger.warning("Sync service is already running");
            return;
        }
        int actualPollingRate = Math.max(1, this.pollingRateSeconds);
        if (actualPollingRate != this.pollingRateSeconds) {
            this.logger.warning("Polling rate adjusted from " + this.pollingRateSeconds + " to " + actualPollingRate + " seconds (minimum is 1 second)");
        }
        this.lastSyncTimestamp = Instant.now().toString();
        this.syncExecutor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "modl-sync");
            t.setDaemon(true);
            return t;
        });
        this.logger.info("MODL Sync service starting - performing initial diagnostic check...");
        this.performDiagnosticCheck();
        this.syncExecutor.scheduleAtFixedRate(this::performSync, 10L, actualPollingRate, TimeUnit.SECONDS);
        this.isRunning = true;
        this.logger.info("MODL Sync service started - syncing every " + actualPollingRate + " seconds");
    }

    public void stop() {
        if (!this.isRunning) {
            return;
        }
        if (this.syncExecutor != null) {
            this.syncExecutor.shutdown();
            try {
                if (!this.syncExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.syncExecutor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.syncExecutor.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
        this.isRunning = false;
        this.logger.info("MODL Sync service stopped");
    }

    private void performDiagnosticCheck() {
        try {
            this.logger.info("Diagnostic: Testing API connectivity...");
            this.logger.info("Diagnostic: API Base URL: " + this.apiUrl);
            this.logger.info("Diagnostic: API Key: " + (String)(this.apiKey.length() > 8 ? this.apiKey.substring(0, 8) + "..." : "***"));
            this.testBasicConnectivity();
            SyncRequest testRequest = new SyncRequest();
            testRequest.setLastSyncTimestamp(Instant.now().toString());
            testRequest.setOnlinePlayers(List.of());
            testRequest.setServerStatus(new SyncRequest.ServerStatus(0, this.platform.getMaxPlayers(), this.platform.getServerVersion(), Instant.now().toString()));
            CompletableFuture<SyncResponse> testFuture = this.httpClient.sync(testRequest);
            ((CompletableFuture)testFuture.thenAccept(response -> {
                this.logger.info("Diagnostic: API connectivity test PASSED");
                this.logger.info("Diagnostic: Server response timestamp: " + response.getTimestamp());
            })).exceptionally(throwable -> {
                if (throwable.getCause() instanceof PanelUnavailableException) {
                    this.logger.warning("Diagnostic: Panel is temporarily unavailable (502 error) - likely restarting");
                } else {
                    this.logger.severe("Diagnostic: API connectivity test FAILED: " + throwable.getMessage());
                    if (throwable.getMessage().contains("502")) {
                        this.logger.severe("Diagnostic: 502 Bad Gateway indicates the API server is unreachable");
                        this.logger.severe("Diagnostic: This usually means:");
                        this.logger.severe("Diagnostic:   1. The panel server is down");
                        this.logger.severe("Diagnostic:   2. The reverse proxy/load balancer is misconfigured");
                        this.logger.severe("Diagnostic:   3. The API endpoint URL is incorrect");
                        this.logger.severe("Diagnostic: Expected sync endpoint: " + this.apiUrl + "/sync");
                    } else if (throwable.getMessage().contains("401") || throwable.getMessage().contains("403")) {
                        this.logger.severe("Diagnostic: Authentication failed - API key may be invalid");
                    } else if (throwable.getMessage().contains("404")) {
                        this.logger.severe("Diagnostic: Endpoint not found - the sync endpoint may not exist");
                    } else if (throwable.getMessage().contains("ConnectException") || throwable.getMessage().contains("UnknownHostException")) {
                        this.logger.severe("Diagnostic: Network connectivity issue - cannot reach " + this.apiUrl);
                    }
                }
                return null;
            });
        }
        catch (Exception e) {
            this.logger.severe("Diagnostic: Failed to perform connectivity test: " + e.getMessage());
        }
    }

    private void testBasicConnectivity() {
        block8: {
            try {
                this.logger.info("Diagnostic: Testing basic connectivity to " + this.apiUrl);
                URL url = new URL(this.apiUrl);
                HttpURLConnection connection = (HttpURLConnection)url.openConnection();
                connection.setRequestMethod("HEAD");
                connection.setConnectTimeout(5000);
                connection.setReadTimeout(5000);
                int responseCode = connection.getResponseCode();
                this.logger.info("Diagnostic: Basic connectivity test - Response code: " + responseCode);
                if (responseCode >= 200 && responseCode < 400) {
                    this.logger.info("Diagnostic: Panel server is reachable");
                } else if (responseCode == 502) {
                    this.logger.warning("Diagnostic: Panel server returned 502 - server may be misconfigured");
                } else {
                    this.logger.warning("Diagnostic: Panel server returned unexpected code: " + responseCode);
                }
                connection.disconnect();
            }
            catch (Exception e) {
                this.logger.severe("Diagnostic: Failed to connect to panel server: " + e.getMessage());
                if (e.getMessage().contains("UnknownHostException")) {
                    this.logger.severe("Diagnostic: Domain name '123.cobl.gg' cannot be resolved - check DNS");
                }
                if (e.getMessage().contains("ConnectException")) {
                    this.logger.severe("Diagnostic: Connection refused - server may be down");
                }
                if (!e.getMessage().contains("SocketTimeoutException")) break block8;
                this.logger.severe("Diagnostic: Connection timeout - server may be slow or unreachable");
            }
        }
    }

    private void performSync() {
        try {
            Collection<AbstractPlayer> onlinePlayers = this.platform.getOnlinePlayers();
            SyncRequest request = new SyncRequest();
            request.setLastSyncTimestamp(this.lastSyncTimestamp);
            request.setOnlinePlayers(this.buildOnlinePlayersList(onlinePlayers));
            request.setServerStatus(this.buildServerStatus(onlinePlayers));
            CompletableFuture<SyncResponse> syncFuture = this.httpClient.sync(request);
            ((CompletableFuture)syncFuture.thenAccept(response -> {
                try {
                    this.handleSyncResponse((SyncResponse)response);
                }
                catch (Exception e) {
                    this.logger.severe("Error handling sync response: " + e.getMessage());
                    e.printStackTrace();
                }
            })).exceptionally(throwable -> {
                if (throwable.getCause() instanceof PanelUnavailableException) {
                    this.logger.warning("Sync request failed: Panel temporarily unavailable (502 error)");
                } else {
                    this.logger.warning("Sync request failed: " + throwable.getMessage());
                }
                return null;
            });
        }
        catch (Exception e) {
            this.logger.severe("Error during sync: " + e.getMessage());
            e.printStackTrace();
        }
    }

    private List<SyncRequest.OnlinePlayer> buildOnlinePlayersList(Collection<AbstractPlayer> onlinePlayers) {
        return onlinePlayers.stream().map(player -> new SyncRequest.OnlinePlayer(player.getUuid().toString(), player.getName(), player.getIpAddress())).collect(Collectors.toList());
    }

    private SyncRequest.ServerStatus buildServerStatus(Collection<AbstractPlayer> onlinePlayers) {
        return new SyncRequest.ServerStatus(onlinePlayers.size(), this.platform.getMaxPlayers(), this.platform.getServerVersion(), Instant.now().toString());
    }

    private void handleSyncResponse(SyncResponse response) {
        this.lastSyncTimestamp = response.getTimestamp();
        SyncResponse.SyncData data = response.getData();
        for (SyncResponse.PendingPunishment pending : data.getPendingPunishments()) {
            this.processPendingPunishment(pending);
        }
        for (SyncResponse.ModifiedPunishment modified : data.getRecentlyModifiedPunishments()) {
            this.processModifiedPunishment(modified);
        }
        if (data.getActiveStaffMembers() != null) {
            for (SyncResponse.ActiveStaffMember staffMember : data.getActiveStaffMembers()) {
                this.processActiveStaffMember(staffMember);
            }
        }
        for (SyncResponse.PlayerNotification notification : data.getPlayerNotifications()) {
            this.processPlayerNotification(notification);
        }
    }

    private void processPendingPunishment(SyncResponse.PendingPunishment pending) {
        String playerUuid = pending.getMinecraftUuid();
        String username = pending.getUsername();
        SimplePunishment punishment = pending.getPunishment();
        this.logger.info(String.format("Processing pending punishment %s for %s - Type: '%s', Ordinal: %d, isBan: %s, isMute: %s, isKick: %s", punishment.getId(), username, punishment.getType(), punishment.getOrdinal(), punishment.isBan(), punishment.isMute(), punishment.isKick()));
        this.platform.runOnMainThread(() -> {
            boolean success = this.executePunishment(playerUuid, username, punishment);
            this.acknowledgePunishment(punishment.getId(), playerUuid, success, null);
        });
    }

    private void processStartedPunishment(SyncResponse.PendingPunishment started) {
        this.processPendingPunishment(started);
    }

    private void processModifiedPunishment(SyncResponse.ModifiedPunishment modified) {
        String playerUuid = modified.getMinecraftUuid();
        String username = modified.getUsername();
        SyncResponse.PunishmentWithModifications punishment = modified.getPunishment();
        this.platform.runOnMainThread(() -> {
            for (SyncResponse.PunishmentModification modification : punishment.getModifications()) {
                this.handlePunishmentModification(playerUuid, username, punishment.getId(), modification);
            }
        });
    }

    private boolean executePunishment(String playerUuid, String username, SimplePunishment punishment) {
        try {
            UUID uuid = UUID.fromString(playerUuid);
            AbstractPlayer player = this.platform.getPlayer(uuid);
            if (punishment.isBan()) {
                this.logger.info(String.format("Executing BAN for %s (type: %s, ordinal: %d)", username, punishment.getType(), punishment.getOrdinal()));
                return this.executeBan(uuid, username, punishment);
            }
            if (punishment.isMute()) {
                this.logger.info(String.format("Executing MUTE for %s (type: %s, ordinal: %d)", username, punishment.getType(), punishment.getOrdinal()));
                return this.executeMute(uuid, username, punishment);
            }
            if (punishment.isKick()) {
                this.logger.info(String.format("Executing KICK for %s (type: %s, ordinal: %d)", username, punishment.getType(), punishment.getOrdinal()));
                return this.executeKick(uuid, username, punishment);
            }
            this.logger.warning(String.format("Unknown punishment type for %s - Type: '%s', Ordinal: %d, isBan: %s, isMute: %s, isKick: %s", username, punishment.getType(), punishment.getOrdinal(), punishment.isBan(), punishment.isMute(), punishment.isKick()));
            return false;
        }
        catch (Exception e) {
            this.logger.severe("Error executing punishment: " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    private boolean executeMute(UUID uuid, String username, SimplePunishment punishment) {
        try {
            this.cache.cacheMute(uuid, punishment);
            String broadcastMessage = PunishmentMessages.formatPunishmentBroadcast(username, punishment, "muted", this.localeManager);
            this.platform.broadcast(broadcastMessage);
            this.logger.info(String.format("Successfully executed mute for %s: %s", username, punishment.getDescription()));
            return true;
        }
        catch (Exception e) {
            this.logger.severe("Error executing mute: " + e.getMessage());
            return false;
        }
    }

    private boolean executeBan(UUID uuid, String username, SimplePunishment punishment) {
        try {
            AbstractPlayer player = this.platform.getPlayer(uuid);
            if (player != null && player.isOnline()) {
                String kickMsg = PunishmentMessages.formatBanMessage(punishment, this.localeManager, PunishmentMessages.MessageContext.SYNC);
                this.platform.kickPlayer(player, kickMsg);
            }
            String broadcastMessage = PunishmentMessages.formatPunishmentBroadcast(username, punishment, "banned", this.localeManager);
            this.platform.broadcast(broadcastMessage);
            this.logger.info(String.format("Successfully executed ban for %s: %s", username, punishment.getDescription()));
            return true;
        }
        catch (Exception e) {
            this.logger.severe("Error executing ban: " + e.getMessage());
            return false;
        }
    }

    private boolean executeKick(UUID uuid, String username, SimplePunishment punishment) {
        try {
            AbstractPlayer player = this.platform.getPlayer(uuid);
            if (player != null && player.isOnline()) {
                String kickMsg = PunishmentMessages.formatKickMessage(punishment, this.localeManager, PunishmentMessages.MessageContext.SYNC);
                this.platform.kickPlayer(player, kickMsg);
                String broadcastMessage = PunishmentMessages.formatPunishmentBroadcast(username, punishment, "kicked", this.localeManager);
                this.platform.broadcast(broadcastMessage);
                this.logger.info(String.format("Successfully executed kick for %s: %s", username, punishment.getDescription()));
                return true;
            }
            this.logger.info(String.format("Player %s is not online, kick punishment ignored", username));
            return true;
        }
        catch (Exception e) {
            this.logger.severe("Error executing kick: " + e.getMessage());
            return false;
        }
    }

    private void handlePunishmentModification(String playerUuid, String username, String punishmentId, SyncResponse.PunishmentModification modification) {
        try {
            UUID uuid = UUID.fromString(playerUuid);
            switch (modification.getType()) {
                case "MANUAL_PARDON": 
                case "APPEAL_ACCEPT": {
                    this.handlePardon(uuid, username, punishmentId);
                    break;
                }
                case "MANUAL_DURATION_CHANGE": 
                case "APPEAL_DURATION_CHANGE": {
                    this.handleDurationChange(uuid, username, punishmentId, modification.getEffectiveDuration());
                    break;
                }
                default: {
                    this.logger.info("Unknown modification type: " + modification.getType());
                    break;
                }
            }
        }
        catch (Exception e) {
            this.logger.severe("Error handling punishment modification: " + e.getMessage());
            e.printStackTrace();
        }
    }

    private void handlePardon(UUID uuid, String username, String punishmentId) {
        this.cache.removePlayer(uuid);
        this.logger.info(String.format("Pardoned punishment %s for %s", punishmentId, username));
        this.platform.broadcast(String.format("\u00a7a%s has been pardoned", username));
    }

    private void handleDurationChange(UUID uuid, String username, String punishmentId, Long newDuration) {
        this.logger.info(String.format("Updated punishment %s duration for %s to %d ms", punishmentId, username, newDuration));
    }

    private void acknowledgePunishment(String punishmentId, String playerUuid, boolean success, String errorMessage) {
        PunishmentAcknowledgeRequest request = new PunishmentAcknowledgeRequest(punishmentId, playerUuid, Instant.now().toString(), success, errorMessage);
        ((CompletableFuture)this.httpClient.acknowledgePunishment(request).thenAccept(response -> this.logger.info(String.format("Acknowledged punishment %s execution: %s", punishmentId, success ? "SUCCESS" : "FAILED")))).exceptionally(throwable -> {
            if (throwable.getCause() instanceof PanelUnavailableException) {
                this.logger.warning("Failed to acknowledge punishment " + punishmentId + ": Panel temporarily unavailable");
            } else {
                this.logger.warning("Failed to acknowledge punishment " + punishmentId + ": " + throwable.getMessage());
            }
            return null;
        });
    }

    private void processActiveStaffMember(SyncResponse.ActiveStaffMember staffMember) {
        try {
            UUID uuid = UUID.fromString(staffMember.getMinecraftUuid());
            AbstractPlayer player = this.platform.getPlayer(uuid);
            if (player != null && player.isOnline()) {
                this.cache.cacheStaffMember(uuid, staffMember);
                this.logger.info(String.format("Updated staff member data for %s (%s) - Role: %s, Permissions: %d", staffMember.getMinecraftUsername(), staffMember.getStaffUsername(), staffMember.getStaffRole(), staffMember.getPermissions().size()));
            }
        }
        catch (Exception e) {
            this.logger.warning("Error processing staff member data: " + e.getMessage());
        }
    }

    private void processPlayerNotification(SyncResponse.PlayerNotification notification) {
        try {
            this.logger.info(String.format("Processing notification %s (type: %s): %s", notification.getId(), notification.getType(), notification.getMessage()));
            String targetPlayerUuid = notification.getTargetPlayerUuid();
            if (targetPlayerUuid != null && !targetPlayerUuid.isEmpty()) {
                try {
                    UUID playerUuid = UUID.fromString(targetPlayerUuid);
                    this.deliverNotificationToPlayer(playerUuid, notification);
                }
                catch (IllegalArgumentException e) {
                    this.logger.warning("Invalid UUID format in notification: " + targetPlayerUuid);
                }
            } else {
                this.handleNotificationForAllOnlinePlayers(notification);
            }
        }
        catch (Exception e) {
            this.logger.severe("Error processing player notification: " + e.getMessage());
            e.printStackTrace();
        }
    }

    private void deliverNotificationToPlayer(UUID playerUuid, SyncResponse.PlayerNotification notification) {
        AbstractPlayer player = this.platform.getPlayer(playerUuid);
        if (player != null && player.isOnline()) {
            Map<String, Object> data = notification.getData();
            if (data != null && data.containsKey("ticketUrl")) {
                String ticketUrl = (String)data.get("ticketUrl");
                String ticketId = (String)data.get("ticketId");
                String message = String.format("{\"text\":\"\",\"extra\":[{\"text\":\"[Ticket] \",\"color\":\"gold\"},{\"text\":\"%s \",\"color\":\"white\"},{\"text\":\"[Click to view]\",\"color\":\"aqua\",\"underlined\":true,\"clickEvent\":{\"action\":\"open_url\",\"value\":\"%s\"},\"hoverEvent\":{\"action\":\"show_text\",\"value\":\"Click to view ticket %s\"}}]}", notification.getMessage().replace("\"", "\\\""), ticketUrl, ticketId);
                this.logger.info("Sending clickable notification JSON: " + message);
                this.platform.runOnMainThread(() -> this.platform.sendJsonMessage(playerUuid, message));
            } else {
                String message = this.localeManager.getMessage("notification.ticket_reply", Map.of("message", notification.getMessage()));
                this.platform.runOnMainThread(() -> this.platform.sendMessage(playerUuid, message));
            }
            this.logger.info(String.format("Delivered notification %s to online player %s", notification.getId(), player.getName()));
        } else {
            this.cache.cacheNotification(playerUuid, notification);
            this.logger.info(String.format("Cached notification %s for offline player %s", notification.getId(), playerUuid));
        }
    }

    public void deliverLoginNotification(UUID playerUuid, SyncResponse.PlayerNotification notification) {
        this.logger.info(String.format("Delivering login notification %s to player %s", notification.getId(), playerUuid));
        this.deliverNotificationToPlayer(playerUuid, notification);
    }

    private void handleNotificationForAllOnlinePlayers(SyncResponse.PlayerNotification notification) {
        Collection<AbstractPlayer> onlinePlayers = this.platform.getOnlinePlayers();
        for (AbstractPlayer player : onlinePlayers) {
            try {
                UUID playerUuid = player.getUuid();
                this.cache.cacheNotification(playerUuid, notification);
                this.deliverNotificationToPlayer(playerUuid, notification);
            }
            catch (Exception e) {
                this.logger.warning("Error handling notification for player " + player.getName() + ": " + e.getMessage());
            }
        }
    }

    public void deliverPendingNotifications(UUID playerUuid) {
        try {
            List<Cache.PendingNotification> pendingNotifications = this.cache.getPendingNotifications(playerUuid);
            if (pendingNotifications.isEmpty()) {
                return;
            }
            ArrayList<Cache.PendingNotification> notificationsToProcess = new ArrayList<Cache.PendingNotification>(pendingNotifications);
            this.logger.info(String.format("Delivering %d pending notifications to player %s", notificationsToProcess.size(), playerUuid));
            ArrayList<String> deliveredNotificationIds = new ArrayList<String>();
            ArrayList<String> expiredNotificationIds = new ArrayList<String>();
            this.deliverNotificationsWithDelay(playerUuid, notificationsToProcess, deliveredNotificationIds, expiredNotificationIds);
            for (String expiredId : expiredNotificationIds) {
                this.cache.removeNotification(playerUuid, expiredId);
            }
            for (String deliveredId : deliveredNotificationIds) {
                this.cache.removeNotification(playerUuid, deliveredId);
            }
            if (!deliveredNotificationIds.isEmpty()) {
                this.acknowledgeNotifications(playerUuid, deliveredNotificationIds);
            }
        }
        catch (Exception e) {
            this.logger.severe("Error delivering pending notifications: " + e.getMessage());
            e.printStackTrace();
        }
    }

    private String formatNotificationMessage(SyncResponse.PlayerNotification notification) {
        return this.localeManager.getMessage("notification.modl_prefix", Map.of("message", notification.getMessage()));
    }

    private String formatNotificationMessage(Cache.PendingNotification notification) {
        return this.localeManager.getMessage("notification.ticket_prefix", Map.of("message", notification.getMessage()));
    }

    private void deliverPendingNotificationToPlayer(UUID playerUuid, Cache.PendingNotification pending) {
        AbstractPlayer player = this.platform.getPlayer(playerUuid);
        if (player != null && player.isOnline()) {
            Map<String, Object> data = pending.getData();
            if (data != null && data.containsKey("ticketUrl")) {
                String ticketUrl = (String)data.get("ticketUrl");
                String ticketId = (String)data.get("ticketId");
                String message = String.format("{\"text\":\"\",\"extra\":[{\"text\":\"[Ticket] \",\"color\":\"gold\"},{\"text\":\"%s \",\"color\":\"white\"},{\"text\":\"[Click to view]\",\"color\":\"aqua\",\"underlined\":true,\"clickEvent\":{\"action\":\"open_url\",\"value\":\"%s\"},\"hoverEvent\":{\"action\":\"show_text\",\"value\":\"Click to view ticket %s\"}}]}", pending.getMessage().replace("\"", "\\\""), ticketUrl, ticketId);
                this.platform.runOnMainThread(() -> this.platform.sendJsonMessage(playerUuid, message));
            } else {
                String message = this.formatNotificationMessage(pending);
                this.platform.runOnMainThread(() -> this.platform.sendMessage(playerUuid, message));
            }
        }
    }

    private void deliverNotificationsWithDelay(UUID playerUuid, List<Cache.PendingNotification> notifications, List<String> deliveredIds, List<String> expiredIds) {
        if (notifications.isEmpty()) {
            return;
        }
        this.syncExecutor.schedule(() -> this.platform.runOnMainThread(() -> this.deliverNotificationAtIndex(playerUuid, notifications, 0, deliveredIds, expiredIds)), 2000L, TimeUnit.MILLISECONDS);
    }

    private void deliverNotificationAtIndex(UUID playerUuid, List<Cache.PendingNotification> notifications, int index, List<String> deliveredIds, List<String> expiredIds) {
        block5: {
            if (index >= notifications.size()) {
                this.finalizePendingNotificationDelivery(playerUuid, deliveredIds, expiredIds);
                return;
            }
            Cache.PendingNotification pending = notifications.get(index);
            try {
                if (pending.isExpired()) {
                    expiredIds.add(pending.getId());
                    break block5;
                }
                AbstractPlayer player = this.platform.getPlayer(playerUuid);
                if (player != null && player.isOnline()) {
                    this.playNotificationSound(playerUuid);
                    this.deliverPendingNotificationToPlayer(playerUuid, pending);
                    deliveredIds.add(pending.getId());
                    this.logger.info(String.format("Delivered pending notification %s to player %s", pending.getId(), playerUuid));
                    break block5;
                }
                this.logger.info(String.format("Player %s disconnected during notification delivery", playerUuid));
                return;
            }
            catch (Exception e) {
                this.logger.warning("Error delivering pending notification " + pending.getId() + ": " + e.getMessage());
            }
        }
        this.syncExecutor.schedule(() -> this.platform.runOnMainThread(() -> this.deliverNotificationAtIndex(playerUuid, notifications, index + 1, deliveredIds, expiredIds)), 1500L, TimeUnit.MILLISECONDS);
    }

    private void finalizePendingNotificationDelivery(UUID playerUuid, List<String> deliveredIds, List<String> expiredIds) {
        try {
            for (String expiredId : expiredIds) {
                this.cache.removeNotification(playerUuid, expiredId);
            }
            for (String deliveredId : deliveredIds) {
                this.cache.removeNotification(playerUuid, deliveredId);
            }
            if (!deliveredIds.isEmpty()) {
                this.acknowledgeNotifications(playerUuid, deliveredIds);
            }
            this.logger.info(String.format("Completed pending notification delivery for player %s. Delivered: %d, Expired: %d", playerUuid, deliveredIds.size(), expiredIds.size()));
        }
        catch (Exception e) {
            this.logger.severe("Error finalizing pending notification delivery: " + e.getMessage());
            e.printStackTrace();
        }
    }

    private void playNotificationSound(UUID playerUuid) {
        this.logger.fine("Playing notification sound for player " + String.valueOf(playerUuid) + " (placeholder)");
    }

    private void acknowledgeNotification(UUID playerUuid, String notificationId) {
        this.acknowledgeNotifications(playerUuid, List.of(notificationId));
    }

    private void acknowledgeNotifications(UUID playerUuid, List<String> notificationIds) {
        try {
            NotificationAcknowledgeRequest request = new NotificationAcknowledgeRequest(playerUuid.toString(), notificationIds, Instant.now().toString());
            ((CompletableFuture)this.httpClient.acknowledgeNotifications(request).thenAccept(response -> this.logger.info(String.format("Acknowledged %d notifications for player %s", notificationIds.size(), playerUuid)))).exceptionally(throwable -> {
                if (throwable.getCause() instanceof PanelUnavailableException) {
                    this.logger.warning("Failed to acknowledge notifications for player " + String.valueOf(playerUuid) + ": Panel temporarily unavailable");
                } else {
                    this.logger.warning("Failed to acknowledge notifications for player " + String.valueOf(playerUuid) + ": " + throwable.getMessage());
                }
                return null;
            });
        }
        catch (Exception e) {
            this.logger.severe("Error acknowledging notifications: " + e.getMessage());
            e.printStackTrace();
        }
    }

    @Generated
    public SyncService(@NotNull Platform platform, @NotNull ModlHttpClient httpClient, @NotNull Cache cache, @NotNull Logger logger, @NotNull LocaleManager localeManager, @NotNull String apiUrl, @NotNull String apiKey, int pollingRateSeconds) {
        if (platform == null) {
            throw new NullPointerException("platform is marked non-null but is null");
        }
        if (httpClient == null) {
            throw new NullPointerException("httpClient is marked non-null but is null");
        }
        if (cache == null) {
            throw new NullPointerException("cache is marked non-null but is null");
        }
        if (logger == null) {
            throw new NullPointerException("logger is marked non-null but is null");
        }
        if (localeManager == null) {
            throw new NullPointerException("localeManager is marked non-null but is null");
        }
        if (apiUrl == null) {
            throw new NullPointerException("apiUrl is marked non-null but is null");
        }
        if (apiKey == null) {
            throw new NullPointerException("apiKey is marked non-null but is null");
        }
        this.platform = platform;
        this.httpClient = httpClient;
        this.cache = cache;
        this.logger = logger;
        this.localeManager = localeManager;
        this.apiUrl = apiUrl;
        this.apiKey = apiKey;
        this.pollingRateSeconds = pollingRateSeconds;
    }
}

