/*
 * Decompiled with CFR 0.152.
 */
package fr.raconteur.chatlogs.database;

import fr.raconteur.chatlogs.ChatLogsMod;
import fr.raconteur.chatlogs.session.SimpleSessionRecorder;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Scanner;

public abstract class AbstractDatabase {
    private final String dbFileName;
    private final Path dbPath;
    private final Path backupDir;
    private final String versioningResourcePath;
    private Connection connection;
    private RandomAccessFile lockRaf;
    private FileLock processLock;

    public AbstractDatabase(String dbFileName) throws SQLException {
        this.dbFileName = dbFileName;
        this.dbPath = SimpleSessionRecorder.CHATLOG_FOLDER.toPath().resolve(dbFileName);
        this.backupDir = SimpleSessionRecorder.CHATLOG_FOLDER.toPath().resolve("backups").resolve(dbFileName.replace(".db", ""));
        this.versioningResourcePath = "database/" + dbFileName.replace(".db", "") + "/";
        this.initializeDatabase();
    }

    private void initializeDatabase() throws SQLException {
        try {
            Files.createDirectories(this.dbPath.getParent(), new FileAttribute[0]);
            Files.createDirectories(this.backupDir, new FileAttribute[0]);
            this.cleanupOrphanedLocks();
            this.acquireProcessLock();
            boolean isNewDatabase = !Files.exists(this.dbPath, new LinkOption[0]);
            String url = "jdbc:sqlite:" + this.dbPath.toString();
            this.connection = DriverManager.getConnection(url);
            this.connection.setAutoCommit(false);
            ChatLogsMod.LOGGER.info("Connected to database: {}", (Object)this.dbPath);
            this.initializeVersionTable();
            this.applyMigrations(isNewDatabase);
        }
        catch (IOException e) {
            this.releaseProcessLock();
            throw new SQLException("Failed to initialize database directories", e);
        }
    }

    private void initializeVersionTable() throws SQLException {
        String createVersionTable = "CREATE TABLE IF NOT EXISTS db_version (\n    version INTEGER PRIMARY KEY,\n    applied_at TEXT NOT NULL,\n    description TEXT\n)\n";
        try (Statement stmt = this.connection.createStatement();){
            stmt.execute(createVersionTable);
        }
    }

    private void applyMigrations(boolean isNewDatabase) throws SQLException {
        int currentVersion = this.getCurrentVersion();
        List<Integer> availableVersions = this.getAvailableVersions();
        if (availableVersions.isEmpty()) {
            ChatLogsMod.LOGGER.warn("No migration files found for database: {}", (Object)this.dbFileName);
            return;
        }
        Collections.sort(availableVersions);
        for (Integer version : availableVersions) {
            if (version <= currentVersion) continue;
            this.applyMigration(version, isNewDatabase && version == availableVersions.get(0));
        }
    }

    private int getCurrentVersion() throws SQLException {
        String query = "SELECT MAX(version) FROM db_version";
        try (Statement stmt = this.connection.createStatement();){
            int n;
            block12: {
                ResultSet rs = stmt.executeQuery(query);
                try {
                    int n2 = n = rs.next() ? rs.getInt(1) : 0;
                    if (rs == null) break block12;
                }
                catch (Throwable throwable) {
                    if (rs != null) {
                        try {
                            rs.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                rs.close();
            }
            return n;
        }
    }

    private List<Integer> getAvailableVersions() {
        ArrayList<Integer> versions = new ArrayList<Integer>();
        int version = 1;
        while (true) {
            String resourcePath = this.versioningResourcePath + "v" + version + ".sql";
            ChatLogsMod.LOGGER.info("Looking for migration file: {}", (Object)resourcePath);
            try (InputStream is = this.getClass().getClassLoader().getResourceAsStream(resourcePath);){
                if (is == null) {
                    ChatLogsMod.LOGGER.info("Migration file not found: {}", (Object)resourcePath);
                    break;
                }
                ChatLogsMod.LOGGER.info("Found migration file: {}", (Object)resourcePath);
                versions.add(version);
                ++version;
            }
            catch (Exception e) {
                throw new RuntimeException("Critical error checking for version file " + resourcePath + ": " + e.getMessage(), e);
            }
        }
        ChatLogsMod.LOGGER.info("Found {} migration files for database: {}", (Object)versions.size(), (Object)this.dbFileName);
        return versions;
    }

    private void applyMigration(int version, boolean skipBackup) throws SQLException {
        String resourcePath = this.versioningResourcePath + "v" + version + ".sql";
        try (InputStream is = this.getClass().getClassLoader().getResourceAsStream(resourcePath);){
            String migrationSql;
            if (is == null) {
                throw new SQLException("Migration file not found: " + resourcePath);
            }
            if (!skipBackup) {
                this.createBackup(version);
            }
            try (Scanner scanner = new Scanner(is, "UTF-8");){
                scanner.useDelimiter("\\A");
                migrationSql = scanner.hasNext() ? scanner.next() : "";
            }
            this.executeInTransaction(() -> {
                try (Statement stmt = this.connection.createStatement();){
                    String[] statements;
                    for (String sql : statements = migrationSql.split(";")) {
                        String trimmedSql = sql.trim();
                        if (trimmedSql.isEmpty()) continue;
                        stmt.execute(trimmedSql);
                    }
                    this.recordMigration(version, "Applied migration v" + version);
                }
                catch (SQLException e) {
                    throw new RuntimeException("Migration failed for version " + version, e);
                }
            });
            ChatLogsMod.LOGGER.info("Applied migration v{} to database: {}", (Object)version, (Object)this.dbFileName);
        }
        catch (IOException e) {
            throw new SQLException("Failed to read migration file: " + resourcePath, e);
        }
    }

    private void createBackup(int version) throws SQLException {
        try {
            String timestamp = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss"));
            String backupFileName = String.format("%s_v%d_%s.db", this.dbFileName.replace(".db", ""), version - 1, timestamp);
            Path backupPath = this.backupDir.resolve(backupFileName);
            Files.copy(this.dbPath, backupPath, StandardCopyOption.REPLACE_EXISTING);
            ChatLogsMod.LOGGER.info("Created backup before migration v{}: {}", (Object)version, (Object)backupPath);
        }
        catch (IOException e) {
            throw new SQLException("Failed to create backup before migration", e);
        }
    }

    private void recordMigration(int version, String description) throws SQLException {
        String insert = "INSERT INTO db_version (version, applied_at, description) VALUES (?, ?, ?)";
        try (PreparedStatement stmt = this.connection.prepareStatement(insert);){
            stmt.setInt(1, version);
            stmt.setString(2, LocalDateTime.now().toString());
            stmt.setString(3, description);
            stmt.executeUpdate();
        }
    }

    public void executeInTransaction(Runnable transaction) throws SQLException {
        try {
            transaction.run();
            this.connection.commit();
        }
        catch (Exception e) {
            this.connection.rollback();
            if (e instanceof SQLException) {
                throw (SQLException)e;
            }
            throw new SQLException("Transaction failed", e);
        }
    }

    public <T> T executeInTransaction(TransactionCallable<T> transaction) throws SQLException {
        try {
            T result = transaction.call(this.connection);
            this.connection.commit();
            return result;
        }
        catch (Exception e) {
            this.connection.rollback();
            if (e instanceof SQLException) {
                throw (SQLException)e;
            }
            throw new SQLException("Transaction failed", e);
        }
    }

    protected Connection getConnection() {
        return this.connection;
    }

    public void close() throws SQLException {
        if (this.connection != null && !this.connection.isClosed()) {
            this.connection.close();
            ChatLogsMod.LOGGER.info("Closed database connection: {}", (Object)this.dbFileName);
        }
        this.releaseProcessLock();
    }

    private void cleanupOrphanedLocks() {
        Path lockPath = Paths.get(this.dbPath.toString() + ".lock", new String[0]);
        if (Files.exists(lockPath, new LinkOption[0])) {
            try (RandomAccessFile testRaf = new RandomAccessFile(lockPath.toFile(), "rw");
                 FileChannel testChannel = testRaf.getChannel();){
                FileLock testLock = testChannel.tryLock();
                if (testLock != null) {
                    testLock.release();
                    Files.delete(lockPath);
                    ChatLogsMod.LOGGER.info("Cleaned up orphaned database lock file: {}", (Object)lockPath);
                }
            }
            catch (IOException e) {
                ChatLogsMod.LOGGER.warn("Failed to clean up potential orphaned lock file: {}", (Object)lockPath, (Object)e);
            }
        }
    }

    private void acquireProcessLock() throws SQLException {
        try {
            Path lockPath = Paths.get(this.dbPath.toString() + ".lock", new String[0]);
            this.lockRaf = new RandomAccessFile(lockPath.toFile(), "rw");
            FileChannel channel = this.lockRaf.getChannel();
            this.processLock = channel.tryLock();
            if (this.processLock == null) {
                this.lockRaf.close();
                throw new SQLException("Minecraft may already be running on this instance. This is not supported by chatlogs mod. (Database locked: " + this.dbFileName + ")");
            }
            this.lockRaf.seek(0L);
            this.lockRaf.writeUTF("PID:" + ProcessHandle.current().pid() + " DB:" + this.dbFileName);
            ChatLogsMod.LOGGER.info("Acquired database process lock: {}", (Object)this.dbFileName);
        }
        catch (IOException e) {
            throw new SQLException("Failed to acquire database process lock: " + this.dbFileName, e);
        }
    }

    private void releaseProcessLock() {
        try {
            if (this.processLock != null) {
                this.processLock.release();
                this.processLock = null;
                ChatLogsMod.LOGGER.debug("Released database process lock: {}", (Object)this.dbFileName);
            }
        }
        catch (IOException e) {
            ChatLogsMod.LOGGER.warn("Error releasing database process lock: {}", (Object)this.dbFileName, (Object)e);
        }
        try {
            if (this.lockRaf != null) {
                this.lockRaf.close();
                this.lockRaf = null;
            }
        }
        catch (IOException e) {
            ChatLogsMod.LOGGER.warn("Error closing database lock file: {}", (Object)this.dbFileName, (Object)e);
        }
        try {
            Path lockPath = Paths.get(this.dbPath.toString() + ".lock", new String[0]);
            if (Files.exists(lockPath, new LinkOption[0])) {
                Files.delete(lockPath);
                ChatLogsMod.LOGGER.debug("Deleted database lock file: {}", (Object)lockPath);
            }
        }
        catch (IOException e) {
            ChatLogsMod.LOGGER.warn("Failed to delete database lock file: {}", (Object)this.dbFileName, (Object)e);
        }
    }

    @FunctionalInterface
    public static interface TransactionCallable<T> {
        public T call(Connection var1) throws SQLException;
    }
}

