/*
 * Decompiled with CFR 0.152.
 */
package fr.raconteur.chatlogs.session;

import fr.raconteur.chatlogs.ChatLogsMod;
import fr.raconteur.chatlogs.database.SessionDatabase;
import fr.raconteur.chatlogs.session.SimpleSessionRecorder;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;

public class CrashRecovery {
    private static final DateTimeFormatter TIMESTAMP_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    private static final File UNSAVED_MARKER = new File(SimpleSessionRecorder.CHATLOG_FOLDER, "unsaved.marker");

    public static void performRecovery() {
        CrashRecovery.recoverSqliteSessions();
        if (!UNSAVED_MARKER.exists()) {
            ChatLogsMod.LOGGER.debug("No unsaved marker found, no recovery needed");
            return;
        }
        List<String> unsavedFiles = CrashRecovery.readUnsavedFiles();
        if (unsavedFiles.isEmpty()) {
            CrashRecovery.cleanupMarker();
            return;
        }
        ChatLogsMod.LOGGER.warn("Crash detected! Found {} unsaved chat log files from previous session:", (Object)unsavedFiles.size());
        int recovered = 0;
        int failed = 0;
        for (String filePath : unsavedFiles) {
            File file = new File(filePath);
            if (file.exists()) {
                if (CrashRecovery.finalizeUnsavedLog(file)) {
                    ++recovered;
                    ChatLogsMod.LOGGER.info("  \u2713 Recovered: {}", (Object)file.getName());
                    continue;
                }
                ++failed;
                ChatLogsMod.LOGGER.error("  \u2717 Failed to recover: {}", (Object)file.getName());
                continue;
            }
            ChatLogsMod.LOGGER.warn("  ? File no longer exists: {}", (Object)filePath);
        }
        CrashRecovery.cleanupMarker();
        ChatLogsMod.LOGGER.info("Recovery complete: {} files recovered, {} failed", (Object)recovered, (Object)failed);
    }

    private static List<String> readUnsavedFiles() {
        ArrayList<String> unsavedFiles = new ArrayList<String>();
        try (Scanner scanner = new Scanner(new FileReader(UNSAVED_MARKER));){
            while (scanner.hasNextLine()) {
                String line = scanner.nextLine().trim();
                if (line.isEmpty()) continue;
                unsavedFiles.add(line);
            }
        }
        catch (IOException e) {
            ChatLogsMod.LOGGER.error("Failed to read unsaved marker file", (Throwable)e);
        }
        return unsavedFiles;
    }

    private static boolean finalizeUnsavedLog(File logFile) {
        boolean bl;
        if (CrashRecovery.hasProperEnding(logFile)) {
            ChatLogsMod.LOGGER.debug("File {} already has proper ending", (Object)logFile.getName());
            return true;
        }
        FileWriter writer = new FileWriter(logFile, true);
        try {
            writer.write("\n");
            writer.write("=====================================\n");
            writer.write("\u26a0\ufe0f  SESSION ENDED UNEXPECTEDLY  \u26a0\ufe0f\n");
            writer.write("Crash detected and recovered automatically\n");
            writer.write("Recovery time: " + LocalDateTime.now().format(TIMESTAMP_FORMAT) + "\n");
            writer.write("=====================================\n");
            ChatLogsMod.LOGGER.debug("Successfully finalized crashed log: {}", (Object)logFile.getName());
            bl = true;
        }
        catch (Throwable throwable) {
            try {
                try {
                    writer.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                ChatLogsMod.LOGGER.error("Failed to finalize crashed log: {}", (Object)logFile.getName(), (Object)e);
                return false;
            }
        }
        writer.close();
        return bl;
    }

    private static boolean hasProperEnding(File logFile) {
        Scanner scanner = new Scanner(logFile);
        try {
            String lastLine = "";
            while (scanner.hasNextLine()) {
                lastLine = scanner.nextLine();
            }
            boolean bl = lastLine.contains("=====================================") || lastLine.contains("Session ended") || lastLine.contains("ENDED UNEXPECTEDLY");
            scanner.close();
            return bl;
        }
        catch (Throwable throwable) {
            try {
                try {
                    scanner.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                ChatLogsMod.LOGGER.error("Failed to check file ending: {}", (Object)logFile.getName(), (Object)e);
                return false;
            }
        }
    }

    private static void cleanupMarker() {
        if (UNSAVED_MARKER.exists()) {
            if (UNSAVED_MARKER.delete()) {
                ChatLogsMod.LOGGER.debug("Cleaned up unsaved marker file");
            } else {
                ChatLogsMod.LOGGER.warn("Failed to delete unsaved marker file");
            }
        }
    }

    public static void forceRecovery() {
        ChatLogsMod.LOGGER.info("Performing forced recovery of all potential unsaved logs...");
        File[] lockFiles = SimpleSessionRecorder.CHATLOG_FOLDER.listFiles((dir, name) -> name.endsWith(".lock"));
        if (lockFiles != null) {
            for (File lockFile : lockFiles) {
                String logFileName = lockFile.getName().replace(".lock", "");
                File logFile = new File(SimpleSessionRecorder.CHATLOG_FOLDER, logFileName);
                if (logFile.exists() && !CrashRecovery.hasProperEnding(logFile)) {
                    ChatLogsMod.LOGGER.info("Found orphaned log file: {}", (Object)logFileName);
                    CrashRecovery.finalizeUnsavedLog(logFile);
                }
                lockFile.delete();
            }
        }
    }

    public static RecoveryStats getRecoveryStats() {
        File[] lockFiles;
        int unsavedCount = 0;
        int orphanedLocks = 0;
        if (UNSAVED_MARKER.exists()) {
            unsavedCount = CrashRecovery.readUnsavedFiles().size();
        }
        if ((lockFiles = SimpleSessionRecorder.CHATLOG_FOLDER.listFiles((dir, name) -> name.endsWith(".lock"))) != null) {
            orphanedLocks = lockFiles.length;
        }
        return new RecoveryStats(unsavedCount, orphanedLocks);
    }

    private static void recoverSqliteSessions() {
        try {
            SessionDatabase db = SessionDatabase.getInstance();
            List<SessionDatabase.SessionData> sessions = db.getAllSessions();
            int recovered = 0;
            for (SessionDatabase.SessionData session : sessions) {
                if (session.endTime != 0L) continue;
                ChatLogsMod.LOGGER.warn("Found unclosed SQLite session: {} (ID: {})", (Object)session.sessionName, (Object)session.id);
                db.endSession(session.id);
                ++recovered;
                ChatLogsMod.LOGGER.info("  \u2713 Closed orphaned SQLite session: {} (ID: {})", (Object)session.sessionName, (Object)session.id);
            }
            if (recovered > 0) {
                ChatLogsMod.LOGGER.info("SQLite recovery completed: {} sessions closed", (Object)recovered);
            } else {
                ChatLogsMod.LOGGER.debug("No orphaned SQLite sessions found");
            }
        }
        catch (Exception e) {
            ChatLogsMod.LOGGER.error("Failed to recover SQLite sessions", (Throwable)e);
            throw new RuntimeException("Critical error during SQLite session recovery", e);
        }
    }

    public static class RecoveryStats {
        public final int unsavedFiles;
        public final int orphanedLocks;

        public RecoveryStats(int unsavedFiles, int orphanedLocks) {
            this.unsavedFiles = unsavedFiles;
            this.orphanedLocks = orphanedLocks;
        }

        public boolean hasRecoveryNeeded() {
            return this.unsavedFiles > 0 || this.orphanedLocks > 0;
        }

        public String toString() {
            return String.format("RecoveryStats{unsaved=%d, orphanedLocks=%d}", this.unsavedFiles, this.orphanedLocks);
        }
    }
}

