package snownee.jade.gui.config.value;

import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.Supplier;

import net.minecraft.ChatFormatting;
import net.minecraft.client.gui.components.EditBox;

public class InputOptionValue<T> extends OptionValue<T> {

	public static final Predicate<String> INTEGER = s -> s.matches("[-+]?[0-9]+");
	public static final Predicate<String> FLOAT = s -> s.matches("[-+]?([0-9]*[.,][0-9]+|[0-9]+)");

	private final EditBox textField;
	private final Predicate<String> validator;

	public InputOptionValue(Runnable responder, String optionName, Supplier<T> getter, Consumer<T> setter, Predicate<String> validator) {
		super(optionName, getter, setter);
		this.validator = validator;
		textField = new EditBox(client.font, 0, 0, 98, 18, getTitle());
		updateValue();
		textField.setResponder(s -> {
			if (this.validator.test(s)) {
				setValue(s);
				textField.setTextColor(Objects.requireNonNull(ChatFormatting.WHITE.getColor()) | 0xFF000000);
			} else {
				textField.setTextColor(Objects.requireNonNull(ChatFormatting.RED.getColor()) | 0xFF000000);
			}
			responder.run();
		});
		addWidget(textField, 0);
	}

	@Override
	public boolean isValidValue() {
		return validator.test(textField.getValue());
	}

	@Override
	public void setValue(T value) {
		textField.setValue(String.valueOf(value));
	}

	@SuppressWarnings("unchecked")
	private void setValue(String text) {
		if (value instanceof String) {
			value = (T) text;
		}
		try {
			switch (value) {
				case Integer ignored -> value = (T) Integer.valueOf(text);
				case Short ignored -> value = (T) Short.valueOf(text);
				case Byte ignored -> value = (T) Byte.valueOf(text);
				case Long ignored -> value = (T) Long.valueOf(text);
				case Double ignored -> value = (T) Double.valueOf(text);
				case Float ignored -> value = (T) Float.valueOf(text);
				default -> {
				}
			}
		} catch (NumberFormatException ignored) {
		}
		save();
	}

	@Override
	public void updateValue() {
		T newValue = getter.get();
		if (!Objects.equals(value, newValue)) {
			value = newValue;
			textField.setValue(String.valueOf(value));
		}
	}

}
