package snownee.jade.impl.theme;

import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import org.apache.commons.lang3.mutable.MutableObject;
import org.jspecify.annotations.Nullable;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.mojang.brigadier.Message;
import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import net.minecraft.CrashReport;
import net.minecraft.ReportedException;
import net.minecraft.advancements.criterion.MinMaxBounds;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.network.chat.Style;
import net.minecraft.resources.FileToIdConverter;
import net.minecraft.resources.Identifier;
import net.minecraft.server.packs.resources.ResourceManager;
import net.minecraft.server.packs.resources.SimpleJsonResourceReloadListener;
import net.minecraft.util.ExtraCodecs;
import net.minecraft.util.Mth;
import net.minecraft.util.profiling.ProfilerFiller;
import snownee.jade.Jade;
import snownee.jade.JadeClient;
import snownee.jade.addon.core.ModNameProvider;
import snownee.jade.api.JadeIds;
import snownee.jade.api.config.IWailaConfig;
import snownee.jade.api.theme.IThemeHelper;
import snownee.jade.api.theme.Theme;
import snownee.jade.api.ui.JadeUI;
import snownee.jade.api.ui.TextElement;
import snownee.jade.impl.config.WailaConfig;
import snownee.jade.overlay.DisplayHelper;
import snownee.jade.util.JadeClientCodecs;
import snownee.jade.util.KeyedReloadListener;

public class ThemeHelper extends SimpleJsonResourceReloadListener<JadeClientCodecs.ThemeHolder> implements IThemeHelper, KeyedReloadListener {
	public static final ThemeHelper INSTANCE = new ThemeHelper();
	public static final Identifier ID = JadeIds.JADE("themes");
	private static final Int2ObjectMap<Style> styleCache = new Int2ObjectOpenHashMap<>(6);
	private final Map<Identifier, Theme> themes = Maps.newTreeMap();
	private final MinMaxBounds.Ints allowedVersions = MinMaxBounds.Ints.between(200, 299);
	private final Style[] modNameStyleCache = new Style[]{Style.EMPTY, Style.EMPTY, Style.EMPTY};
	private @Nullable Theme theme;
	private @Nullable Theme fallback;
	private int generation;
	private @Nullable Theme themeOverride;

	public ThemeHelper() {
		super(
				RecordCodecBuilder.create(i -> i.group(
						ExtraCodecs.NON_NEGATIVE_INT.fieldOf("version").forGetter(JadeClientCodecs.ThemeHolder::version),
						Codec.BOOL.optionalFieldOf("autoEnable", false).forGetter(JadeClientCodecs.ThemeHolder::autoEnable),
						JadeClientCodecs.THEME.forGetter(JadeClientCodecs.ThemeHolder::theme)
				).apply(i, JadeClientCodecs.ThemeHolder::new)), FileToIdConverter.json("jade_themes"));
	}

	public static Style colorStyle(int color) {
		return styleCache.computeIfAbsent(color, Style.EMPTY::withColor);
	}

	@Override
	public Theme theme() {
		return themeOverride != null ? themeOverride : Objects.requireNonNull(theme);
	}

	@Override
	public Collection<Theme> getThemes() {
		return themes.values();
	}

	@Override
	public Theme getTheme(Identifier id) {
		return Preconditions.checkNotNull(themes.getOrDefault(id, Objects.requireNonNull(fallback)), "Theme not found: %s", id);
	}

	@Override
	public boolean hasTheme(Identifier id) {
		return themes.containsKey(id);
	}

	@Override
	public MutableComponent info(Object componentOrString) {
		return color(componentOrString, theme().text.colors().info());
	}

	@Override
	public MutableComponent success(Object componentOrString) {
		return color(componentOrString, theme().text.colors().success());
	}

	@Override
	public MutableComponent warning(Object componentOrString) {
		return color(componentOrString, theme().text.colors().warning());
	}

	@Override
	public MutableComponent danger(Object componentOrString) {
		return color(componentOrString, theme().text.colors().danger());
	}

	@Override
	public MutableComponent failure(Object componentOrString) {
		return color(componentOrString, theme().text.colors().failure());
	}

	@Override
	public MutableComponent title(Object componentOrString) {
		Component component;
		if (componentOrString instanceof MutableComponent) {
			component = (MutableComponent) componentOrString;
		} else {
			component = Component.literal(Objects.toString(componentOrString));
		}
		return color(DisplayHelper.INSTANCE.stripColor(component), theme().text.colors().title());
	}

	@Override
	public TextElement modName(Object componentOrString) {
		MutableComponent component;
		if (componentOrString instanceof MutableComponent) {
			component = (MutableComponent) componentOrString;
		} else {
			component = Component.literal(Objects.toString(componentOrString));
		}
		Style itemStyle = IWailaConfig.get().formatting().getItemModNameStyle();
		Style themeStyle = theme().text.modNameStyle();
		if (modNameStyleCache[0] != itemStyle || modNameStyleCache[1] != themeStyle) {
			Style style = themeStyle.applyTo(itemStyle);
			modNameStyleCache[0] = itemStyle;
			modNameStyleCache[1] = themeStyle;
			modNameStyleCache[2] = style;
		}
		return JadeUI
				.text(component.withStyle(modNameStyleCache[2]))
				.scale(Objects.equals(IWailaConfig.get().plugin().getEnum(JadeIds.CORE_MOD_NAME), ModNameProvider.Mode.SMALLER) ?
						0.75F :
						1F);
	}

	@Override
	public MutableComponent seconds(int ticks, float tickRate, boolean alwaysOnePart) {
		int seconds = Mth.floor(ticks / tickRate);
		if (seconds >= 3600) {
			int hours = seconds / 3600;
			seconds %= 3600;
			int minutes = seconds / 60;
			if (alwaysOnePart || minutes == 0) {
				return info(JadeClient.format("jade.hours", hours));
			} else {
				return info(JadeClient.format("jade.hours_minutes", hours, minutes));
			}
		}
		if (seconds >= 60) {
			int minutes = seconds / 60;
			seconds %= 60;
			if (alwaysOnePart || seconds == 0) {
				return info(JadeClient.format("jade.minutes", minutes));
			} else {
				return info(JadeClient.format("jade.minutes_seconds", minutes, seconds));
			}
		}
		return info(JadeClient.format("jade.seconds", seconds));
	}

	@Override
	public int generation() {
		return generation;
	}

	public void setTheme(Theme theme) {
		if (this.theme == theme) {
			return;
		}
		this.theme = theme;
		generation++;
	}

	@Override
	public void setThemeOverride(@Nullable Theme theme) {
		if (themeOverride == theme) {
			return;
		}
		generation++;
		themeOverride = theme;
	}

	protected MutableComponent color(Object componentOrString, int color) {
		if (componentOrString instanceof Number number) {
			componentOrString = DisplayHelper.dfCommas.format(number.doubleValue());
		}
		if (componentOrString instanceof MutableComponent component) {
			if (component.getStyle().isEmpty()) {
				return component.setStyle(colorStyle(color));
			} else {
				return component.setStyle(component.getStyle().withColor(color));
			}
		}
		if (componentOrString instanceof Component component) {
			if (component.getStyle().isEmpty()) {
				return component.copy().setStyle(colorStyle(color));
			} else {
				return component.copy().setStyle(component.getStyle().withColor(color));
			}
		}
		if (componentOrString instanceof Message message) {
			return Component.literal(message.getString()).setStyle(colorStyle(color));
		}
		return Component.literal(Objects.toString(componentOrString)).setStyle(colorStyle(color));
	}

	@Override
	protected void apply(
			Map<Identifier, JadeClientCodecs.ThemeHolder> map,
			ResourceManager resourceManager,
			ProfilerFiller profilerFiller) {
		Set<Identifier> existingKeys = Set.copyOf(themes.keySet());
		MutableObject<@Nullable Theme> enable = new MutableObject<>();
		WailaConfig.Overlay config = Jade.config().overlay();
		WailaConfig.History history = Jade.history();
		themes.clear();
		map.forEach((id, holder) -> {
			if (!allowedVersions.matches(holder.version())) {
				Jade.LOGGER.warn("Theme {} has unsupported version {}. Skipping.", id, holder.version());
				return;
			}
			Theme theme = holder.theme();
			theme.id = id;
			themes.put(id, theme);
			if (enable.get() == null && holder.autoEnable() && !existingKeys.contains(id)) {
				enable.setValue(theme);
			}
		});
		fallback = themes.get(JadeIds.DEFAULT_THEME);
		if (fallback == null) {
			CrashReport crashreport = CrashReport.forThrowable(new NullPointerException(), "Missing default theme");
			throw new ReportedException(crashreport);
		}
		int hash = 0;
		for (Identifier id : themes.keySet()) {
			hash = 31 * hash + id.hashCode();
		}
		if (hash != history.themesHash) {
			Theme theme = enable.get();
			if (hash != 0 && theme != null) {
				config.activeTheme = theme.fullId();
				Jade.LOGGER.info("Auto enabled theme {}", theme.fullId());
				if (theme.changeOpacity != 0) {
					config.setAlpha(theme.changeOpacity);
				}
			}
			history.themesHash = hash;
			IWailaConfig.get().save();
		}
		config.applyTheme(config.activeTheme);
	}

	@Override
	public Identifier getUid() {
		return ID;
	}
}
