package snownee.jade.util;

import java.io.File;
import java.nio.charset.StandardCharsets;
import java.text.DateFormat;
import java.util.Date;
import java.util.Set;
import java.util.function.Consumer;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.jspecify.annotations.Nullable;

import com.google.common.collect.Sets;

import net.minecraft.ChatFormatting;
import net.minecraft.network.chat.Component;
import net.minecraft.resources.Identifier;
import snownee.jade.Jade;
import snownee.jade.api.IJadeProvider;
import snownee.jade.api.TraceableException;
import snownee.jade.api.config.IWailaConfig;

public class WailaExceptionHandler {

	private static final Set<IJadeProvider> ERRORS = Sets.newConcurrentHashSet();
	private static boolean NULL_ERROR = false;
	private static final File ERROR_OUTPUT = new File("logs", "JadeErrorOutput.txt");

	public static void handleErr(Throwable e, @Nullable IJadeProvider provider, @Nullable Consumer<Component> tooltip) {
		if (CommonProxy.isDevEnv() && !IWailaConfig.get().general().isDebug()) {
			ExceptionUtils.wrapAndThrow(e);
			return;
		}
		if (provider == null) {
			if (!NULL_ERROR) {
				NULL_ERROR = true;
				writeLog(e, null);
			}
		} else if (!ERRORS.contains(provider)) {
			ERRORS.add(provider);
			writeLog(e, provider);
		}
		if (tooltip != null) {
			String modid = null;
			if (e instanceof TraceableException traceableException) {
				modid = traceableException.getNamespace();
			} else if (provider != null) {
				modid = provider.getUid().getNamespace();
			}
			if (modid == null || Identifier.DEFAULT_NAMESPACE.equals(modid)) {
				modid = Jade.ID;
			}
			tooltip.accept(Component.translatable("jade.error", ModIdentification.getModFullName(modid).orElse(modid))
					.withStyle(ChatFormatting.DARK_RED));
		}
	}

	private static void writeLog(Throwable e, @Nullable IJadeProvider provider) {
		Jade.LOGGER.error("Caught unhandled exception : [{}] {}", provider, e);
		Jade.LOGGER.error("See JadeErrorOutput.txt for more information");
		Jade.LOGGER.error("", e);
		try {
			FileUtils.writeStringToFile(
					ERROR_OUTPUT,
					DateFormat.getDateTimeInstance().format(new Date()) + "\n" + provider + "\n" + ExceptionUtils.getStackTrace(e) + "\n",
					StandardCharsets.UTF_8,
					true);
		} catch (Exception ignored) {
		}
	}
}
