// fsang:bifrost — Coordinate keypad + rainbow aura + group teleport (icons match coord_menu; no quick-saves)

global.dimensionOptions = [
    "minecraft:overworld",
    "minecraft:the_nether",
    "minecraft:the_end",
    "fsang:pocket_dimension"
];

// Session-only per-player state
if (!global.playerCoordState) global.playerCoordState = {};

// ---- Particle emitter (used in tick) ----
function emitBifrostParticles(player) {
    const px = player.x, py = player.y + 1, pz = player.z;

    // 7-color palette; minecraft:dust uses 0..1 RGB + scale
    const palette = [
        [1.0, 0.0, 0.0],
        [1.0, 0.5, 0.0],
        [1.0, 1.0, 0.0],
        [0.0, 1.0, 0.0],
        [0.0, 0.5, 1.0],
        [0.3, 0.0, 0.8],
        [0.7, 0.0, 0.7]
    ];

    // Sprinkle a dozen particles within ~2.5 blocks around the player
    for (let i = 0; i < 24; i++) {
        const [r, g, b] = palette[(Math.random() * palette.length) | 0];
        const ox = (Math.random() * 5) - 2.5;
        const oy = (Math.random() * 5) - 2.5;
        const oz = (Math.random() * 5) - 2.5;

        // Execute positioned at the player; use relative offsets
        player.runCommandSilent(
            `execute positioned ${px.toFixed(2)} ${py.toFixed(2)} ${pz.toFixed(2)} run ` +
            `particle minecraft:dust ${r.toFixed(3)} ${g.toFixed(3)} ${b.toFixed(3)} 1 ` +
            `~${ox.toFixed(2)} ~${oy.toFixed(2)} ~${oz.toFixed(2)} 0 0 0 0 1 normal`
        );
    }
}

StartupEvents.registry('palladium:abilities', event => {
    event.create('fsang:bifrost')
        .icon(palladium.createItemIcon('minecraft:ender_pearl'))
        .tick((entity, entry, holder, enabled) => {
            if (!enabled || !entity || !entity.isPlayer()) return;

            const player = entity;
            const uuid = player.uuid;

            if (!global.playerCoordState[uuid]) {
                global.playerCoordState[uuid] = {
                    x: 0, y: 0, z: 0,
                    dim: "minecraft:overworld",
                    dimIndex: 0,
                    editing: null,
                    input: "",
                    tickCounter: 0
                };
            }

            const s = global.playerCoordState[uuid];

            // Keep GUI refreshed (matching your working coord_menu pattern)
            global.openBifrostGUI(player);

            // Rainbow particles while the menu is open (ability enabled)
            s.tickCounter = (s.tickCounter + 1) % 20;
            if (s.tickCounter % 2 === 0) {
                emitBifrostParticles(player);
            }
        });
});

global.openBifrostGUI = (player) => {
    const uuid = player.uuid;

    if (!global.playerCoordState[uuid]) {
        global.playerCoordState[uuid] = {
            x: 0, y: 0, z: 0,
            dim: "minecraft:overworld",
            dimIndex: 0,
            editing: null,
            input: "",
            tickCounter: 0
        };
    }

    const s = global.playerCoordState[uuid];

    player.openChestGUI(Text.of("§0Bifrost — Enter Coordinates"), 6, gui => {
        gui.playerSlots = false;

        // ===== Dimension selection (TOP ROW y = 0) =====
        global.dimensionOptions.forEach((dimId, index) => {
            if (index > 8) return;
            gui.slot(index, 0, slot => {
                const isSelected = s.dim === dimId;
                slot.item = Item.of('fsang:num_dimension', 1, {
                    display: {
                        Name: JSON.stringify({ text: dimId, color: isSelected ? "gold" : "gray" }),
                        Lore: [JSON.stringify({ text: isSelected ? "Current Dimension" : "Click to select", color: "dark_gray" })]
                    }
                });
                slot.leftClicked = () => {
                    s.dim = dimId;
                    s.dimIndex = index;
                    global.openBifrostGUI(player);
                };
            });
        });

        // ===== Numeric keypad (cols 3–5; rows 1–3; 0 at row 4) =====
        const makeNumButton = (num, sx, sy) => {
            gui.slot(sx, sy, slot => {
                slot.item = Item.of(`fsang:num_${num}`, 1, {
                    display: { Name: JSON.stringify({ text: `${num}`, color: "white" }) }
                });
                slot.leftClicked = () => {
                    s.input += `${num}`;
                    global.openBifrostGUI(player);
                };
            });
        };
        [1, 2, 3].forEach((n, i) => makeNumButton(n, 3 + i, 1));
        [4, 5, 6].forEach((n, i) => makeNumButton(n, 3 + i, 2));
        [7, 8, 9].forEach((n, i) => makeNumButton(n, 3 + i, 3));
        makeNumButton(0, 4, 4);

        // Clear input (x=3, y=4)
        gui.slot(3, 4, slot => {
            slot.item = Item.of('fsang:num_clear', 1, {
                display: { Name: JSON.stringify({ text: "Clear Input", color: "red" }) }
            });
            slot.leftClicked = () => {
                s.input = "";
                global.openBifrostGUI(player);
            };
        });

        // Toggle negative (x=5, y=4)
        gui.slot(5, 4, slot => {
            slot.item = Item.of('fsang:num_negative', 1, {
                display: { Name: JSON.stringify({ text: "Toggle Negative", color: "dark_red" }) }
            });
            slot.leftClicked = () => {
                s.input = s.input.startsWith("-") ? s.input.substring(1) : "-" + s.input;
                global.openBifrostGUI(player);
            };
        });

        // Set X/Y/Z selectors (x=6, y=1..3)
        ['x', 'y', 'z'].forEach((axis, i) => {
            gui.slot(6, 1 + i, slot => {
                slot.item = Item.of(`fsang:num_${axis}`, 1, {
                    display: { Name: JSON.stringify({ text: `Set ${axis.toUpperCase()}`, color: "green" }) }
                });
                slot.leftClicked = () => {
                    s.editing = axis;
                    s.input = "";
                    global.openBifrostGUI(player);
                };
            });
        });

        // Confirm entry (x=6, y=4)
        gui.slot(6, 4, slot => {
            slot.item = Item.of('fsang:num_confirm', 1, {
                display: { Name: JSON.stringify({ text: "Confirm Entry", color: "aqua", bold: true }) }
            });
            slot.leftClicked = () => {
                if (s.editing) {
                    let v = parseInt(s.input);
                    if (!isNaN(v)) s[s.editing] = v;
                    s.input = "";
                    s.editing = null;
                    global.openBifrostGUI(player);
                } else {
                    player.tell("§cPlease select X/Y/Z first!");
                }
            };
        });

        // Status / preview (x=3, y=5)
        gui.slot(3, 5, slot => {
            slot.item = Item.of('minecraft:map', 1, {
                display: {
                    Name: JSON.stringify({ text: `Input: ${s.input || "(empty)"}`, color: "yellow" }),
                    Lore: [
                        JSON.stringify({ text: `Editing: ${s.editing || "(none)"}`, color: "gray" }),
                        JSON.stringify({ text: `X: ${s.x}`, color: "gray" }),
                        JSON.stringify({ text: `Y: ${s.y}`, color: "gray" }),
                        JSON.stringify({ text: `Z: ${s.z}`, color: "gray" }),
                        JSON.stringify({ text: `Dimension: ${s.dim}`, color: "gray" })
                    ]
                }
            });
        });

        // Teleport (x=5, y=5) — move player + all nearby entities (~5×5×5) cross-dimension in one shot
        gui.slot(5, 5, slot => {
            slot.item = Item.of('minecraft:ender_pearl', 1, {
                display: {
                    Name: JSON.stringify({ text: "Teleport", color: "light_purple", bold: true }),
                    Lore: [
                        JSON.stringify({ text: "Teleports you + all nearby entities (~5×5×5)", color: "gray" }),
                        JSON.stringify({ text: "Cross-dimension supported", color: "gray" })
                    ]
                }
            });
            slot.leftClicked = () => {
                if (Number.isInteger(s.x) && Number.isInteger(s.y) && Number.isInteger(s.z) && typeof s.dim === 'string') {
                    try {
                        // Capture origin BEFORE moving anyone
                        let ox = player.x.toFixed(2);
                        let oy = player.y.toFixed(2);
                        let oz = player.z.toFixed(2);

                        // Target (distinct names to avoid Rhino "redeclaration")
                        let tx = s.x;
                        let ty = s.y;
                        let tz = s.z;
                        let tdim = s.dim;

                        // One command: select everyone within ~2.6 blocks of origin (≈5×5×5 cube) and move them
                        player.server.runCommandSilent(
                            `execute positioned ${ox} ${oy} ${oz} run ` +
                            `execute as @e[distance=..2.6] in ${tdim} run tp @s ${tx} ${ty} ${tz}`
                        );
                    } catch (err) {
                        player.tell(`§cTeleportation failed: ${err}`);
                    }
                } else {
                    player.tell("§cInvalid coordinate input.");
                }
            };
        });
    });
};
