package com.github.razorplay01.ismah.client.config;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import net.minecraft.client.Minecraft;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.util.GsonHelper;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

public class ModConfig {
    private ModConfig() {
        // []
    }

    private static final Logger LOGGER = LogManager.getLogger();
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();

    private static final Set<String> blacklistedArmors = new HashSet<>();

    private static File getConfigFile() {
        return new File(Minecraft.getInstance().gameDirectory, "config/ismah.json");
    }

    public static void loadConfig() {
        File configFile = getConfigFile();
        if (!configFile.exists()) {
            saveConfig();
        }

        try (FileReader reader = new FileReader(configFile)) {
            JsonObject json = JsonParser.parseReader(reader).getAsJsonObject();
            blacklistedArmors.clear();
            GsonHelper.getAsJsonArray(json, "blacklisted_armors", null).forEach(element -> {
                if (element.isJsonPrimitive()) {
                    blacklistedArmors.add(element.getAsString());
                }
            });
            LOGGER.info("Loaded {} blacklisted armors from config.", blacklistedArmors.size());
        } catch (IOException | IllegalStateException e) {
            LOGGER.error("Failed to load config: {}", e.getMessage());
            saveConfig();
        }
    }

    public static void saveConfig() {
        File configFile = getConfigFile();
        JsonObject json = new JsonObject();
        json.add("blacklisted_armors", GSON.toJsonTree(blacklistedArmors.toArray(new String[0])));
        try (FileWriter writer = new FileWriter(configFile)) {
            GSON.toJson(json, writer);
        } catch (IOException e) {
            LOGGER.error("Failed to save config: {}", e.getMessage());
        }
    }

    public static boolean isArmorBlacklisted(ResourceLocation itemId) {
        return blacklistedArmors.contains(itemId.toString());
    }

    public static Set<String> getBlacklistedArmors() {
        return Collections.unmodifiableSet(blacklistedArmors);
    }

    public static void addBlacklistedArmor(String id) {
        blacklistedArmors.add(id);
    }

    public static void removeBlacklistedArmor(String id) {
        blacklistedArmors.remove(id);
    }
}