package com.github.razorplay01.ismah.client.api;

import net.minecraft.world.entity.player.Player;
import net.minecraft.world.item.ItemStack;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

/**
 * Registry for custom armor renderers in first-person view.
 * This class allows mod developers to register their implementations of {@link CustomArmorRenderer}
 * to handle custom armor rendering on the player's arm in first-person perspective.
 */
public class ArmorRendererRegistry {
    private ArmorRendererRegistry() {
        // []
    }

    private static final List<CustomArmorRenderer> renderers = new ArrayList<>();
    private static final List<Function<Player, ItemStack>> chestplateProviders = new ArrayList<>();

    /**
     * Registers a custom armor renderer.
     * Mods should call this method during initialization to add their renderer to the registry.
     * Renderers are checked in the order they are registered, and the first matching renderer
     * for an ItemStack will be used.
     *
     * @param renderer The custom armor renderer to register.
     */
    public static void register(CustomArmorRenderer renderer) {
        renderers.add(renderer);
    }

    /**
     * Retrieves the first registered renderer that can handle the given ItemStack.
     * This method is used internally by the ISMAH mod to determine which renderer to use.
     *
     * @param stack The ItemStack of the equipped armor.
     * @return The first {@link CustomArmorRenderer} that returns {@code true} for {@link CustomArmorRenderer#canRender},
     * or {@code null} if no renderer matches.
     */
    public static CustomArmorRenderer getRenderer(ItemStack stack) {
        for (CustomArmorRenderer renderer : renderers) {
            if (renderer.canRender(stack)) {
                return renderer;
            }
        }
        return null;
    }

    public static void registerChestplateProvider(Function<Player, ItemStack> provider) {
        chestplateProviders.add(provider);
    }
    //? if >= 1.19 && <= 1.21.4 {
    public static ItemStack getChestplate(Player player) {
        for (Function<Player, ItemStack> provider : chestplateProviders) {
            ItemStack stack = provider.apply(player);
            if (stack != null && !stack.isEmpty() && stack.getItem() instanceof net.minecraft.world.item.ArmorItem) {
                return stack;
            }
        }
        return player.getInventory().getArmor(2);
    }
    //?}

    //? if >= 1.21.5 {
    /*public static ItemStack getChestplate(Player player) {
        for (Function<Player, ItemStack> provider : chestplateProviders) {
            ItemStack stack = provider.apply(player);
            if (stack != null && !stack.isEmpty()) {
                net.minecraft.world.item.equipment.Equippable equippable = stack.get(net.minecraft.core.component.DataComponents.EQUIPPABLE);
                if (equippable.slot() == net.minecraft.world.entity.EquipmentSlot.CHEST) {
                    return stack;
                }
            }
        }
        return ((com.github.razorplay01.ismah.common.mixin.InventoryAccessor) player.getInventory()).getEquipment().get(net.minecraft.world.entity.EquipmentSlot.CHEST);
    }
    *///?}
}