package io.github.fishstiz.minecraftcursor;

import io.github.fishstiz.minecraftcursor.api.CursorType;
import io.github.fishstiz.minecraftcursor.config.AnimationData;
import io.github.fishstiz.minecraftcursor.config.Config;
import io.github.fishstiz.minecraftcursor.config.ConfigLoader;
import io.github.fishstiz.minecraftcursor.cursor.Cursor;
import io.github.fishstiz.minecraftcursor.cursor.CursorManager;
import io.github.fishstiz.minecraftcursor.util.SettingsUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Optional;
import net.minecraft.class_1011;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3298;
import net.minecraft.class_3300;

import static io.github.fishstiz.minecraftcursor.MinecraftCursor.CONFIG;
import static io.github.fishstiz.minecraftcursor.MinecraftCursor.LOGGER;

public class CursorResourceLoader {
    private static final String ANIMATION_TYPE = ".mcmeta";
    private static final class_2960 SETTINGS_LOCATION = MinecraftCursor.loc("atlases/cursors.json");
    private static final class_2960 DIR = MinecraftCursor.loc("textures/cursors/");
    private static volatile Config.Resource resourceConfig;

    private CursorResourceLoader() {
    }

    public static class_2960 getDirectory() {
        return DIR;
    }

    static void reloadSettings(class_3300 manager) {
        onReload();
        loadResourceSettings(manager);
        CursorManager.INSTANCE.getCursors().forEach(cursor -> cursor.setLazy(true));
        class_310.method_1551().execute(() -> {
            onReload();
            CursorManager.INSTANCE.reapplyCursor();
        });
    }

    public static void reload() {
        class_3300 manager = class_310.method_1551().method_1478();
        loadResourceSettings(manager);
        for (Cursor cursor : CursorManager.INSTANCE.getCursors()) {
            Config.Settings settings = CONFIG.getOrCreateSettings(cursor);
            loadCursorTexture(manager, cursor, settings);
        }
        class_310.method_1551().execute(CursorManager.INSTANCE::reapplyCursor);
    }

    static void onReload() {
        CursorManager.INSTANCE.setCurrentCursor(CursorType.DEFAULT);
    }

    private static void loadResourceSettings(class_3300 manager) {
        getLayeredSettings(manager.method_14489(SETTINGS_LOCATION)).ifPresent(config -> {
            if (config.isDifferent(CONFIG)) {
                LOGGER.info("[minecraft-cursor] New resource pack settings detected, updating config...");
                CONFIG.setHash(config.getHash());
                CONFIG.merge(config);
                CONFIG.getGlobal().setActiveAll(false);
                CONFIG.save();
            }
            resourceConfig = config;
        });
    }

    private static Optional<Config.Resource> getLayeredSettings(List<class_3298> configResources) {
        Config.Resource layeredResources = null;
        for (class_3298 configResource : configResources) {
            try (InputStream stream = configResource.method_14482()) {
                Config.Resource resourceConfig = ConfigLoader.loadResource(stream);
                if (layeredResources == null) {
                    layeredResources = resourceConfig;
                } else {
                    layeredResources.layer(resourceConfig.getAllSettings());
                }
            } catch (IOException e) {
                LOGGER.error("[minecraft-cursor] Failed to load settings of resource pack '{}'", configResource.method_14480());
            }
        }
        return Optional.ofNullable(layeredResources);
    }

    public static boolean isResourceSetting(@NotNull Cursor cursor, @Nullable Config.Settings settings) {
        return SettingsUtil.equalSettings(resourceConfig.getOrCreateSettings(cursor), settings, true);
    }

    public static boolean retoreActiveResourceSettings(@NotNull Cursor cursor) {
        if (cursor.isLoaded() && resourceConfig != null) {
            CONFIG.replaceActiveSettings(resourceConfig, cursor);
            cursor.apply(CONFIG.getGlobal().apply(CONFIG.getOrCreateSettings(cursor)));
            return true;
        } else {
            LOGGER.error("Failed to apply resource settings for '{}'", cursor.getTypeKey());
        }
        return false;
    }

    public static void restoreResourceSettings() {
        if (resourceConfig != null) {
            CONFIG.setHash(resourceConfig.getHash());
            CONFIG.merge(resourceConfig);
            for (Cursor cursor : CursorManager.INSTANCE.getCursors()) {
                cursor.apply(CONFIG.getGlobal().apply(CONFIG.getOrCreateSettings(cursor)));
            }
        } else {
            LOGGER.error("Failed to apply resource config: Not Found.");
        }
    }

    public static boolean loadCursorTexture(class_3300 manager, Cursor cursor) {
        return loadCursorTexture(manager, cursor, CONFIG.getOrCreateSettings(cursor));
    }

    public static boolean loadCursorTexture(Cursor cursor) {
        return loadCursorTexture(class_310.method_1551().method_1478(), cursor);
    }

    private static boolean loadCursorTexture(class_3300 manager, Cursor cursor, Config.Settings settings) {
        LOGGER.info("[minecraft-cursor] Loading {} cursor...", cursor.getTypeKey());

        class_2960 location = cursor.getLocation();
        Optional<class_3298> cursorResource = manager.method_14486(location);

        try {
            if (cursorResource.isEmpty()) {
                LOGGER.error("[minecraft-cursor] Cursor Type: '{}' not found", cursor.getTypeKey());
                cursor.destroy();
                return false;
            }

            try (InputStream cursorStream = cursorResource.get().method_14482(); class_1011 image = class_1011.method_4309(cursorStream)) {
                AnimationData animation = loadAnimation(manager, location, cursorResource.get());
                cursor = CursorManager.INSTANCE.loadCursor(cursor, image, CONFIG.getGlobal().apply(settings), animation);
                return true;
            } catch (IOException e) {
                LOGGER.error("[minecraft-cursor] Failed to load cursor at '{}': {}", location, e.getMessage());
                return false;
            }
        } finally {
            class_310.method_1551().execute(() -> class_310.method_1551().method_1531().method_4615(location));
            cursor.setLazy(false);
        }
    }

    private static AnimationData loadAnimation(class_3300 manager, class_2960 location, class_3298 cursorResource) {
        Optional<class_3298> animationResource = manager.method_14486(location.method_48331(ANIMATION_TYPE));
        if (animationResource.isPresent() && animationResource.get().method_14480().equals(cursorResource.method_14480())) {
            try (InputStream stream = animationResource.get().method_14482()) {
                return ConfigLoader.loadAnimationData(stream);
            } catch (IOException e) {
                LOGGER.error("[minecraft-cursor] Failed to load animation data for '{}'", location);
            }
        }
        return null;
    }
}
