package io.github.fishstiz.cursors_extended.util;

import io.github.fishstiz.cursors_extended.CursorsExtended;
import io.github.fishstiz.cursors_extended.config.CursorProperties;
import io.github.fishstiz.cursors_extended.mixin.util.NativeImageAccess;
import org.lwjgl.glfw.GLFW;
import org.lwjgl.glfw.GLFWImage;
import org.lwjgl.system.MemoryUtil;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.WritableByteChannel;
import net.minecraft.class_1011;

public class NativeImageUtil {
    private NativeImageUtil() {
    }

    public static class_1011 cropImage(class_1011 src, int xOffset, int yOffset, int width, int height) {
        int srcWidth = src.method_4307();
        int srcHeight = src.method_4323();

        if (xOffset < 0 || yOffset < 0 || xOffset + width > srcWidth || yOffset + height > srcHeight) {
            CursorsExtended.LOGGER.error(
                    "[cursors-extended] Image size {}x{} is invalid. Valid size: {}x{} at y {}",
                    srcWidth, srcHeight, width, height, yOffset
            );
            return src;
        }

        class_1011 out = new class_1011(width, height, true);
        for (int y = 0; y < height; y++) {
            for (int x = 0; x < width; x++) {
                int color = src.method_61940(x + xOffset, y + yOffset);
                out.method_61941(x, y, color);
            }
        }

        return out;
    }

    public static class_1011 scaleImage(class_1011 src, double scale) {
        int width = (int) Math.round(src.method_4307() * scale);
        int height = (int) Math.round(src.method_4323() * scale);
        class_1011 scaled = new class_1011(width, height, true);

        for (int y = 0; y < height; y++) { // nearest neighbor
            for (int x = 0; x < width; x++) {
                int srcX = (int) (x / scale);
                int srcY = (int) (y / scale);
                scaled.method_61941(x, y, src.method_61940(srcX, srcY));
            }
        }
        return scaled;
    }

    public static void writePixelsRGBA(class_1011 image, ByteBuffer buffer) {
        int[] pixelsABGR = image.method_48463();

        for (int abgr : pixelsABGR) {
            buffer.put((byte) (abgr & 0xFF));
            buffer.put((byte) ((abgr >> 8) & 0xFF));
            buffer.put((byte) ((abgr >> 16) & 0xFF));
            buffer.put((byte) ((abgr >> 24) & 0xFF));
        }

        buffer.flip();
    }

    public static byte[] getBytes(class_1011 image) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        WritableByteChannel channel = Channels.newChannel(baos);

        boolean success = ((NativeImageAccess) (Object) image).cursors_extended$writeToChannel(channel);
        channel.close();

        if (!success) throw new IOException("Failed to write NativeImage to PNG bytes.");

        return baos.toByteArray();
    }

    public static long createCursor(class_1011 image, CursorProperties settings) throws IOException {
        float scale = SettingsUtil.sanitizeScale(settings.scale());
        int xhot = SettingsUtil.sanitizeHotspot(settings.xhot(), image.method_4307());
        int yhot = SettingsUtil.sanitizeHotspot(settings.yhot(), image.method_4323());

        float trueScale = SettingsUtil.getAutoScale(scale);
        int scaledXHot = scale == 1 ? xhot : Math.round(xhot * trueScale);
        int scaledYHot = scale == 1 ? yhot : Math.round(yhot * trueScale);

        ByteBuffer pixels = null;
        class_1011 scaledImage = null;

        try {
            if (scale != 1) {
                scaledImage = NativeImageUtil.scaleImage(image, trueScale);
            }

            GLFWImage glfwImage = GLFWImage.create();
            class_1011 validImage = scaledImage != null ? scaledImage : image;

            pixels = MemoryUtil.memAlloc(validImage.method_4307() * validImage.method_4323() * 4);
            NativeImageUtil.writePixelsRGBA(validImage, pixels);

            glfwImage.set(validImage.method_4307(), validImage.method_4323(), pixels);

            long handle = GLFW.glfwCreateCursor(glfwImage, scaledXHot, scaledYHot);
            if (handle == MemoryUtil.NULL) {
                throw new IOException("Could not create GLFW Cursor");
            }

            return handle;
        } finally {
            if (scaledImage != null) {
                scaledImage.close();
            }
            if (pixels != null) {
                MemoryUtil.memFree(pixels);
            }
        }
    }
}
