package io.github.fishstiz.minecraftcursor.gui.screen.panel;

import io.github.fishstiz.minecraftcursor.CursorResourceLoader;
import io.github.fishstiz.minecraftcursor.cursor.CursorManager;
import io.github.fishstiz.minecraftcursor.MinecraftCursor;
import io.github.fishstiz.minecraftcursor.api.CursorController;
import io.github.fishstiz.minecraftcursor.config.Config;
import io.github.fishstiz.minecraftcursor.cursor.AnimatedCursor;
import io.github.fishstiz.minecraftcursor.cursor.Cursor;
import io.github.fishstiz.minecraftcursor.gui.CursorAnimationHelper;
import io.github.fishstiz.minecraftcursor.gui.screen.CatalogItem;
import io.github.fishstiz.minecraftcursor.gui.widget.*;
import io.github.fishstiz.minecraftcursor.gui.MouseEvent;
import io.github.fishstiz.minecraftcursor.util.SettingsUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_339;
import net.minecraft.class_4185;
import net.minecraft.class_5244;
import net.minecraft.class_6379;
import net.minecraft.class_7845;
import net.minecraft.class_7919;
import net.minecraft.class_8021;
import net.minecraft.class_8133;

import static io.github.fishstiz.minecraftcursor.util.SettingsUtil.*;

public class CursorOptionsPanel extends AbstractOptionsPanel {
    private static final class_2561 HOTSPOT_GUIDE_TEXT = class_2561.method_43471("minecraft-cursor.options.hotspot-guide");
    private static final class_2561 ANIMATE_TEXT = class_2561.method_43471("minecraft-cursor.options.animate");
    private static final class_2561 RESET_ANIMATION_TEXT = class_2561.method_43471("minecraft-cursor.options.animate-reset");
    private static final class_2561 RESET_DEFAULTS_TEXT = class_2561.method_43471("minecraft-cursor.options.reset-defaults");
    private static final class_7919 GLOBAL_SCALE_TOOLTIP = createGlobalTooltip(SCALE_TEXT);
    private static final class_7919 GLOBAL_XHOT_TOOLTIP = createGlobalTooltip(XHOT_TEXT);
    private static final class_7919 GLOBAL_YHOT_TOOLTIP = createGlobalTooltip(YHOT_TEXT);
    private static final int CELL_SIZE_STEP = 32;
    private static final int SCALE_CURSOR_OVERRIDE = -20;
    private final CursorAnimationHelper animationHelper;
    private final Runnable refreshCursors;
    private final CatalogItem globalOptions;
    private final @NotNull Config.Settings settings;
    private final @NotNull Cursor cursor;
    private class_7845 layout;
    private OptionsList optionsList;
    private ToggleWidget enableToggler;
    private SliderWidget scaleSlider;
    private ButtonWidget guiScaleButton;
    private SliderWidget xhotSlider;
    private SliderWidget yhotSlider;
    private ButtonWidget resetToDefaultsButton;
    private CursorHotspotWidget hotspotWidget;
    private CursorPreviewWidget previewWidget;
    private ToggleWidget hotspotGuideToggler;

    public CursorOptionsPanel(
            CursorAnimationHelper animationHelper,
            Runnable refreshCursors,
            CatalogItem globalOptions,
            Cursor cursor
    ) {
        super(class_2561.method_43469("minecraft-cursor.options.cursor-type", cursor.getText()));

        this.animationHelper = animationHelper;
        this.refreshCursors = refreshCursors;
        this.globalOptions = globalOptions;
        this.settings = MinecraftCursor.CONFIG.getOrCreateSettings(cursor);
        this.cursor = cursor;
    }

    @Override
    protected void initContents() {
        final int row = 0;
        int column = 0;

        this.layout = new class_7845().method_48637(this.getSpacing());
        this.layout.method_46452(this.setupFirstColumnWidgets(), row, column);
        this.layout.method_46452(this.setupSecondColumnWidgets(), row, ++column);
        this.layout.method_48206(this::addRenderableWidget);
    }

    private @NotNull class_8021 setupFirstColumnWidgets() {
        this.optionsList = new OptionsList(this.getMinecraft(), class_4185.field_39501, this.getSpacing());

        this.enableToggler = this.optionsList.addOption(new ToggleWidget(
                this.cursor.isLoaded() && this.cursor.isEnabled(),
                ENABLE_TEXT,
                this::onToggleEnable
        ));

        if (this.cursor.isLoaded()) {
            this.scaleSlider = this.optionsList.addOption(
                    new SliderWidget(
                            sanitizeScale(this.settings.getScale()),
                            SCALE_MIN,
                            SCALE_MAX,
                            SCALE_STEP,
                            this::onChangeScale,
                            SCALE_TEXT,
                            class_5244.field_39003,
                            SettingsUtil::getAutoText,
                            this::onScaleMouseEvent
                    ),
                    this.bindGlobalInfo(GLOBAL_SCALE_TOOLTIP, MinecraftCursor.CONFIG.getGlobal().isScaleActive())
            );
            this.guiScaleButton = this.optionsList.addOption(new ButtonWidget(GUI_SCALE_TEXT, this::setGuiScale));
            this.xhotSlider = this.optionsList.addOption(
                    new SliderWidget(
                            sanitizeHotspot(this.settings.getXHot(), this.cursor),
                            HOT_MIN,
                            getMaxHotspot(this.cursor),
                            HOT_STEP,
                            this::onChangeXHot,
                            XHOT_TEXT,
                            HOTSPOT_SUFFIX
                    ),
                    this.bindGlobalInfo(GLOBAL_XHOT_TOOLTIP, MinecraftCursor.CONFIG.getGlobal().isXHotActive())
            );
            this.yhotSlider = this.optionsList.addOption(
                    new SliderWidget(
                            sanitizeHotspot(this.settings.getYHot(), this.cursor),
                            HOT_MIN,
                            getMaxHotspot(this.cursor),
                            HOT_STEP,
                            this::onChangeYHot,
                            YHOT_TEXT,
                            HOTSPOT_SUFFIX
                    ),
                    this.bindGlobalInfo(GLOBAL_YHOT_TOOLTIP, MinecraftCursor.CONFIG.getGlobal().isYHotActive())
            );
            this.hotspotGuideToggler = this.optionsList.addOption(new ToggleWidget(
                    true,
                    HOTSPOT_GUIDE_TEXT,
                    this::onToggleGuide
            ));

            if (this.cursor instanceof AnimatedCursor animatedCursor) {
                this.optionsList.addOption(new ToggleWidget(
                        animatedCursor.isAnimated(),
                        ANIMATE_TEXT,
                        this::onToggleAnimate
                ));
                this.optionsList.addOption(new ButtonWidget(RESET_ANIMATION_TEXT, this::restartAnimation));
            }

            this.resetToDefaultsButton = this.optionsList.addOption(new ButtonWidget(RESET_DEFAULTS_TEXT, this::resetToDefaults));

            this.refreshGuiScaleButton(this.settings.getScale());
            this.refreshDefaultsButton();
        }

        return this.optionsList;
    }

    private @NotNull class_8021 setupSecondColumnWidgets() {
        final int column = 0;
        int row = 0;

        class_7845 cursorWidgetsLayout = new class_7845().method_48637(this.getSpacing());

        if (this.cursor.isLoaded()) {
            this.hotspotWidget = cursorWidgetsLayout.method_46452(
                    new CursorHotspotWidget(
                            this.cursor,
                            this.animationHelper,
                            Objects.requireNonNull(this.xhotSlider),
                            Objects.requireNonNull(this.yhotSlider),
                            this::onHotspotWidgetMouseEvent
                    ),
                    row,
                    column
            );
            this.previewWidget = cursorWidgetsLayout.method_46452(
                    new CursorPreviewWidget(this.cursor, this.getFont()),
                    ++row,
                    column
            );
        }

        return cursorWidgetsLayout;
    }

    @Override
    protected void repositionContents(int x, int y) {
        if (this.layout != null) {
            int cellSize = clampCell(this.getWidth() / 2 - this.getSpacing() / 2);
            this.layout.method_48206(widget -> widget.method_25358(cellSize));

            if (this.optionsList != null) {
                this.optionsList.method_53533(this.computeMaxHeight(y));
            }
            if (this.hotspotWidget != null) {
                this.hotspotWidget.method_53533(cellSize);
            }

            this.layout.method_48229(x, y);
            this.layout.method_48222();

            if (this.previewWidget != null) {
                this.previewWidget.method_53533(Math.min(cellSize, this.getBottom() - this.previewWidget.method_46427()));
            }
        }
    }

    private <T extends class_339> Function<T, class_339> bindGlobalInfo(class_7919 tooltip, boolean global) {
        return widget -> {
            if (widget != null && global) {
                widget.field_22763 = false;
                return new InactiveInfoWidget(widget, tooltip, this::toGlobalOptionsPanel);
            }
            return null;
        };
    }

    private void toGlobalOptionsPanel() {
        this.changeItem(this.globalOptions);
    }

    private void onToggleEnable(ToggleWidget target, boolean enabled) {
        if (!this.cursor.isLoaded() && this.loadCursor(this.cursor)) {
            Cursor loaded = Objects.requireNonNull(CursorManager.INSTANCE.getCursor(this.cursor.getType()));
            loaded.enable(true);
            this.settings.setEnabled(true);
            this.refreshCursors.run();
            return;
        }

        if (this.cursor.isLoaded()) {
            this.cursor.enable(enabled);
            this.settings.setEnabled(enabled);
            this.refreshCursors.run();
        } else {
            target.setValue(false);
        }
    }

    private void onChangeScale(double scale) {
        this.cursor.setScale(scale);
        this.settings.setScale(scale);
        this.refreshGuiScaleButton(scale);
        this.refreshDefaultsButton();
    }

    private void onChangeXHot(double xhot) {
        this.cursor.setXHot(xhot);
        this.settings.setXHot(this.cursor, (int) xhot);
        this.refreshDefaultsButton();
    }

    private void onChangeYHot(double yhot) {
        this.cursor.setYHot(yhot);
        this.settings.setYHot(this.cursor, (int) yhot);
        this.refreshDefaultsButton();
    }

    private void onToggleAnimate(boolean animated) {
        if (!(this.cursor instanceof AnimatedCursor animatedCursor)) {
            throw new IllegalStateException("Cursor is not an animated cursor");
        }
        animatedCursor.setAnimated(animated);
        this.settings.setAnimated(animated);
        this.refreshDefaultsButton();
    }

    private void onToggleGuide(boolean shown) {
        Objects.requireNonNull(this.hotspotWidget).setRenderRuler(shown);
    }

    private void restartAnimation() {
        if (this.cursor instanceof AnimatedCursor animatedCursor) {
            this.animationHelper.reset(animatedCursor);
        }
    }

    private void setGuiScale(class_4185 target) {
        target.method_25365(false);
        target.field_22763 = false;
        if (this.scaleSlider != null) {
            this.scaleSlider.applyMappedValue(SCALE_AUTO_PREFERRED);
            this.method_25395(this.scaleSlider);
        }
    }

    private void refreshGuiScaleButton(double scale) {
        if (this.guiScaleButton != null) {
            this.guiScaleButton.field_22763 = !isAutoScale(scale) && !MinecraftCursor.CONFIG.getGlobal().isScaleActive();
        }
    }

    private void refreshDefaultsButton() {
        if (this.resetToDefaultsButton != null) {
            this.resetToDefaultsButton.field_22763 = !CursorResourceLoader.isResourceSetting(this.cursor, this.settings);
        }
    }

    private void resetToDefaults() {
        if (CursorResourceLoader.retoreActiveResourceSettings(Objects.requireNonNull(this.cursor))) {
            this.refreshCursors.run();
            if (this.enableToggler != null) {
                this.method_25395(this.enableToggler);
            }
        }
    }

    private void onScaleMouseEvent(SliderWidget target, MouseEvent mouseEvent, double mappedValue) {
        if (mouseEvent.clicked() && CursorManager.INSTANCE.isEnabled(this.cursor)) {
            CursorController.getInstance().overrideCursor(this.cursor.getType(), SCALE_CURSOR_OVERRIDE);
        } else if (mouseEvent.released()) {
            removeScaleOverride();
        }
    }

    private void onHotspotWidgetMouseEvent(CursorHotspotWidget target, MouseEvent mouseEvent, int xhot, int yhot) {
        if (mouseEvent.clicked() || mouseEvent.dragged()) {
            target.setRenderRuler(true);
            if (this.hotspotGuideToggler != null) {
                this.hotspotGuideToggler.setValue(true);
            }
        }
    }

    @Override
    protected void removed() {
        removeScaleOverride();
    }

    public static void removeScaleOverride() {
        CursorController.getInstance().removeOverride(SCALE_CURSOR_OVERRIDE);
    }

    private static int clampCell(int cell) {
        return Math.round(cell / (float) CELL_SIZE_STEP) * CELL_SIZE_STEP;
    }

    private static class_7919 createGlobalTooltip(class_2561 option) {
        return class_7919.method_47407(class_2561.method_43469("minecraft-cursor.options.global.inactive.tooltip", option));
    }

    private static class OptionsList extends AbstractListWidget<OptionsList.Entry> implements class_8133 {
        private static final int BACKGROUND_PADDING_Y = 2;
        private final ElementSlidingBackground hoveredBackground = new ElementSlidingBackground(0x26FFFFFF); // 15% white

        private OptionsList(class_310 minecraft, int itemHeight, int spacing) {
            super(minecraft, 0, 0, 0, itemHeight, spacing);
        }

        private <T extends class_339> @NotNull T addOption(@NotNull T optionWidget, @Nullable Function<T, class_339> decoration) {
            this.method_25321(new io.github.fishstiz.minecraftcursor.gui.screen.panel.CursorOptionsPanel.OptionsList.Entry(optionWidget, decoration != null ? decoration.apply(optionWidget) : null));
            return optionWidget;
        }

        private <T extends class_339> @NotNull T addOption(@NotNull T optionWidget) {
            return this.addOption(optionWidget, null);
        }

        @Override
        protected void method_57715(@NotNull class_332 guiGraphics) {
            // remove background
        }

        @Override
        protected void method_57713(@NotNull class_332 guiGraphics) {
            // remove separators
        }

        @Override
        public void method_48227(@NotNull Consumer<class_8021> visitor) {
            this.method_25396().forEach(visitor);
        }

        @Override
        public void method_48222() {
            class_8133.super.method_48222();
            this.method_60322();
        }

        protected void renderSlidingBackground(@NotNull class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
            int paddingX = this.rowGap;

            int minX = this.method_46426() - paddingX;
            int minY = this.method_46427() - BACKGROUND_PADDING_Y;
            int maxX = this.method_55442() + field_45909;
            int maxY = this.method_55443() + BACKGROUND_PADDING_Y;
            guiGraphics.method_44379(minX, minY, maxX, maxY);

            io.github.fishstiz.minecraftcursor.gui.screen.panel.CursorOptionsPanel.OptionsList.Entry entry = this.method_25308(mouseX, mouseY);
            if (entry == null) {
                this.hoveredBackground.reset();
            } else {
                int x = entry.method_46426() - paddingX;
                int y = entry.method_46427() - BACKGROUND_PADDING_Y;
                int width = entry.method_25368() + paddingX + (this.method_57717() ? field_45909 : BACKGROUND_PADDING_Y);
                int height = entry.method_25364() + BACKGROUND_PADDING_Y * 2;
                this.hoveredBackground.render(guiGraphics, x, y, width, height, partialTick);
            }

            guiGraphics.method_44380();
        }

        @Override
        public void method_48579(@NotNull class_332 guiGraphics, int mouseX, int mouseY, float partialTick) {
            this.renderSlidingBackground(guiGraphics, mouseX, mouseY, partialTick);
            super.method_48579(guiGraphics, mouseX, mouseY, partialTick);
        }

        private class Entry extends AbstractListWidget<io.github.fishstiz.minecraftcursor.gui.screen.panel.CursorOptionsPanel.OptionsList.Entry>.Entry implements class_8133 {
            private final List<class_339> children = new ArrayList<>();
            private final class_339 optionWidget;
            private final class_339 decoration;

            private Entry(@NotNull class_339 optionWidget, @Nullable class_339 decoration) {
                this.optionWidget = optionWidget;
                this.decoration = decoration;

                if (this.decoration != null) {
                    this.children.add(this.decoration);
                }
                this.children.add(this.optionWidget);
            }

            @Override
            public void render(@NotNull class_332 guiGraphics, int index, int top, int left, int width, int height, int mouseX, int mouseY, boolean hovering, float partialTick) {
                this.optionWidget.method_48229(left, top);
                this.optionWidget.method_25394(guiGraphics, mouseX, mouseY, partialTick);

                if (this.decoration != null) {
                    this.decoration.method_25394(guiGraphics, mouseX, mouseY, partialTick);
                }
            }

            @Override
            public @NotNull List<class_339> children() {
                return this.children;
            }

            @Override
            public @NotNull List<? extends class_6379> narratables() {
                return this.children;
            }

            @Override
            public void method_46421(int x) {
                this.optionWidget.method_46421(x);
            }

            @Override
            public void method_46419(int y) {
                this.optionWidget.method_46419(y);
            }

            @Override
            public void method_48227(@NotNull Consumer<class_8021> visitor) {
                visitor.accept(this.optionWidget);
            }

            @Override
            public void method_48206(@NotNull Consumer<class_339> visitor) {
                visitor.accept(this.optionWidget);
            }

            @Override
            public void method_48222() {
                this.optionWidget.method_25358(this.method_25368());
            }
        }
    }
}
