package io.github.fishstiz.minecraftcursor.cursor;

import com.mojang.blaze3d.platform.NativeImage;
import io.github.fishstiz.minecraftcursor.CursorResourceLoader;
import io.github.fishstiz.minecraftcursor.MinecraftCursor;
import io.github.fishstiz.minecraftcursor.api.CursorType;
import io.github.fishstiz.minecraftcursor.api.CursorTypeRegistrar;
import io.github.fishstiz.minecraftcursor.compat.ExternalCursorTracker;
import io.github.fishstiz.minecraftcursor.config.AnimationData;
import io.github.fishstiz.minecraftcursor.config.Config;
import it.unimi.dsi.fastutil.objects.Object2ObjectLinkedOpenHashMap;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.GuiGraphics;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.util.*;

import static io.github.fishstiz.minecraftcursor.MinecraftCursor.CONFIG;

public final class CursorManager implements CursorTypeRegistrar {
    public static final CursorManager INSTANCE = new CursorManager();
    private final Map<String, Cursor> cursors = new Object2ObjectLinkedOpenHashMap<>();
    private final TreeMap<Integer, String> overrides = new TreeMap<>();
    private final AnimationState animationState = new AnimationState();
    private @NotNull Cursor currentCursor = Cursor.createDummy();
    private @NotNull CursorRenderer renderer = CONFIG.isVirtualMode() ? new CursorRenderer.Virtual() : new CursorRenderer.Native();

    private CursorManager() {
    }

    @Override
    public void register(CursorType... cursorTypes) {
        for (CursorType cursorType : cursorTypes) {
            register(cursorType);
        }
    }

    @Override
    public CursorType register(String key) {
        return register(CursorType.of(key));
    }

    private CursorType register(CursorType cursorType) {
        String key = cursorType.getKey();

        if (key.isEmpty()) {
            throw new NullPointerException("Cursor type key cannot be empty.");
        }

        if (cursors.containsKey(key)) {
            MinecraftCursor.LOGGER.error("[minecraft-cursor] Cursor type '{}' is already registered.", key);
            return cursorType;
        }

        cursors.put(key, new Cursor(cursorType, this::onLoad));

        return cursorType;
    }

    public Cursor loadCursor(
            Cursor cursor,
            NativeImage image,
            Config.Settings settings,
            @Nullable AnimationData animation
    ) throws IOException {
        if (!cursors.containsKey(cursor.getTypeKey())) {
            throw new IllegalStateException("Attempting to load an unregistered cursor: " + cursor.getTypeKey());
        }

        boolean animated = animation != null;
        if (animated != (cursor instanceof AnimatedCursor)) {
            cursor.destroy();
            cursor = animated
                    ? new AnimatedCursor(cursor.getType(), this::onLoad)
                    : new Cursor(cursor.getType(), this::onLoad);
            cursors.put(cursor.getTypeKey(), cursor);
        }

        if (cursor instanceof AnimatedCursor animatedCursor) {
            animatedCursor.loadImage(image, settings, animation);
        } else {
            cursor.loadImage(image, settings);
        }

        return cursor;
    }

    private void onLoad(Cursor cursor) {
        final CursorType cursorType = cursor.getType();
        final long id = cursor.getId();

        Minecraft.getInstance().execute(() -> {
            Cursor appliedCursor = getAppliedCursor();
            if (appliedCursor.isLoaded() && appliedCursor.getType().isKey(cursorType) && appliedCursor.getId() == id) {
                reapplyCursor();
            }
        });
    }

    public void setCurrentCursor(@NotNull CursorType type) {
        Cursor override = getOverride();
        Cursor cursor = override != null ? override : this.cursors.get(type.getKey());

        if (cursor != null && cursor.isLazy() && CONFIG.getOrCreateSettings(cursor).isEnabled()) {
            CursorResourceLoader.loadCursorTexture(cursor);
        }

        if (cursor instanceof AnimatedCursor animatedCursor && cursor.getId() != 0) {
            handleCursorAnimation(animatedCursor);
            return;
        }

        if (cursor == null || !type.isDefault() && cursor.getId() == 0 || !cursor.isEnabled()) {
            cursor = getCursor(CursorType.DEFAULT);
        }

        updateCursor(cursor);
    }

    private void handleCursorAnimation(AnimatedCursor cursor) {
        if (!getAppliedCursor().getType().isKey(cursor.getType())) {
            animationState.reset();
        }

        Cursor currentFrameCursor = cursor.nextFrame(animationState).cursor();
        updateCursor(currentFrameCursor.getId() != 0 ? currentFrameCursor : cursor);
    }

    private void updateCursor(Cursor cursor) {
        if (cursor == null
            || (!CONFIG.isAggressiveCursor() && cursor.getId() == currentCursor.getId())
            || ExternalCursorTracker.get().isCustom()) {
            return;
        }

        this.currentCursor = cursor;
        this.renderer.setCursor(this.currentCursor);
    }

    public void reapplyCursor() {
        if (!ExternalCursorTracker.get().isCustom()) {
            this.renderer.setCursor(this.getAppliedCursor());
        }
    }

    public void overrideCurrentCursor(CursorType type, int index) {
        Cursor cursor = getCursor(type);
        if (cursor != null && cursor.isEnabled()) {
            overrides.put(index, type.getKey());
        } else {
            overrides.remove(index);
        }
    }

    public void removeOverride(int index) {
        overrides.remove(index);
    }

    public @Nullable Cursor getOverride() {
        while (!overrides.isEmpty()) {
            Map.Entry<Integer, String> lastEntry = overrides.lastEntry();
            Cursor cursor = this.cursors.get(lastEntry.getValue());

            if (cursor == null || cursor.getId() == 0) {
                overrides.remove(lastEntry.getKey());
            } else {
                return cursor;
            }
        }

        return null;
    }

    public @NotNull Cursor getAppliedCursor() {
        Cursor override = getOverride();
        Cursor cursor = override != null ? override : currentCursor;

        if (cursor instanceof AnimatedCursor animatedCursor) {
            return animatedCursor.getFrame(animationState.getCurrentFrame()).cursor();
        }

        return cursor;
    }

    public boolean isEnabled(@NotNull CursorType type) {
        return isEnabled(cursors.get(type.getKey()));
    }

    public boolean isEnabled(@Nullable Cursor cursor) {
        return cursor != null && ((cursor.isLazy() && CONFIG.getOrCreateSettings(cursor).isEnabled()) || cursor.isEnabled());
    }

    public @Nullable Cursor getCursor(CursorType type) {
        return cursors.get(type.getKey());
    }

    public @Nullable Cursor getCursor(String type) {
        return cursors.get(type);
    }

    public long getCurrentId() {
        return getAppliedCursor().getId();
    }

    public Collection<Cursor> getCursors() {
        return cursors.values();
    }

    public boolean isAdaptive() {
        for (Cursor cursor : cursors.values()) {
            if (!cursor.getType().isDefault() && ((cursor.isLazy() || cursor.isEnabled()) && CONFIG.getOrCreateSettings(cursor).isEnabled())) {
                return true;
            }
        }
        return false;
    }

    public boolean isVirtual() {
        return this.renderer instanceof CursorRenderer.Virtual;
    }

    public void toggleVirtual() {
        this.renderer.resetCursor();
        this.renderer = this.isVirtual() ? new CursorRenderer.Native() : new CursorRenderer.Virtual();
        this.reapplyCursor();
    }

    public void renderCursor(Minecraft minecraft, GuiGraphics guiGraphics, int mouseX, int mouseY) {
        this.renderer.render(minecraft, guiGraphics, mouseX, mouseY);
    }
}
