package io.github.fishstiz.minecraftcursor;

import io.github.fishstiz.minecraftcursor.api.CursorType;
import io.github.fishstiz.minecraftcursor.config.AnimatedCursorConfig;
import io.github.fishstiz.minecraftcursor.config.CursorConfig;
import io.github.fishstiz.minecraftcursor.config.CursorConfigLoader;
import javax.imageio.ImageIO;
import net.minecraft.class_2960;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Optional;

import static io.github.fishstiz.minecraftcursor.MinecraftCursor.CONFIG;
import static io.github.fishstiz.minecraftcursor.MinecraftCursor.MOD_ID;

abstract class AbstractCursorResourceReloadListener {
    private static final String IMG_TYPE = ".png";
    private static final String ANIMATION_TYPE = IMG_TYPE + ".mcmeta";
    private static final String CONFIG_PATH = "atlases/cursors.json";
    private static final String CURSORS_DIR = "textures/cursors/";

    protected AbstractCursorResourceReloadListener() {
    }

    public class_2960 getId() {
        return class_2960.method_43902(MOD_ID, CURSORS_DIR);
    }

    public void reloadMinecraftCursor(class_3300 manager) {
        loadConfig(manager);
        loadCursorTextures(manager);
        CursorManager.INSTANCE.setCurrentCursor(CursorType.DEFAULT);
    }

    private void loadConfig(class_3300 manager) {
        List<class_3298> configResources = manager.method_14489(class_2960.method_43902(MOD_ID, CONFIG_PATH));

        if (configResources.isEmpty()) return;

        getConfigFromResources(configResources).ifPresent(config -> {
            if (!config.get_hash().equals(CONFIG.get_hash())) {
                MinecraftCursor.LOGGER.info("[minecraft-cursor] New resource pack settings detected, updating config...");
                CONFIG.set_hash(config.get_hash());
                CONFIG.setSettings(config.getSettings());
                CONFIG.getGlobal().setActiveAll(false);
                CONFIG.save();
            }
        });
    }

    private Optional<CursorConfig> getConfigFromResources(List<class_3298> configResources) {
        CursorConfig combinedConfig = null;

        for (class_3298 configResource : configResources) {
            try (InputStream stream = configResource.method_14482()) {
                CursorConfig loadedConfig = CursorConfigLoader.fromStream(stream);

                if (combinedConfig == null) {
                    combinedConfig = loadedConfig;
                } else {
                    combinedConfig.setSettings(loadedConfig.getSettings());
                }
            } catch (IOException e) {
                MinecraftCursor.LOGGER.error("[minecraft-cursor] Failed to load settings of resource pack '{}'", configResource.method_14480());
            }
        }

        return Optional.ofNullable(combinedConfig);
    }

    private void loadCursorTextures(class_3300 manager) {
        for (CursorType cursorType : CursorManager.INSTANCE.getCursorTypes()) {
            String basePath = CURSORS_DIR + cursorType.getKey();
            loadCursorTexture(manager, cursorType, basePath);
        }
    }

    private void loadCursorTexture(class_3300 manager, CursorType cursorType, String basePath) {
        class_2960 cursorId = class_2960.method_43902(MOD_ID, basePath + IMG_TYPE);
        class_3298 cursorResource = manager.method_14486(cursorId).orElse(null);


        if (cursorResource == null) {
            MinecraftCursor.LOGGER.error("[minecraft-cursor] Cursor Type: '{}' not found", cursorType.getKey());
            return;
        }

        BufferedImage image = null;
        try (InputStream cursorStream = cursorResource.method_14482()) {
            image = ImageIO.read(cursorStream);
            if (image == null) {
                MinecraftCursor.LOGGER.error("[minecraft-cursor] Invalid file for cursor type '{}'", cursorType);
                return;
            }

            AnimatedCursorConfig animation = loadAnimationConfig(manager, basePath, cursorResource);
            CursorManager.INSTANCE.loadCursorImage(cursorType, cursorId, image, CONFIG.getOrCreateCursorSettings(cursorType), animation);
        } catch (IOException e) {
            MinecraftCursor.LOGGER.error("[minecraft-cursor] Failed to load cursor image for '{}'", basePath);
        } finally {
            if (image != null) image.flush();
        }
    }

    private AnimatedCursorConfig loadAnimationConfig(class_3300 manager, String basePath, class_3298 cursorResource) {
        class_3298 animationResource = manager
                .method_14486(class_2960.method_43902(MOD_ID, basePath + ANIMATION_TYPE))
                .orElse(null);

        if (animationResource != null && animationResource.method_14480().equals(cursorResource.method_14480())) {
            try (InputStream stream = animationResource.method_14482()) {
                return CursorConfigLoader.getAnimationConfig(stream);
            } catch (IOException e) {
                MinecraftCursor.LOGGER.error("[minecraft-cursor] Failed to load animation config for '{}'", basePath);
            }
        }

        return null;
    }
}
