package io.github.fishstiz.minecraftcursor;

import io.github.fishstiz.minecraftcursor.api.CursorType;
import io.github.fishstiz.minecraftcursor.config.AnimationData;
import io.github.fishstiz.minecraftcursor.config.Config;
import io.github.fishstiz.minecraftcursor.config.ConfigLoader;
import io.github.fishstiz.minecraftcursor.cursor.Cursor;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Optional;
import net.minecraft.class_1011;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import net.minecraft.class_370;

import static io.github.fishstiz.minecraftcursor.MinecraftCursor.CONFIG;
import static io.github.fishstiz.minecraftcursor.MinecraftCursor.LOGGER;
import static io.github.fishstiz.minecraftcursor.MinecraftCursor.MOD_ID;

public class CursorLoader {
    private static final String ANIMATION_TYPE = ".mcmeta";
    private static final class_2960 SETTINGS_LOCATION = class_2960.method_60655(MOD_ID, "atlases/cursors.json");
    private static final class_2960 DIR = class_2960.method_60655(MOD_ID, "textures/cursors/");
    private static Config.Resource resourceConfig;

    private CursorLoader() {
    }

    public static class_2960 getDirectory() {
        return DIR;
    }

    public static void reload(class_3300 manager) {
        onReload();
        loadResourceSettings(manager);
        loadCursorTextures(manager);
        class_310.method_1551().execute(CursorLoader::onReload);
    }

    static void onReload() {
        CursorManager.INSTANCE.setCurrentCursor(CursorType.DEFAULT);
    }

    private static void loadResourceSettings(class_3300 manager) {
        getLayeredSettings(manager.method_14489(SETTINGS_LOCATION)).ifPresent(config -> {
            if (config.isDifferent(CONFIG)) {
                LOGGER.info("[minecraft-cursor] New resource pack settings detected, updating config...");
                CONFIG.setHash(config.getHash());
                CONFIG.mergeResources(config);
                CONFIG.getGlobal().setActiveAll(false);
                CONFIG.save();
            }
            resourceConfig = config;
        });
    }

    private static Optional<Config.Resource> getLayeredSettings(List<class_3298> configResources) {
        Config.Resource layeredResources = null;
        for (class_3298 configResource : configResources) {
            try (InputStream stream = configResource.method_14482()) {
                Config.Resource resourceConfig = ConfigLoader.loadResource(stream);
                if (layeredResources == null) {
                    layeredResources = resourceConfig;
                } else {
                    layeredResources.layer(resourceConfig.getSettings());
                }
            } catch (IOException e) {
                LOGGER.error("[minecraft-cursor] Failed to load settings of resource pack '{}'", configResource.method_14480());
            }
        }
        return Optional.ofNullable(layeredResources);
    }

    public static void applyResourceSettings() {
        if (resourceConfig != null) {
            CONFIG.setHash(resourceConfig.getHash());
            CONFIG.mergeResources(resourceConfig);
            for (Cursor cursor : CursorManager.INSTANCE.getCursors()) {
                cursor.applySettings(CONFIG.getOrCreateCursorSettings(cursor));
            }
        } else {
            LOGGER.error("Failed to apply resource config: Not Found.");
        }
    }

    private static void loadCursorTextures(class_3300 manager) {
        for (Cursor cursor : CursorManager.INSTANCE.getCursors()) {
            Config.Settings settings = CONFIG.getOrCreateCursorSettings(cursor);

            if (!CONFIG.isDeferredLoading() || settings.isEnabled()) {
                loadCursorTexture(manager, cursor, settings);
            } else {
                LOGGER.info("[minecraft-cursor] Skipped loading of disabled cursor '{}'.", cursor.getTypeKey());
            }
        }
    }

    public static boolean loadCursorTexture(Cursor cursor) {
        class_310 minecraft = class_310.method_1551();
        if (!loadCursorTexture(minecraft.method_1478(), cursor, CONFIG.getOrCreateCursorSettings(cursor))) {
            minecraft.method_1566().method_1999(class_370.method_29047(
                    minecraft,
                    class_370.class_9037.field_47585,
                    class_2561.method_43471("resourcePack.load_fail"),
                    class_2561.method_43469("minecraft-cursor.options.deferred_loading.fail", cursor.getText())
            ));
            return false;
        }
        return true;
    }

    private static boolean loadCursorTexture(class_3300 manager, Cursor cursor, Config.Settings settings) {
        LOGGER.info("[minecraft-cursor] Loading cursor '{}'...", cursor.getTypeKey());

        class_2960 location = cursor.getLocation();
        Optional<class_3298> cursorResource = manager.method_14486(location);

        if (cursorResource.isEmpty()) {
            LOGGER.error("[minecraft-cursor] Cursor Type: '{}' not found", cursor.getTypeKey());
            return false;
        }

        try (InputStream cursorStream = cursorResource.get().method_14482(); class_1011 image = class_1011.method_4309(cursorStream)) {
            AnimationData animation = loadAnimation(manager, location, cursorResource.get());
            CursorManager.INSTANCE.loadCursor(cursor, image, CONFIG.getGlobal().apply(settings), animation);
            return true;
        } catch (IOException e) {
            LOGGER.error("[minecraft-cursor] Failed to load cursor at '{}'", location);
            return false;
        }
    }

    private static AnimationData loadAnimation(class_3300 manager, class_2960 location, class_3298 cursorResource) {
        Optional<class_3298> animationResource = manager.method_14486(location.method_48331(ANIMATION_TYPE));
        if (animationResource.isPresent() && animationResource.get().method_14480().equals(cursorResource.method_14480())) {
            try (InputStream stream = animationResource.get().method_14482()) {
                return ConfigLoader.getAnimationConfig(stream);
            } catch (IOException e) {
                LOGGER.error("[minecraft-cursor] Failed to load animation config for '{}'", location);
            }
        }
        return null;
    }
}
