package io.github.fishstiz.minecraftcursor;

import io.github.fishstiz.minecraftcursor.api.CursorHandler;
import io.github.fishstiz.minecraftcursor.api.CursorProvider;
import io.github.fishstiz.minecraftcursor.api.CursorType;
import io.github.fishstiz.minecraftcursor.api.ElementRegistrar;
import io.github.fishstiz.minecraftcursor.cursor.InternalCursorProvider;
import io.github.fishstiz.minecraftcursor.inspect.ElementInspector;
import io.github.fishstiz.minecraftcursor.platform.Services;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;
import net.minecraft.class_364;
import net.minecraft.class_4069;

class CursorTypeResolver implements ElementRegistrar {
    private final List<ElementEntry<? extends class_364>> registry = new ArrayList<>();
    private final HashMap<String, CursorTypeFunction<? extends class_364>> cache = new HashMap<>();
    private ElementInspector inspector = ElementInspector.NO_OP;
    String lastFailedElement;

    CursorTypeResolver() {
    }

    @Override
    public <T extends class_364> void register(CursorHandler<T> cursorHandler) {
        CursorHandler.TargetElement<T> targetElement = cursorHandler.getTargetElement();

        if (targetElement.elementClass().isPresent()) {
            register(targetElement.elementClass().get(), cursorHandler::getCursorType);
        } else if (targetElement.fullyQualifiedClassName().isPresent()) {
            register(targetElement.fullyQualifiedClassName().get(), cursorHandler::getCursorType);
        } else {
            throw new NullPointerException("Could not register cursor handler: "
                                           + cursorHandler.getClass().getName()
                                           + " - Target Element Class and FQCN not present");
        }
    }

    @Override
    public <T extends class_364> void register(String binaryName, CursorTypeFunction<T> elementToCursorType) {
        try {
            @SuppressWarnings("unchecked")
            Class<T> elementClass = (Class<T>) Class.forName(Services.PLATFORM.mapClassName("intermediary", binaryName));
            if (!class_364.class.isAssignableFrom(elementClass)) {
                throw new ClassCastException(binaryName + " is not a subclass of Element");
            }
            register(elementClass, elementToCursorType);
        } catch (ClassNotFoundException e) {
            MinecraftCursor.LOGGER.error("[minecraft-cursor] Error registering element. Class not found: {}", binaryName);
        } catch (ClassCastException e) {
            MinecraftCursor.LOGGER.error("[minecraft-cursor] Error registering element. Invalid class: {}", e.getMessage());
        }
    }

    @Override
    public <T extends class_364> void register(Class<T> elementClass, CursorTypeFunction<T> elementToCursorType) {
        registry.add(new ElementEntry<>(elementClass, elementToCursorType));
    }

    @SuppressWarnings("unchecked")
    public <T extends class_364> CursorType resolve(T element, double mouseX, double mouseY) {
        String elementName = element.getClass().getName();

        try {
            if (element instanceof CursorProvider cursorProvider) {
                CursorType providedCursorType = cursorProvider.getCursorType(mouseX, mouseY);
                if (providedCursorType != null && !providedCursorType.isDefault()) {
                    return providedCursorType;
                }
            }

            CursorTypeFunction<T> mapper = (CursorTypeFunction<T>) cache.get(elementName);
            if (mapper == null) {
                mapper = (CursorTypeFunction<T>) resolveMapper(element);
                if (!inspector.setFocused(element, true)) {
                    cache.put(elementName, mapper);
                }
            }

            CursorType mapped = mapper.getCursorType(element, mouseX, mouseY);
            if (mapped != null && !mapped.isDefault()) {
                return mapped;
            }

            if (element instanceof InternalCursorProvider internalCursorProvider) {
                return internalCursorProvider.minecraft_cursor$getCursorType(mouseX, mouseY);
            }

            return CursorType.DEFAULT;
        } catch (LinkageError | Exception e) {
            if (!elementName.equals(lastFailedElement)) {
                lastFailedElement = elementName;
                MinecraftCursor.LOGGER.error(
                        "Could not get cursor type for element: {}",
                        Services.PLATFORM.unmapClassName("intermediary", elementName)
                );
            }
        }
        return CursorType.DEFAULT;
    }

    private CursorTypeFunction<? extends class_364> resolveMapper(class_364 element) {
        for (int i = registry.size() - 1; i >= 0; i--) {
            if (registry.get(i).element.isInstance(element)) {
                inspector.setFocused(element, true);
                return registry.get(i).mapper;
            }
        }
        if (element instanceof class_4069) {
            return (CursorTypeFunction<class_4069>) this::resolveChild;
        }
        return ElementRegistrar::elementToDefault;
    }

    private <T extends class_4069> CursorType resolveChild(T parent, double mouseX, double mouseY) {
        Optional<class_364> child = parent.method_19355(mouseX, mouseY);
        if (child.isPresent()) {
            class_364 hoveredElement = child.get();
            if (hoveredElement instanceof class_4069 nestedParent) {
                CursorType cursorType = resolveChild(nestedParent, mouseX, mouseY);
                if (!cursorType.isDefault()) return cursorType;
            }
            inspector.setFocused(hoveredElement, false);
            return resolve(hoveredElement, mouseX, mouseY);
        }
        return CursorType.DEFAULT;
    }

    public ElementInspector getInspector() {
        return inspector;
    }

    public void toggleInspector() {
        inspector = ElementInspector.toggle(inspector);
        cache.clear();
    }

    record ElementEntry<T extends class_364>(Class<T> element, CursorTypeFunction<T> mapper) {
    }
}
